// SPDX-FileCopyrightText: 2024-2025 Wesley Benica <wesley@benica.dev>
//
// SPDX-License-Identifier: GPL-3.0-or-later
/**
 * Release information for different versions.
 * Add new entries when releasing new versions.
 */
export const RELEASE_MESSAGES = {
    "1.1.0": {
        version: "1.1.0",
        features: {
            primary: ({ _ }) => _("Complete color customization with accent-based and custom themes."),
            secondary: [
                ({ _ }) => _("Enhanced preferences interface"),
                ({ _ }) => _("Divider customization options"),
                ({ _ }) => _("Improved translations"),
            ],
        },
        welcomeFeatures: {
            primary: ({ _ }) => _("Complete color customization with accent-based and custom themes."),
            secondary: [
                ({ _ }) => _("Modern preferences interface"),
                ({ _ }) => _("Flexible divider customization"),
            ],
        },
    },
};
/**
 * Get release information for a specific version.
 *
 * @param version - Version string (e.g., "1.0.6")
 * @returns Release info if found, undefined otherwise
 */
export function getReleaseInfo(version) {
    return RELEASE_MESSAGES[version];
}
/**
 * Generate a formatted notification message for a version.
 *
 * @param version - Version string
 * @param gettext - Gettext functions for translation
 * @param isFirstInstall - Whether this is a first install (affects messaging style)
 * @param formatAsList - Whether to format secondary features as a bullet list (default: true)
 * @returns Formatted notification body text
 */
export function generateUpdateMessage(version, gettext, isFirstInstall = false, formatAsList = true) {
    const releaseInfo = getReleaseInfo(version);
    if (!releaseInfo) {
        // Fallback for versions without specific release info
        if (isFirstInstall) {
            return gettext._("Check Preferences to explore features.");
        }
        return gettext._("Check Preferences for new features.");
    }
    const { _ } = gettext;
    // Choose appropriate feature set based on install type
    const featureSet = isFirstInstall && releaseInfo.welcomeFeatures
        ? releaseInfo.welcomeFeatures
        : releaseInfo.features;
    const primaryFeature = featureSet.primary(gettext);
    if (featureSet.secondary && featureSet.secondary.length > 0) {
        const secondaryFeatures = featureSet.secondary.map((fn) => fn(gettext));
        if (formatAsList) {
            // Format as bullet list with primary feature as intro
            const bulletList = secondaryFeatures
                .map((item) => `• ${item}`)
                .join("\n");
            return `${primaryFeature}\n${bulletList}`;
        }
        else {
            // Legacy comma-separated format
            const secondaryText = secondaryFeatures.join(", ");
            if (isFirstInstall) {
                return _("Features: %s Additional capabilities: %s")
                    .replace("%s", primaryFeature)
                    .replace("%s", secondaryText);
            }
            else {
                return _("New: %s Additional improvements: %s")
                    .replace("%s", primaryFeature)
                    .replace("%s", secondaryText);
            }
        }
    }
    return primaryFeature;
}
/**
 * Get the notification title for a version update.
 *
 * @param version - Version string
 * @param gettext - Gettext functions for translation
 * @param isFirstInstall - Whether this is a first install (no previous version seen)
 * @returns Notification title
 */
export function getNotificationTitle(version, gettext, isFirstInstall = false) {
    const releaseInfo = getReleaseInfo(version);
    if (releaseInfo?.customTitle) {
        return releaseInfo.customTitle(gettext);
    }
    if (isFirstInstall) {
        return gettext._("Text Clock %s").replace("%s", version);
    }
    return gettext._("Text Clock updated to %s").replace("%s", version);
}
