/*
 * SPDX-FileCopyrightText: 2024 Wesley Benica <wesley@benica.dev>
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
import GObject from "gi://GObject";
import St from "gi://St";
import { Extension, gettext as _, ngettext, pgettext, } from "resource:///org/gnome/shell/extensions/extension.js";
import { panel } from "resource:///org/gnome/shell/ui/main.js";
import { NotificationService } from "./services/notification_service.js";
import { SettingsManager } from "./services/settings_manager.js";
import { StyleService } from "./services/style_service.js";
import SystemSettingsMonitor from "./services/system_settings_monitor.js";
import { TextClockLabel, CLOCK_LABEL_PROPERTIES, } from "./presentation/widgets/clock_widget.js";
import { logErr, logWarn } from "./utils/error_utils.js";
import { extensionGettext, initExtensionGettext, } from "./utils/gettext/gettext_utils_ext.js";
import { fuzzinessFromEnumIndex } from "./utils/parse_utils.js";
import { createTranslatePack } from "./utils/translate/translate_pack_utils.js";
import { maybeShowUpdateNotification } from "./utils/update_notification_utils.js";
const CLOCK_STYLE_CLASS_NAME = "clock";
/**
 * Main Text Clock extension class for GNOME Shell.
 *
 * This extension replaces GNOME Shell's default clock with a textual
 * view of the time, using phrases like "five past noon" instead
 * of "12:05". It integrates seamlessly with the GNOME Shell top bar and
 * provides comprehensive customization options.
 *
 * Key features:
 * - Textual time display with multiple formats and fuzziness levels
 * - Optional date and weekday display
 * - Comprehensive color customization with accent color integration
 * - Live system accent color monitoring
 * - Automatic preference migration and update notifications
 * - Clean integration with GNOME Shell's UI and theming
 *
 * The extension manages multiple services:
 * - SettingsManager: Reactive settings handling with type safety
 * - StyleService: Color and appearance management with live updates
 * - NotificationService: User notifications for updates and errors
 */
export default class TextClock extends Extension {
    #settings;
    #settingsManager;
    #styleService;
    #notificationService;
    #systemSettingsMonitor;
    #dateMenu;
    #clock;
    #clockDisplay;
    #topBox;
    #clockLabel;
    #clockBinding;
    #translatePack;
    /**
     * Enable the extension - called by GNOME Shell when extension activates.
     *
     * Performs complete extension initialization including service setup,
     * UI integration, and settings binding. This method must complete successfully
     * for the extension to function properly.
     *
     * Initialization sequence:
     * 1. Initialize core services (settings, styling, notifications)
     * 2. Show update notifications if needed
     * 3. Get references to GNOME Shell's date menu components
     * 4. Create and place the text clock widget
     * 5. Bind settings
     */
    enable() {
        initExtensionGettext(_, ngettext, pgettext);
        this.#initServices();
        maybeShowUpdateNotification({
            settingsManager: this.#settingsManager,
            notificationService: this.#notificationService,
            metadata: this.metadata,
            openPreferences: () => this.openPreferences(),
        });
        this.#retrieveDateMenu();
        this.#placeClockLabel();
        this.#bindSettingsToClockLabel();
    }
    /**
     * Disable the extension - called by GNOME Shell when extension deactivates.
     *
     * Performs complete cleanup to restore GNOME Shell's original state and
     * prevent memory leaks. Essential for proper extension lifecycle management
     * in the GNOME Shell environment.
     *
     * Cleanup sequence:
     * 1. Restore original clock display in the top bar
     * 2. Clean up all services and disconnect signal handlers
     * 3. Clear references to prevent memory leaks
     */
    disable() {
        this.#restoreClockDisplay();
        this.#cleanup();
    }
    /**
     * Initialize all core services required by the extension.
     *
     * Sets up the service layer including settings management, styling system,
     * and notification handling. Services are initialized in dependency order
     * to ensure proper functionality.
     *
     * Starts system settings monitor for changes in system accent color, and
     * show date/time format if applicable.
     *
     * @private
     */
    #initServices() {
        this.#settings = this.getSettings();
        this.#settingsManager = new SettingsManager(this.#settings);
        this.#styleService = new StyleService(this.#settings);
        this.#notificationService = new NotificationService("Text Clock");
        this.#systemSettingsMonitor = new SystemSettingsMonitor(this.#settings);
        try {
            this.#systemSettingsMonitor?.start();
        }
        catch (e) {
            logWarn(`Failed to start SystemSettingsMonitor: ${e}`);
        }
    }
    /**
     * Initialize class properties to undefined
     */
    #resetProperties() {
        this.#settings = undefined;
        this.#settingsManager = undefined;
        this.#styleService = undefined;
        this.#notificationService = undefined;
        this.#dateMenu = undefined;
        this.#clock = undefined;
        this.#clockDisplay = undefined;
        this.#topBox = undefined;
        this.#clockLabel = undefined;
        this.#clockBinding = undefined;
        this.#translatePack = undefined;
    }
    // Retrieve the date menu from the status area
    #retrieveDateMenu() {
        this.#dateMenu = panel.statusArea.dateMenu;
        if (!this.#dateMenu) {
            return;
        }
        const { _clock, _clockDisplay } = this.#dateMenu;
        this.#clock = _clock;
        this.#clockDisplay = _clockDisplay;
    }
    // Place the clock label in the top box
    #placeClockLabel() {
        this.#translatePack = createTranslatePack(extensionGettext);
        // Create the top box
        this.#topBox = new St.BoxLayout({
            style_class: CLOCK_STYLE_CLASS_NAME,
        });
        // Create the clock label with current settings
        const currentStyles = this.#styleService.getCurrentStyles();
        this.#clockLabel = new TextClockLabel({
            translatePack: this.#translatePack,
            showDate: this.#settingsManager.getBoolean("show-date" /* SettingsKey.SHOW_DATE */),
            showWeekday: this.#settingsManager.getBoolean("show-weekday" /* SettingsKey.SHOW_WEEKDAY */),
            timeFormat: this.#settingsManager.getString("time-format" /* SettingsKey.TIME_FORMAT */),
            dividerText: currentStyles.dividerText || " | ",
        });
        const fuzzValue = this.#settingsManager.getFuzziness();
        this.#clockLabel.fuzzyMinutes = fuzzValue;
        this.#topBox.add_child(this.#clockLabel);
        this.#applyStyles();
        const clockDisplayBox = this.#findClockDisplayBox();
        clockDisplayBox.add_child(this.#topBox);
        this.#clockDisplay.remove_style_class_name(CLOCK_STYLE_CLASS_NAME);
        this.#clockDisplay.set_width(0);
        this.#clockDisplay.hide();
    }
    // Bind settings to their clock label properties
    #bindSettingsToClockLabel() {
        if (!this.#settingsManager || !this.#clockLabel) {
            logErr("Required services or clock label not available for binding");
            return;
        }
        this.#clockLabel.showDate = this.#settingsManager.getBoolean("show-date" /* SettingsKey.SHOW_DATE */);
        this.#clockLabel.showWeekday = this.#settingsManager.getBoolean("show-weekday" /* SettingsKey.SHOW_WEEKDAY */);
        this.#settingsManager.subscribe("show-date" /* SettingsKey.SHOW_DATE */, (newVal) => {
            this.#clockLabel.showDate = Boolean(newVal);
        });
        this.#settingsManager.subscribe("show-weekday" /* SettingsKey.SHOW_WEEKDAY */, (newVal) => {
            this.#clockLabel.showWeekday = Boolean(newVal);
        });
        // Bind wall clock to clock label - store the binding for cleanup
        this.#clockBinding = this.#clock.bind_property("clock", this.#clockLabel, CLOCK_LABEL_PROPERTIES.CLOCK_UPDATE, GObject.BindingFlags.DEFAULT);
        // Subscribe to fuzziness changes
        this.#settingsManager.subscribe("fuzziness" /* SettingsKey.FUZZINESS */, () => {
            const enumIndex = this.#settingsManager.getEnum("fuzziness" /* SettingsKey.FUZZINESS */, 1);
            const fuzzValue = fuzzinessFromEnumIndex(enumIndex);
            this.#clockLabel.fuzzyMinutes = fuzzValue;
        });
        // Subscribe to time format changes
        this.#settingsManager.subscribe("time-format" /* SettingsKey.TIME_FORMAT */, (newValue) => {
            if (newValue) {
                this.#clockLabel.timeFormat = newValue;
            }
        });
        this.#styleService.registerTarget(this.#clockLabel);
    }
    // Apply styles to the clock label
    #applyStyles() {
        if (!this.#clockLabel || !this.#styleService)
            return;
        this.#styleService.applyStyles(this.#clockLabel);
    }
    // Destroys created objects and sets properties to undefined
    #cleanup() {
        // Destroy services
        if (this.#styleService)
            this.#styleService.destroy();
        if (this.#settingsManager)
            this.#settingsManager.destroy();
        if (this.#notificationService)
            this.#notificationService.destroy();
        if (this.#systemSettingsMonitor)
            this.#systemSettingsMonitor.stop();
        // Destroy UI components
        if (this.#clockBinding) {
            this.#clockBinding.unbind();
            this.#clockBinding = undefined;
        }
        if (this.#clockLabel)
            this.#clockLabel.destroy();
        if (this.#topBox)
            this.#topBox.destroy();
        this.#resetProperties();
    }
    // Restore the clock display to its original appearance
    #restoreClockDisplay() {
        if (!this.#clockDisplay) {
            return;
        }
        this.#clockDisplay.add_style_class_name(CLOCK_STYLE_CLASS_NAME);
        this.#clockDisplay.set_width(-1);
        this.#clockDisplay.show();
    }
    // Finds the St.BoxLayout child with style class 'clock-display-box'
    #findClockDisplayBox() {
        const children = this.#dateMenu?.get_children
            ? this.#dateMenu.get_children()
            : [];
        const box = children.find((child) => child instanceof St.BoxLayout &&
            child.has_style_class_name("clock-display-box"));
        if (box) {
            return box;
        }
        throw new Error(_("Could not find clock display box"));
    }
}
