// src/utils/dbus-utils.ts
import Gio from "gi://Gio";
var createDBusService = (interfaceDefinition, serviceObject) => {
  try {
    const dbus = Gio.DBusExportedObject.wrapJSObject(
      interfaceDefinition,
      serviceObject
    );
    return dbus;
  } catch (error2) {
    throw new Error(`Failed to create D-Bus service: ${error2}`);
  }
};
var exportDBusService = (dbus, path) => {
  try {
    dbus.export(Gio.DBus.session, path);
    return true;
  } catch (error2) {
    throw new Error(`Failed to export D-Bus service at ${path}: ${error2}`);
  }
};
var unexportDBusService = (dbus) => {
  try {
    dbus.flush();
    dbus.unexport();
    return true;
  } catch (error2) {
    throw new Error(`Failed to unexport D-Bus service: ${error2}`);
  }
};

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};
var logger = {
  debug,
  info,
  warn,
  error
};

// src/core/dbus/interfaces/clipboard.ts
var CLIPBOARD_DBUS_IFACE = `
<node>
   <interface name="org.gnome.Shell.Extensions.Clipboard">
      <method name="ListenToClipboardChanges">
      </method>
      <method name="GetClipboardMimeTypes">
         <arg type="as" direction="out" name="mimeTypes" />
      </method>
      <method name="StopListening">
      </method>
      <method name="TriggerClipboardChange">
         <arg type="s" direction="in" name="content" />
         <arg type="s" direction="in" name="source" />
      </method>
      <method name="GetCurrentContent">
         <arg type="s" direction="out" name="content" />
      </method>
      <method name="SetContent">
         <arg type="s" direction="in" name="content" />
      </method>
      <method name="SetContentBinary">
         <arg type="ay" direction="in" name="content" />
         <arg type="s" direction="in" name="mimeType" />
      </method>
      <signal name="ClipboardChanged">
         <arg type="ay" name="content" />
         <arg type="s" name="mimeType" />
         <arg type="s" name="sourceApp" />
      </signal>
   </interface>
</node>`;

// src/core/dbus/interfaces/windows.ts
var WINDOWS_DBUS_IFACE = `
<node>
   <interface name="org.gnome.Shell.Extensions.Windows">
      <method name="List">
         <arg type="s" direction="out" name="win" />
      </method>
      <method name="Details">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="out" name="win" />
      </method>
      <method name="GetTitle">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="out" name="win" />
      </method>
      <method name="GetFrameRect">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="out" name="frameRect" />
      </method>
      <method name="GetFrameBounds">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="out" name="frameBounds" />
      </method>
      <method name="MoveToWorkspace">
         <arg type="u" direction="in" name="winid" />
         <arg type="u" direction="in" name="workspaceNum" />
      </method>
      <method name="MoveResize">
         <arg type="u" direction="in" name="winid" />
         <arg type="i" direction="in" name="x" />
         <arg type="i" direction="in" name="y" />
         <arg type="u" direction="in" name="width" />
         <arg type="u" direction="in" name="height" />
      </method>
      <method name="Resize">
         <arg type="u" direction="in" name="winid" />
         <arg type="u" direction="in" name="width" />
         <arg type="u" direction="in" name="height" />
      </method>
      <method name="Move">
         <arg type="u" direction="in" name="winid" />
         <arg type="i" direction="in" name="x" />
         <arg type="i" direction="in" name="y" />
      </method>
      <method name="Maximize">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Minimize">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Unmaximize">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Unminimize">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Activate">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Close">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="ListWorkspaces">
         <arg type="s" direction="out" name="workspaces" />
      </method>
      <method name="GetActiveWorkspace">
         <arg type="s" direction="out" name="workspace" />
      </method>
      <method name="GetWorkspaceWindows">
         <arg type="u" direction="in" name="workspaceIndex" />
         <arg type="s" direction="out" name="windows" />
      </method>
      <method name="SendShortcut">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="in" name="key" />
         <arg type="s" direction="in" name="modifiers" />
         <arg type="b" direction="out" name="success" />
      </method>
      <method name="GetFocusedWindowSync">
         <arg type="s" direction="out" name="window" />
      </method>
      <signal name="openwindow">
         <arg type="s" name="windowAddress" />
         <arg type="s" name="workspaceName" />
         <arg type="s" name="wmClass" />
         <arg type="s" name="title" />
      </signal>
      <signal name="closewindow">
         <arg type="s" name="windowAddress" />
      </signal>
      <signal name="focuswindow">
         <arg type="s" name="windowAddress" />
      </signal>
      <signal name="movewindow">
         <arg type="s" name="windowAddress" />
         <arg type="i" name="x" />
         <arg type="i" name="y" />
         <arg type="u" name="width" />
         <arg type="u" name="height" />
      </signal>
      <signal name="statewindow">
         <arg type="s" name="windowAddress" />
         <arg type="s" name="state" />
      </signal>
      <signal name="workspacechanged">
         <arg type="s" name="workspaceId" />
      </signal>
      <signal name="monitorlayoutchanged">
      </signal>
   </interface>
</node>`;

// src/core/dbus/services/clipboard-service.ts
import GLib from "gi://GLib";

// src/utils/window-utils.ts
var getWindowById = (winid) => {
  if (!winid || winid <= 0) return null;
  try {
    const windowActors = global.get_window_actors();
    return windowActors.find((w) => {
      try {
        return w.meta_window && w.meta_window.get_id() === winid;
      } catch {
        return false;
      }
    });
  } catch (_error) {
    return null;
  }
};
var getCurrentTime = () => {
  return global.get_current_time();
};
var getFocusedWindow = () => {
  const windowActors = global.get_window_actors();
  return windowActors.find((w) => w.meta_window.has_focus());
};
var getFocusedWindowApp = () => {
  const focusedWindow = getFocusedWindow();
  if (focusedWindow) {
    const wmClass = focusedWindow.meta_window.get_wm_class();
    const title = focusedWindow.meta_window.get_title();
    return wmClass || title || "unknown";
  }
  return "gnome-shell";
};
var isMaximized = (win) => {
  if (win.is_maximized !== void 0) {
    return win.is_maximized();
  }
  return win.get_maximized();
};

// src/utils/clipboard-utils.ts
var CLIPBOARD_CONFIG = {
  MAX_CLIPBOARD_SIZE: 10 * 1024 * 1024
  // 10MB - reasonable clipboard limit
};
function calculateClipboardMetadata(event) {
  const content = event.content;
  let mimeType = "text/plain";
  const sourceApp = getFocusedWindowApp();
  if (event.source === "image") {
    if (content.startsWith("data:image/")) {
      const match = content.match(/^data:(image\/[^;]+);/);
      mimeType = match ? match[1] : "image/png";
    }
  } else if (content.startsWith("data:")) {
    const match = content.match(/^data:([^;]+);/);
    mimeType = match ? match[1] : "application/octet-stream";
  } else {
    mimeType = "text/plain";
    if (content.includes("<") && content.includes(">") && (content.includes("<html") || content.includes("<div") || content.includes("<p"))) {
      mimeType = "text/html";
    }
  }
  return {
    mimeType,
    sourceApp
  };
}

// src/core/dbus/services/clipboard-service.ts
var ClipboardService = class {
  clipboardManager;
  dbusObject = null;
  clipboardListener = null;
  isListening = false;
  constructor(clipboardManager, _extension) {
    this.clipboardManager = clipboardManager;
    logger.info("ClipboardService initialized with binary-only protocol");
  }
  // Method to set the D-Bus exported object (called by DBusManager)
  setDBusObject(dbusObject) {
    this.dbusObject = dbusObject;
    this.clipboardListener = (event) => {
      try {
        if (!event.content || event.content.length === 0) {
          logger.debug("Skipping empty clipboard content");
          return;
        }
        const metadata = calculateClipboardMetadata(event);
        let content;
        let finalMimeType = metadata.mimeType;
        if (event.content.startsWith("[BINARY_IMAGE:")) {
          const binaryInfo = this.clipboardManager.getBinaryData(
            event.content
          );
          if (binaryInfo) {
            content = this.bufferLikeToUint8Array(binaryInfo.data);
            finalMimeType = binaryInfo.mimeType;
            logger.debug(
              `Using direct binary data: ${finalMimeType}, ${content.length} bytes`
            );
          } else {
            logger.warn(
              `Binary data not found for marker: ${event.content}`
            );
            return;
          }
        } else {
          content = new TextEncoder().encode(event.content);
        }
        if (content.length > CLIPBOARD_CONFIG.MAX_CLIPBOARD_SIZE) {
          logger.warn(
            `Clipboard data too large: ${content.length} bytes, skipping`
          );
          return;
        }
        logger.debug(
          `Processing clipboard: ${finalMimeType}, ${content.length} bytes from ${metadata.sourceApp}`
        );
        this.emitBinarySignal(content, {
          mimeType: finalMimeType,
          sourceApp: metadata.sourceApp
        });
      } catch (signalError) {
        logger.error("Error processing clipboard event", {
          error: signalError,
          errorType: typeof signalError,
          errorMessage: signalError instanceof Error ? signalError.message : String(signalError),
          stack: signalError instanceof Error ? signalError.stack : void 0
        });
      }
    };
    this.clipboardManager.onClipboardChange(this.clipboardListener);
    this.isListening = true;
    logger.info("\u{1F4E1} Binary-only D-Bus clipboard listener activated");
  }
  bufferLikeToUint8Array(buffer) {
    if (buffer instanceof Uint8Array) {
      return buffer;
    }
    if (buffer instanceof ArrayBuffer) {
      return new Uint8Array(buffer);
    }
    if (buffer && typeof buffer === "object" && "length" in buffer) {
      const uint8Array = new Uint8Array(buffer.length);
      for (let i = 0; i < buffer.length; i++) {
        uint8Array[i] = buffer[i];
      }
      return uint8Array;
    }
    throw new Error(`Unsupported buffer type: ${typeof buffer}`);
  }
  emitBinarySignal(content, metadata) {
    try {
      logger.debug(
        `Emitting binary signal: ${metadata.mimeType}, ${content.length} bytes`
      );
      this.dbusObject?.emit_signal(
        "ClipboardChanged",
        GLib.Variant.new("(ayss)", [
          content,
          String(metadata.mimeType),
          String(metadata.sourceApp)
        ])
      );
      logger.info(
        `Binary signal emitted: ${metadata.mimeType}, ${content.length} bytes`
      );
    } catch (error2) {
      logger.error("Failed to emit binary signal", error2);
      throw error2;
    }
  }
  // Method to start listening to clipboard changes
  ListenToClipboardChanges() {
    try {
      if (!this.isListening && this.clipboardListener) {
        logger.debug("D-Bus: Starting clipboard listener...");
        this.clipboardManager.onClipboardChange(this.clipboardListener);
        this.isListening = true;
        logger.info(
          "\u{1F4E1} Binary D-Bus clipboard listener activated via method call"
        );
      } else if (this.isListening) {
        logger.debug("D-Bus: Clipboard listener already active");
      } else {
        logger.warn("D-Bus: No clipboard listener available");
      }
    } catch (error2) {
      logger.error("D-Bus: Error starting clipboard listener", error2);
      throw error2;
    }
  }
  // Method to manually trigger a clipboard change (for testing)
  TriggerClipboardChange(content, source = "user") {
    try {
      this.clipboardManager.triggerClipboardChange(
        content,
        source
      );
    } catch (error2) {
      logger.error("D-Bus: Error triggering clipboard change", error2);
      throw error2;
    }
  }
  // Method to get current clipboard content
  GetCurrentContent() {
    try {
      return this.clipboardManager.getCurrentContent();
    } catch (error2) {
      logger.error(
        "D-Bus: Error getting current clipboard content",
        error2
      );
      throw error2;
    }
  }
  // Method to set clipboard content
  SetContent(content) {
    try {
      this.clipboardManager.setContent(content);
    } catch (error2) {
      logger.error("D-Bus: Error setting clipboard content", error2);
      throw error2;
    }
  }
  // Method to set binary clipboard content
  SetContentBinary(content, mimeType) {
    try {
      this.clipboardManager.setContentBinary(content, mimeType);
    } catch (error2) {
      logger.error(
        "D-Bus: Error setting binary clipboard content",
        error2
      );
      throw error2;
    }
  }
  // Method to get available MIME types
  GetClipboardMimeTypes() {
    try {
      return [
        "text/plain",
        "text/html",
        "image/png",
        "image/jpeg",
        "image/gif",
        "image/webp",
        "application/x-vicinae-concealed"
      ];
    } catch (error2) {
      logger.error("D-Bus: Error getting clipboard MIME types", error2);
      return [];
    }
  }
  // Method to stop listening to clipboard changes
  StopListening() {
    try {
      if (this.clipboardListener && this.isListening) {
        this.clipboardManager.removeClipboardListener(
          this.clipboardListener
        );
        this.clipboardListener = null;
        this.isListening = false;
        logger.info("\u{1F515} Binary D-Bus clipboard listener deactivated");
      }
    } catch (error2) {
      logger.error("Error stopping clipboard listener", error2);
    }
  }
  // Cleanup method
  destroy() {
    this.StopListening();
    this.dbusObject = null;
    logger.info("ClipboardService destroyed");
  }
};

// src/core/dbus/services/windows-service.ts
import GLib3 from "gi://GLib";

// src/core/windows/window-manager.ts
import GLib2 from "gi://GLib";
import Meta from "gi://Meta";

// src/core/windows/workspace-manager.ts
var WorkspaceManager = class {
  getWorkspaceCount() {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_n_workspaces();
    } catch (error2) {
      logger.error("Error getting workspace count", error2);
      return 0;
    }
  }
  getCurrentWorkspaceIndex() {
    try {
      const workspaceManager = global.workspace_manager;
      const currentWorkspace = workspaceManager.get_active_workspace();
      return currentWorkspace.index();
    } catch (error2) {
      logger.error("Error getting current workspace index", error2);
      return 0;
    }
  }
  getWorkspaceByIndex(index) {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_workspace_by_index(index);
    } catch (error2) {
      logger.error("Error getting workspace by index", error2);
      return null;
    }
  }
  switchToWorkspace(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        workspace.activate(global.get_current_time());
      }
    } catch (error2) {
      logger.error("Error switching to workspace", error2);
      throw error2;
    }
  }
  getWorkspaceInfo(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        const windows = workspace.list_windows();
        const hasFullscreen = windows.some(
          (win) => isMaximized(win) === 3
        );
        let monitor = 0;
        if (windows.length > 0) {
          monitor = windows[0].get_monitor();
        }
        return {
          index: workspace.index(),
          name: `Workspace ${workspace.index() + 1}`,
          isActive: workspace === global.workspace_manager.get_active_workspace(),
          windowCount: windows.length,
          monitor,
          hasfullscreen: hasFullscreen
        };
      }
      return null;
    } catch (error2) {
      logger.error("Error getting workspace info", error2);
      return null;
    }
  }
  getAllWorkspaces() {
    try {
      const workspaceManager = global.workspace_manager;
      const workspaces = [];
      for (let i = 0; i < workspaceManager.get_n_workspaces(); i++) {
        const workspace = workspaceManager.get_workspace_by_index(i);
        if (workspace) {
          const workspaceInfo = this.getWorkspaceInfo(i);
          if (workspaceInfo) {
            workspaces.push(workspaceInfo);
          }
        }
      }
      return workspaces;
    } catch (error2) {
      logger.error("Error getting all workspaces", error2);
      return [];
    }
  }
};

// src/core/windows/window-manager.ts
var VicinaeWindowManager = class {
  clipboardManager;
  PASTE_DELAY = 100;
  appClass;
  constructor(clipboardManager, appClass) {
    this.appClass = appClass;
    this.clipboardManager = clipboardManager;
  }
  isTargetWindow(wmClass) {
    return wmClass.toLowerCase().includes(this.appClass.toLowerCase()) || this.appClass.toLowerCase().includes(wmClass.toLowerCase());
  }
  list() {
    const windowActors = global.get_window_actors();
    const workspaceManager = global.workspace_manager;
    const windows = windowActors.map((w) => {
      const metaWindow = w.meta_window;
      const windowWorkspace = metaWindow.get_workspace();
      const frame = metaWindow.get_frame_rect();
      return {
        id: metaWindow.get_id(),
        title: metaWindow.get_title(),
        wm_class: metaWindow.get_wm_class(),
        wm_class_instance: metaWindow.get_wm_class_instance(),
        pid: metaWindow.get_pid(),
        maximized: isMaximized(metaWindow) !== 0,
        // 0 means not maximized
        display: metaWindow.get_display(),
        frame_type: metaWindow.get_frame_type(),
        window_type: metaWindow.get_window_type(),
        layer: metaWindow.get_layer(),
        monitor: metaWindow.get_monitor(),
        role: metaWindow.get_role(),
        width: frame.width,
        height: frame.height,
        x: frame.x,
        y: frame.y,
        in_current_workspace: metaWindow.located_on_workspace?.(
          workspaceManager.get_active_workspace?.()
        ),
        canclose: metaWindow.can_close(),
        canmaximize: metaWindow.can_maximize(),
        canminimize: metaWindow.can_minimize(),
        canshade: false,
        // can_shade() is not in the type definitions
        moveable: metaWindow.allows_move(),
        resizeable: metaWindow.allows_resize(),
        has_focus: metaWindow.has_focus(),
        workspace: windowWorkspace ? windowWorkspace.index() : -1
      };
    });
    return windows;
  }
  details(winid) {
    const w = getWindowById(winid);
    if (!w) {
      throw new Error("Window not found");
    }
    const metaWindow = w.meta_window;
    const workspaceManager = global.workspace_manager;
    const windowWorkspace = metaWindow.get_workspace();
    const frame = metaWindow.get_frame_rect();
    const win = {
      id: metaWindow.get_id(),
      title: metaWindow.get_title(),
      wm_class: metaWindow.get_wm_class(),
      wm_class_instance: metaWindow.get_wm_class_instance(),
      pid: metaWindow.get_pid(),
      maximized: isMaximized(metaWindow) !== 0,
      // 0 means not maximized
      display: metaWindow.get_display(),
      frame_type: metaWindow.get_frame_type(),
      window_type: metaWindow.get_window_type(),
      layer: metaWindow.get_layer(),
      monitor: metaWindow.get_monitor(),
      role: metaWindow.get_role(),
      width: frame.width,
      height: frame.height,
      x: frame.x,
      y: frame.y,
      in_current_workspace: metaWindow.located_on_workspace?.(
        workspaceManager.get_active_workspace?.()
      ),
      canclose: metaWindow.can_close(),
      canmaximize: metaWindow.can_maximize(),
      canminimize: metaWindow.can_minimize(),
      canshade: false,
      // can_shade() is not in the type definitions
      moveable: metaWindow.allows_move(),
      resizeable: metaWindow.allows_resize(),
      has_focus: metaWindow.has_focus(),
      workspace: windowWorkspace ? windowWorkspace.index() : -1
    };
    return win;
  }
  getTitle(winid) {
    const w = getWindowById(winid);
    if (w) {
      return w.meta_window.get_title();
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameRect(winid) {
    const w = getWindowById(winid);
    if (w) {
      const frame = w.meta_window.get_frame_rect();
      return {
        x: frame.x,
        y: frame.y,
        width: frame.width,
        height: frame.height
      };
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameBounds(winid) {
    const w = getWindowById(winid);
    if (w) {
      return {
        frame_bounds: w.meta_window.get_frame_bounds()
      };
    } else {
      throw new Error("Window not found");
    }
  }
  moveToWorkspace(winid, workspaceNum) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.change_workspace_by_index(workspaceNum, false);
    } else {
      throw new Error("Window not found");
    }
  }
  moveResize(winid, x, y, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(true, x, y, width, height);
    } else {
      throw new Error("Window not found");
    }
  }
  resize(winid, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(
        true,
        win.get_x(),
        win.get_y(),
        width,
        height
      );
    } else {
      throw new Error("Window not found");
    }
  }
  move(winid, x, y) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_frame(true, x, y);
    } else {
      throw new Error("Window not found");
    }
  }
  maximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.maximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  minimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.minimize();
    } else {
      throw new Error("Window not found");
    }
  }
  unmaximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unmaximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  unminimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unminimize();
    } else {
      throw new Error("Window not found");
    }
  }
  activate(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      const currentTime = getCurrentTime();
      const workspace = win.get_workspace();
      if (workspace) {
        workspace.activate_with_focus(win, currentTime);
      } else {
        win.activate(currentTime);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  close(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      try {
        if (win.get_id() === winid) {
          win.delete(getCurrentTime());
        } else {
          throw new Error(
            "Window ID mismatch - window may be destroyed"
          );
        }
      } catch (error2) {
        throw new Error(`Failed to close window ${winid}: ${error2}`);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  listWorkspaces() {
    const workspaceManager = new WorkspaceManager();
    return workspaceManager.getAllWorkspaces();
  }
  getActiveWorkspace() {
    const workspaceManager = new WorkspaceManager();
    const currentIndex = workspaceManager.getCurrentWorkspaceIndex();
    const workspace = workspaceManager.getWorkspaceInfo(currentIndex);
    if (!workspace) {
      throw new Error("No active workspace found");
    }
    return workspace;
  }
  pasteToFocusedWindow(winid, key, modifiers) {
    if (!this.clipboardManager) {
      logger.error("No clipboard manager available for paste operation");
      return false;
    }
    GLib2.timeout_add(GLib2.PRIORITY_DEFAULT, this.PASTE_DELAY, () => {
      logger.debug(
        `Sending keyboard paste for window ${winid} ${key} ${modifiers} after ${this.PASTE_DELAY}ms delay`
      );
      this.clipboardManager.triggerKeyboardPaste();
      return false;
    });
    logger.debug(
      `Triggered keyboard paste for window ${winid} ${key} ${modifiers}`
    );
    return true;
  }
  sendShortcut(winid, key, modifiers) {
    this.activate(winid);
    const modifiersArray = modifiers.split("|");
    const isNormalPaste = key === "v" && modifiersArray.length === 1 && modifiersArray[0] === "CONTROL";
    const isShiftPaste = key === "v" && modifiersArray.length === 2 && modifiersArray[0] === "CONTROL" && modifiersArray[1] === "SHIFT";
    if (isNormalPaste || isShiftPaste) {
      return this.pasteToFocusedWindow(winid, key, modifiers);
    } else {
      logger.warn(`Unhandled shortcut: ${key} ${modifiers}`);
      return false;
    }
  }
};

// src/core/dbus/services/windows-service.ts
var WindowsService = class {
  windowManager;
  dbusObject = null;
  // Signal connection IDs for cleanup
  windowOpenedSignalId = 0;
  windowFocusSignalId = 0;
  workspaceChangedSignalId = 0;
  // Track individual window signal IDs
  windowDestroySignalIds = /* @__PURE__ */ new Map();
  windowSizeSignalIds = /* @__PURE__ */ new Map();
  // Track previous focused window for paste-to-active-app functionality
  previousFocusedWindow = null;
  constructor(clipboardManager, appClass) {
    this.windowManager = new VicinaeWindowManager(
      clipboardManager,
      appClass
    );
  }
  // Method to set the D-Bus exported object (called by DBusManager)
  setDBusObject(dbusObject) {
    this.dbusObject = dbusObject;
    logger.debug("WindowsService: D-Bus object set for signal emission");
    this.setupWindowEventListeners();
  }
  setupWindowEventListeners() {
    logger.debug("WindowsService: Setting up GNOME window event listeners");
    const _display = global.display;
    this.windowOpenedSignalId = global.display.connect(
      "window-created",
      (_display2, window) => {
        try {
          const windowInfo = this.getWindowInfo(window);
          this.emitOpenWindow(
            windowInfo.id.toString(),
            windowInfo.workspace.toString(),
            windowInfo.wm_class,
            windowInfo.title
          );
          this.connectToWindowDestroy(window);
          this.connectToWindowSizeChanges(window);
        } catch (error2) {
          logger.debug(
            `Error handling window opened event: ${error2}`
          );
        }
      }
    );
    this.connectToExistingWindows();
    this.windowFocusSignalId = global.display.connect(
      "notify::focus-window",
      () => {
        try {
          GLib3.idle_add(GLib3.PRIORITY_DEFAULT_IDLE, () => {
            const focusWindow = global.display.focus_window;
            if (focusWindow) {
              const currentWmClass = focusWindow.get_wm_class() || "";
              if (!this.windowManager.isTargetWindow(
                currentWmClass
              )) {
                this.previousFocusedWindow = {
                  id: focusWindow.get_id(),
                  wmClass: currentWmClass
                };
              }
              this.emitFocusWindow(
                focusWindow.get_id().toString()
              );
            }
            return GLib3.SOURCE_REMOVE;
          });
        } catch (error2) {
          logger.debug(`Error handling window focus event: ${error2}`);
        }
      }
    );
    this.workspaceChangedSignalId = global.workspace_manager?.connect(
      "notify::active-workspace",
      () => {
        try {
          const activeWorkspace = global.workspace_manager?.get_active_workspace();
          if (activeWorkspace) {
            this.emitWorkspaceChanged(
              activeWorkspace.index().toString()
            );
          }
        } catch (error2) {
          logger.debug(
            `Error handling workspace changed event: ${error2}`
          );
        }
      }
    );
    logger.debug(
      "WindowsService: GNOME window event listeners set up successfully"
    );
  }
  connectToWindowDestroy(window) {
    const windowId = window.get_id();
    try {
      let signalId;
      let connectedSignal = "";
      try {
        logger.debug(
          `Attempting to connect 'destroy' signal for window ${windowId}`
        );
        signalId = window.connect("destroy", () => {
          try {
            logger.debug(
              `Window ${windowId} destroy signal triggered - emitting closewindow`
            );
            this.emitCloseWindow(windowId.toString());
            this.windowDestroySignalIds.delete(windowId);
          } catch (error2) {
            logger.debug(
              `Error emitting closewindow for ${windowId}: ${error2}`
            );
          }
        });
        connectedSignal = "destroy";
        logger.debug(
          `Successfully connected to 'destroy' signal for window ${windowId}`
        );
      } catch (_destroyError) {
        logger.debug(
          `'destroy' signal not available for window ${windowId}, trying 'unmanaged'`
        );
        try {
          signalId = window.connect("unmanaged", () => {
            try {
              logger.debug(
                `Window ${windowId} unmanaged signal triggered - emitting closewindow`
              );
              this.emitCloseWindow(windowId.toString());
              this.windowDestroySignalIds.delete(windowId);
            } catch (error2) {
              logger.debug(
                `Error emitting closewindow for ${windowId}: ${error2}`
              );
            }
          });
          connectedSignal = "unmanaged";
          logger.debug(
            `Successfully connected to 'unmanaged' signal for window ${windowId}`
          );
        } catch (_unmanagedError) {
          logger.debug(
            `No suitable destroy signal for window ${windowId}, skipping signal connection`
          );
          return;
        }
      }
      if (signalId !== void 0) {
        this.windowDestroySignalIds.set(windowId, signalId);
        logger.debug(
          `Successfully connected ${connectedSignal} signal for window ${windowId} (signal ID: ${signalId})`
        );
      }
    } catch (error2) {
      logger.debug(
        `Failed to connect any destroy signal for window ${windowId}: ${error2}`
      );
    }
  }
  connectToWindowSizeChanges(window) {
    const windowId = window.get_id();
    try {
      const signalId = window.connect("size-changed", () => {
        try {
          const windowInfo = this.getWindowInfo(window);
          logger.debug(
            `Window ${windowId} size changed - emitting movewindow`
          );
          this.emitMoveWindow(
            windowInfo.id.toString(),
            windowInfo.x,
            windowInfo.y,
            windowInfo.width,
            windowInfo.height
          );
        } catch (error2) {
          logger.debug(
            `Error handling size change for window ${windowId}: ${error2}`
          );
        }
      });
      this.windowSizeSignalIds.set(windowId, signalId);
      logger.debug(
        `Connected size-changed signal for window ${windowId}`
      );
    } catch (error2) {
      logger.debug(
        `Failed to connect size-changed signal for window ${windowId}: ${error2}`
      );
    }
  }
  connectToExistingWindows() {
    try {
      const windowActors = global.get_window_actors();
      logger.debug(
        `WindowsService: Connecting to ${windowActors.length} existing windows`
      );
      for (const actor of windowActors) {
        if (actor.meta_window) {
          this.connectToWindowDestroy(actor.meta_window);
          this.connectToWindowSizeChanges(actor.meta_window);
        }
      }
    } catch (error2) {
      logger.debug(`Error connecting to existing windows: ${error2}`);
    }
  }
  getWindowInfo(window) {
    let x = 0, y = 0, width = 0, height = 0;
    try {
      if (typeof window.get_frame_rect === "function") {
        const frame = window.get_frame_rect();
        x = frame.x;
        y = frame.y;
        width = frame.width;
        height = frame.height;
      } else {
        logger.debug(
          `Window ${window.get_id()} does not have get_frame_rect method`
        );
      }
    } catch (error2) {
      logger.debug(
        `Error getting frame rect for window ${window.get_id()}: ${error2}`
      );
    }
    const workspace = window.get_workspace();
    return {
      id: window.get_id(),
      title: window.get_title(),
      wm_class: window.get_wm_class() || "",
      workspace: workspace ? workspace.index() : -1,
      x,
      y,
      width,
      height
    };
  }
  // Cleanup method to disconnect signals
  destroy() {
    logger.debug("WindowsService: Cleaning up window event listeners");
    if (this.windowOpenedSignalId) {
      global.display.disconnect(this.windowOpenedSignalId);
    }
    if (this.windowFocusSignalId) {
      global.display.disconnect(this.windowFocusSignalId);
    }
    if (this.workspaceChangedSignalId && global.workspace_manager) {
      global.workspace_manager.disconnect(this.workspaceChangedSignalId);
    }
    const allWindowIds = /* @__PURE__ */ new Set([
      ...this.windowDestroySignalIds.keys(),
      ...this.windowSizeSignalIds.keys()
    ]);
    for (const windowId of allWindowIds) {
      try {
        const windowActors = global.get_window_actors();
        const windowActor = windowActors.find(
          (actor) => actor.meta_window?.get_id() === windowId
        );
        if (windowActor?.meta_window) {
          const destroySignalId = this.windowDestroySignalIds.get(windowId);
          if (destroySignalId) {
            windowActor.meta_window.disconnect(destroySignalId);
          }
          const sizeSignalId = this.windowSizeSignalIds.get(windowId);
          if (sizeSignalId) {
            windowActor.meta_window.disconnect(sizeSignalId);
          }
        }
      } catch (error2) {
        logger.debug(
          `Error disconnecting signals for window ${windowId}: ${error2}`
        );
      }
    }
    this.windowDestroySignalIds.clear();
    this.windowSizeSignalIds.clear();
    logger.debug("WindowsService: Window event listeners cleaned up");
  }
  List() {
    try {
      GLib3.usleep(1e3);
      const windows = this.windowManager.list();
      const filteredWindows = windows.filter(
        (window) => !this.windowManager.isTargetWindow(window.wm_class)
      );
      return JSON.stringify(filteredWindows);
    } catch (error2) {
      logger.error("D-Bus: Error listing windows", error2);
      throw error2;
    }
  }
  Details(winid) {
    try {
      const details = this.windowManager.details(winid);
      return JSON.stringify(details);
    } catch (error2) {
      logger.error("D-Bus: Error getting window details", error2);
      throw error2;
    }
  }
  GetTitle(winid) {
    try {
      return this.windowManager.getTitle(winid);
    } catch (error2) {
      logger.error("D-Bus: Error getting window title", error2);
      throw error2;
    }
  }
  GetFrameRect(winid) {
    try {
      const frameRect = this.windowManager.getFrameRect(winid);
      return JSON.stringify(frameRect);
    } catch (error2) {
      logger.error("D-Bus: Error getting window frame rect", error2);
      throw error2;
    }
  }
  GetFrameBounds(winid) {
    try {
      const frameBounds = this.windowManager.getFrameBounds(winid);
      return JSON.stringify(frameBounds);
    } catch (error2) {
      logger.error("D-Bus: Error getting window frame bounds", error2);
      throw error2;
    }
  }
  MoveToWorkspace(winid, workspaceNum) {
    try {
      this.windowManager.moveToWorkspace(winid, workspaceNum);
    } catch (error2) {
      logger.error("D-Bus: Error moving window to workspace", error2);
      throw error2;
    }
  }
  MoveResize(winid, x, y, width, height) {
    try {
      this.windowManager.moveResize(winid, x, y, width, height);
    } catch (error2) {
      logger.error("D-Bus: Error move resizing window", error2);
      throw error2;
    }
  }
  Resize(winid, width, height) {
    try {
      this.windowManager.resize(winid, width, height);
    } catch (error2) {
      logger.error("D-Bus: Error resizing window", error2);
      throw error2;
    }
  }
  Move(winid, x, y) {
    try {
      this.windowManager.move(winid, x, y);
    } catch (error2) {
      logger.error("D-Bus: Error moving window", error2);
      throw error2;
    }
  }
  Maximize(winid) {
    try {
      this.windowManager.maximize(winid);
    } catch (error2) {
      logger.error("D-Bus: Error maximizing window", error2);
      throw error2;
    }
  }
  Minimize(winid) {
    try {
      this.windowManager.minimize(winid);
    } catch (error2) {
      logger.error("D-Bus: Error minimizing window", error2);
      throw error2;
    }
  }
  Unmaximize(winid) {
    try {
      this.windowManager.unmaximize(winid);
    } catch (error2) {
      logger.error("D-Bus: Error unmaximizing window", error2);
      throw error2;
    }
  }
  Unminimize(winid) {
    try {
      this.windowManager.unminimize(winid);
    } catch (error2) {
      logger.error("D-Bus: Error unminimizing window", error2);
      throw error2;
    }
  }
  Activate(winid) {
    try {
      this.windowManager.activate(winid);
    } catch (error2) {
      logger.error("D-Bus: Error activating window", error2);
      throw error2;
    }
  }
  Close(winid) {
    try {
      this.windowManager.close(winid);
    } catch (error2) {
      logger.error("D-Bus: Error closing window", error2);
      throw error2;
    }
  }
  ListWorkspaces() {
    try {
      const workspaces = this.windowManager.listWorkspaces();
      return JSON.stringify(workspaces);
    } catch (error2) {
      logger.error("D-Bus: Error listing workspaces", error2);
      throw error2;
    }
  }
  GetActiveWorkspace() {
    try {
      const workspace = this.windowManager.getActiveWorkspace();
      return JSON.stringify(workspace);
    } catch (error2) {
      logger.error("D-Bus: Error getting active workspace", error2);
      throw error2;
    }
  }
  GetWorkspaceWindows(workspaceIndex) {
    try {
      const windows = this.windowManager.list();
      const workspaceWindows = windows.filter(
        (win) => win.workspace === workspaceIndex
      );
      return JSON.stringify(workspaceWindows);
    } catch (error2) {
      logger.error("D-Bus: Error getting workspace windows", error2);
      throw error2;
    }
  }
  SendShortcut(winid, key, modifiers) {
    let success = false;
    try {
      success = this.windowManager.sendShortcut(winid, key, modifiers);
    } catch (error2) {
      logger.error("D-Bus: Error sending shortcut", error2);
      return false;
    }
    return success;
  }
  GetFocusedWindowSync() {
    try {
      const focusedWindow = getFocusedWindow();
      if (!focusedWindow) {
        logger.debug("GetFocusedWindowSync: No focused window");
        return JSON.stringify(null);
      }
      const currentWmClass = focusedWindow.meta_window.get_wm_class() || "";
      if (this.windowManager.isTargetWindow(currentWmClass)) {
        if (this.previousFocusedWindow) {
          logger.debug(
            `GetFocusedWindowSync: Vicinae focused, returning previous window: ${this.previousFocusedWindow.wmClass}`
          );
          const windowDetails2 = this.windowManager.details(
            this.previousFocusedWindow.id
          );
          return JSON.stringify(windowDetails2);
        } else {
          logger.debug(
            "GetFocusedWindowSync: Vicinae focused but no previous window"
          );
          return JSON.stringify(null);
        }
      }
      logger.debug(
        `GetFocusedWindowSync: Returning current focused window: ${currentWmClass}`
      );
      const windowDetails = this.windowManager.details(
        focusedWindow.meta_window.get_id()
      );
      return JSON.stringify(windowDetails);
    } catch (error2) {
      logger.error("D-Bus: Error in GetFocusedWindowSync", error2);
      return JSON.stringify(null);
    }
  }
  // Signal emission methods - called by window manager when events occur
  emitOpenWindow(windowAddress, workspaceName, wmClass, title) {
    try {
      logger.debug(
        `Emitting openwindow signal for ${title} (${wmClass})`
      );
      this.dbusObject?.emit_signal(
        "openwindow",
        GLib3.Variant.new("(ssss)", [
          String(windowAddress),
          String(workspaceName),
          String(wmClass),
          String(title)
        ])
      );
    } catch (signalError) {
      logger.error("Error emitting openwindow signal", signalError);
    }
  }
  emitCloseWindow(windowAddress) {
    try {
      logger.debug(`Emitting closewindow signal for ${windowAddress}`);
      this.dbusObject?.emit_signal(
        "closewindow",
        GLib3.Variant.new("(s)", [String(windowAddress)])
      );
    } catch (signalError) {
      logger.error("Error emitting closewindow signal", signalError);
    }
  }
  emitFocusWindow(windowAddress) {
    try {
      logger.debug(`Emitting focuswindow signal for ${windowAddress}`);
      this.dbusObject?.emit_signal(
        "focuswindow",
        GLib3.Variant.new("(s)", [String(windowAddress)])
      );
    } catch (signalError) {
      logger.error("Error emitting focuswindow signal", signalError);
    }
  }
  emitMoveWindow(windowAddress, x, y, width, height) {
    try {
      logger.debug(`Emitting movewindow signal for ${windowAddress}`);
      this.dbusObject?.emit_signal(
        "movewindow",
        GLib3.Variant.new("(siiuu)", [
          String(windowAddress),
          x,
          y,
          width,
          height
        ])
      );
    } catch (signalError) {
      logger.error("Error emitting movewindow signal", signalError);
    }
  }
  emitStateWindow(windowAddress, state) {
    try {
      logger.debug(
        `Emitting statewindow signal for ${windowAddress}: ${state}`
      );
      this.dbusObject?.emit_signal(
        "statewindow",
        GLib3.Variant.new("(ss)", [
          String(windowAddress),
          String(state)
        ])
      );
    } catch (signalError) {
      logger.error("Error emitting statewindow signal", signalError);
    }
  }
  emitWorkspaceChanged(workspaceId) {
    try {
      logger.debug(`Emitting workspacechanged signal for ${workspaceId}`);
      this.dbusObject?.emit_signal(
        "workspacechanged",
        GLib3.Variant.new("(s)", [String(workspaceId)])
      );
    } catch (signalError) {
      logger.error("Error emitting workspacechanged signal", signalError);
    }
  }
  emitMonitorLayoutChanged() {
    try {
      logger.debug("Emitting monitorlayoutchanged signal");
      this.dbusObject?.emit_signal(
        "monitorlayoutchanged",
        GLib3.Variant.new("()", [])
      );
    } catch (signalError) {
      logger.error(
        "Error emitting monitorlayoutchanged signal",
        signalError
      );
    }
  }
};

// src/core/dbus/manager.ts
var DBusManager = class {
  clipboardService;
  windowsService;
  clipboardServiceInstance;
  windowsServiceInstance;
  constructor(appClass, extension, clipboardManager) {
    if (!clipboardManager) {
      throw new Error(
        "ClipboardManager instance is required for DBusManager"
      );
    }
    this.clipboardServiceInstance = new ClipboardService(
      clipboardManager,
      extension
    );
    this.windowsServiceInstance = new WindowsService(
      clipboardManager,
      appClass
    );
    this.clipboardService = createDBusService(
      CLIPBOARD_DBUS_IFACE,
      // biome-ignore lint/suspicious/noExplicitAny: we need to cast the instance to any to avoid type errors
      this.clipboardServiceInstance
    );
    this.windowsService = createDBusService(
      WINDOWS_DBUS_IFACE,
      // biome-ignore lint/suspicious/noExplicitAny: we need to cast the instance to any to avoid type errors
      this.windowsServiceInstance
    );
    this.clipboardServiceInstance.setDBusObject(this.clipboardService);
    this.windowsServiceInstance.setDBusObject(this.windowsService);
  }
  exportServices() {
    try {
      exportDBusService(
        this.clipboardService,
        "/org/gnome/Shell/Extensions/Clipboard"
      );
      exportDBusService(
        this.windowsService,
        "/org/gnome/Shell/Extensions/Windows"
      );
      logger.info("D-Bus services exported successfully");
    } catch (_error) {
      logger.error("Failed to export D-Bus services", _error);
      throw _error;
    }
  }
  unexportServices() {
    try {
      this.clipboardServiceInstance.destroy();
      this.windowsServiceInstance.destroy();
      unexportDBusService(this.clipboardService);
      unexportDBusService(this.windowsService);
      logger.info("D-Bus services unexported successfully");
    } catch (_error) {
      logger.error("Failed to unexport D-Bus services", _error);
      throw _error;
    }
  }
  getClipboardService() {
    return this.clipboardServiceInstance;
  }
  getWindowsService() {
    return this.windowsServiceInstance;
  }
};
export {
  DBusManager
};
