// src/core/dbus/services/clipboard-service.ts
import GLib from "gi://GLib";

// src/utils/window-utils.ts
var getFocusedWindow = () => {
  const windowActors = global.get_window_actors();
  return windowActors.find((w) => w.meta_window.has_focus());
};
var getFocusedWindowApp = () => {
  const focusedWindow = getFocusedWindow();
  if (focusedWindow) {
    const wmClass = focusedWindow.meta_window.get_wm_class();
    const title = focusedWindow.meta_window.get_title();
    return wmClass || title || "unknown";
  }
  return "gnome-shell";
};

// src/utils/clipboard-utils.ts
var CLIPBOARD_CONFIG = {
  MAX_CLIPBOARD_SIZE: 10 * 1024 * 1024
  // 10MB - reasonable clipboard limit
};
function calculateClipboardMetadata(event) {
  const content = event.content;
  let mimeType = "text/plain";
  const sourceApp = getFocusedWindowApp();
  if (event.source === "image") {
    if (content.startsWith("data:image/")) {
      const match = content.match(/^data:(image\/[^;]+);/);
      mimeType = match ? match[1] : "image/png";
    }
  } else if (content.startsWith("data:")) {
    const match = content.match(/^data:([^;]+);/);
    mimeType = match ? match[1] : "application/octet-stream";
  } else {
    mimeType = "text/plain";
    if (content.includes("<") && content.includes(">") && (content.includes("<html") || content.includes("<div") || content.includes("<p"))) {
      mimeType = "text/html";
    }
  }
  return {
    mimeType,
    sourceApp
  };
}

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};
var logger = {
  debug,
  info,
  warn,
  error
};

// src/core/dbus/services/clipboard-service.ts
var ClipboardService = class {
  clipboardManager;
  dbusObject = null;
  clipboardListener = null;
  isListening = false;
  constructor(clipboardManager, _extension) {
    this.clipboardManager = clipboardManager;
    logger.info("ClipboardService initialized with binary-only protocol");
  }
  // Method to set the D-Bus exported object (called by DBusManager)
  setDBusObject(dbusObject) {
    this.dbusObject = dbusObject;
    this.clipboardListener = (event) => {
      try {
        if (!event.content || event.content.length === 0) {
          logger.debug("Skipping empty clipboard content");
          return;
        }
        const metadata = calculateClipboardMetadata(event);
        let content;
        let finalMimeType = metadata.mimeType;
        if (event.content.startsWith("[BINARY_IMAGE:")) {
          const binaryInfo = this.clipboardManager.getBinaryData(
            event.content
          );
          if (binaryInfo) {
            content = this.bufferLikeToUint8Array(binaryInfo.data);
            finalMimeType = binaryInfo.mimeType;
            logger.debug(
              `Using direct binary data: ${finalMimeType}, ${content.length} bytes`
            );
          } else {
            logger.warn(
              `Binary data not found for marker: ${event.content}`
            );
            return;
          }
        } else {
          content = new TextEncoder().encode(event.content);
        }
        if (content.length > CLIPBOARD_CONFIG.MAX_CLIPBOARD_SIZE) {
          logger.warn(
            `Clipboard data too large: ${content.length} bytes, skipping`
          );
          return;
        }
        logger.debug(
          `Processing clipboard: ${finalMimeType}, ${content.length} bytes from ${metadata.sourceApp}`
        );
        this.emitBinarySignal(content, {
          mimeType: finalMimeType,
          sourceApp: metadata.sourceApp
        });
      } catch (signalError) {
        logger.error("Error processing clipboard event", {
          error: signalError,
          errorType: typeof signalError,
          errorMessage: signalError instanceof Error ? signalError.message : String(signalError),
          stack: signalError instanceof Error ? signalError.stack : void 0
        });
      }
    };
    this.clipboardManager.onClipboardChange(this.clipboardListener);
    this.isListening = true;
    logger.info("\u{1F4E1} Binary-only D-Bus clipboard listener activated");
  }
  bufferLikeToUint8Array(buffer) {
    if (buffer instanceof Uint8Array) {
      return buffer;
    }
    if (buffer instanceof ArrayBuffer) {
      return new Uint8Array(buffer);
    }
    if (buffer && typeof buffer === "object" && "length" in buffer) {
      const uint8Array = new Uint8Array(buffer.length);
      for (let i = 0; i < buffer.length; i++) {
        uint8Array[i] = buffer[i];
      }
      return uint8Array;
    }
    throw new Error(`Unsupported buffer type: ${typeof buffer}`);
  }
  emitBinarySignal(content, metadata) {
    try {
      logger.debug(
        `Emitting binary signal: ${metadata.mimeType}, ${content.length} bytes`
      );
      this.dbusObject?.emit_signal(
        "ClipboardChanged",
        GLib.Variant.new("(ayss)", [
          content,
          String(metadata.mimeType),
          String(metadata.sourceApp)
        ])
      );
      logger.info(
        `Binary signal emitted: ${metadata.mimeType}, ${content.length} bytes`
      );
    } catch (error2) {
      logger.error("Failed to emit binary signal", error2);
      throw error2;
    }
  }
  // Method to start listening to clipboard changes
  ListenToClipboardChanges() {
    try {
      if (!this.isListening && this.clipboardListener) {
        logger.debug("D-Bus: Starting clipboard listener...");
        this.clipboardManager.onClipboardChange(this.clipboardListener);
        this.isListening = true;
        logger.info(
          "\u{1F4E1} Binary D-Bus clipboard listener activated via method call"
        );
      } else if (this.isListening) {
        logger.debug("D-Bus: Clipboard listener already active");
      } else {
        logger.warn("D-Bus: No clipboard listener available");
      }
    } catch (error2) {
      logger.error("D-Bus: Error starting clipboard listener", error2);
      throw error2;
    }
  }
  // Method to manually trigger a clipboard change (for testing)
  TriggerClipboardChange(content, source = "user") {
    try {
      this.clipboardManager.triggerClipboardChange(
        content,
        source
      );
    } catch (error2) {
      logger.error("D-Bus: Error triggering clipboard change", error2);
      throw error2;
    }
  }
  // Method to get current clipboard content
  GetCurrentContent() {
    try {
      return this.clipboardManager.getCurrentContent();
    } catch (error2) {
      logger.error(
        "D-Bus: Error getting current clipboard content",
        error2
      );
      throw error2;
    }
  }
  // Method to set clipboard content
  SetContent(content) {
    try {
      this.clipboardManager.setContent(content);
    } catch (error2) {
      logger.error("D-Bus: Error setting clipboard content", error2);
      throw error2;
    }
  }
  // Method to set binary clipboard content
  SetContentBinary(content, mimeType) {
    try {
      this.clipboardManager.setContentBinary(content, mimeType);
    } catch (error2) {
      logger.error(
        "D-Bus: Error setting binary clipboard content",
        error2
      );
      throw error2;
    }
  }
  // Method to get available MIME types
  GetClipboardMimeTypes() {
    try {
      return [
        "text/plain",
        "text/html",
        "image/png",
        "image/jpeg",
        "image/gif",
        "image/webp",
        "application/x-vicinae-concealed"
      ];
    } catch (error2) {
      logger.error("D-Bus: Error getting clipboard MIME types", error2);
      return [];
    }
  }
  // Method to stop listening to clipboard changes
  StopListening() {
    try {
      if (this.clipboardListener && this.isListening) {
        this.clipboardManager.removeClipboardListener(
          this.clipboardListener
        );
        this.clipboardListener = null;
        this.isListening = false;
        logger.info("\u{1F515} Binary D-Bus clipboard listener deactivated");
      }
    } catch (error2) {
      logger.error("Error stopping clipboard listener", error2);
    }
  }
  // Cleanup method
  destroy() {
    this.StopListening();
    this.dbusObject = null;
    logger.info("ClipboardService destroyed");
  }
};
export {
  ClipboardService
};
