// src/core/windows/window-manager.ts
import GLib from "gi://GLib";
import Meta from "gi://Meta";

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};
var logger = {
  debug,
  info,
  warn,
  error
};

// src/utils/window-utils.ts
var getWindowById = (winid) => {
  if (!winid || winid <= 0) return null;
  try {
    const windowActors = global.get_window_actors();
    return windowActors.find((w) => {
      try {
        return w.meta_window && w.meta_window.get_id() === winid;
      } catch {
        return false;
      }
    });
  } catch (_error) {
    return null;
  }
};
var getCurrentTime = () => {
  return global.get_current_time();
};
var isMaximized = (win) => {
  if (win.is_maximized !== void 0) {
    return win.is_maximized();
  }
  return win.get_maximized();
};

// src/core/windows/workspace-manager.ts
var WorkspaceManager = class {
  getWorkspaceCount() {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_n_workspaces();
    } catch (error2) {
      logger.error("Error getting workspace count", error2);
      return 0;
    }
  }
  getCurrentWorkspaceIndex() {
    try {
      const workspaceManager = global.workspace_manager;
      const currentWorkspace = workspaceManager.get_active_workspace();
      return currentWorkspace.index();
    } catch (error2) {
      logger.error("Error getting current workspace index", error2);
      return 0;
    }
  }
  getWorkspaceByIndex(index) {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_workspace_by_index(index);
    } catch (error2) {
      logger.error("Error getting workspace by index", error2);
      return null;
    }
  }
  switchToWorkspace(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        workspace.activate(global.get_current_time());
      }
    } catch (error2) {
      logger.error("Error switching to workspace", error2);
      throw error2;
    }
  }
  getWorkspaceInfo(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        const windows = workspace.list_windows();
        const hasFullscreen = windows.some(
          (win) => isMaximized(win) === 3
        );
        let monitor = 0;
        if (windows.length > 0) {
          monitor = windows[0].get_monitor();
        }
        return {
          index: workspace.index(),
          name: `Workspace ${workspace.index() + 1}`,
          isActive: workspace === global.workspace_manager.get_active_workspace(),
          windowCount: windows.length,
          monitor,
          hasfullscreen: hasFullscreen
        };
      }
      return null;
    } catch (error2) {
      logger.error("Error getting workspace info", error2);
      return null;
    }
  }
  getAllWorkspaces() {
    try {
      const workspaceManager = global.workspace_manager;
      const workspaces = [];
      for (let i = 0; i < workspaceManager.get_n_workspaces(); i++) {
        const workspace = workspaceManager.get_workspace_by_index(i);
        if (workspace) {
          const workspaceInfo = this.getWorkspaceInfo(i);
          if (workspaceInfo) {
            workspaces.push(workspaceInfo);
          }
        }
      }
      return workspaces;
    } catch (error2) {
      logger.error("Error getting all workspaces", error2);
      return [];
    }
  }
};

// src/core/windows/window-manager.ts
var VicinaeWindowManager = class {
  clipboardManager;
  PASTE_DELAY = 100;
  appClass;
  constructor(clipboardManager, appClass) {
    this.appClass = appClass;
    this.clipboardManager = clipboardManager;
  }
  isTargetWindow(wmClass) {
    return wmClass.toLowerCase().includes(this.appClass.toLowerCase()) || this.appClass.toLowerCase().includes(wmClass.toLowerCase());
  }
  list() {
    const windowActors = global.get_window_actors();
    const workspaceManager = global.workspace_manager;
    const windows = windowActors.map((w) => {
      const metaWindow = w.meta_window;
      const windowWorkspace = metaWindow.get_workspace();
      const frame = metaWindow.get_frame_rect();
      return {
        id: metaWindow.get_id(),
        title: metaWindow.get_title(),
        wm_class: metaWindow.get_wm_class(),
        wm_class_instance: metaWindow.get_wm_class_instance(),
        pid: metaWindow.get_pid(),
        maximized: isMaximized(metaWindow) !== 0,
        // 0 means not maximized
        display: metaWindow.get_display(),
        frame_type: metaWindow.get_frame_type(),
        window_type: metaWindow.get_window_type(),
        layer: metaWindow.get_layer(),
        monitor: metaWindow.get_monitor(),
        role: metaWindow.get_role(),
        width: frame.width,
        height: frame.height,
        x: frame.x,
        y: frame.y,
        in_current_workspace: metaWindow.located_on_workspace?.(
          workspaceManager.get_active_workspace?.()
        ),
        canclose: metaWindow.can_close(),
        canmaximize: metaWindow.can_maximize(),
        canminimize: metaWindow.can_minimize(),
        canshade: false,
        // can_shade() is not in the type definitions
        moveable: metaWindow.allows_move(),
        resizeable: metaWindow.allows_resize(),
        has_focus: metaWindow.has_focus(),
        workspace: windowWorkspace ? windowWorkspace.index() : -1
      };
    });
    return windows;
  }
  details(winid) {
    const w = getWindowById(winid);
    if (!w) {
      throw new Error("Window not found");
    }
    const metaWindow = w.meta_window;
    const workspaceManager = global.workspace_manager;
    const windowWorkspace = metaWindow.get_workspace();
    const frame = metaWindow.get_frame_rect();
    const win = {
      id: metaWindow.get_id(),
      title: metaWindow.get_title(),
      wm_class: metaWindow.get_wm_class(),
      wm_class_instance: metaWindow.get_wm_class_instance(),
      pid: metaWindow.get_pid(),
      maximized: isMaximized(metaWindow) !== 0,
      // 0 means not maximized
      display: metaWindow.get_display(),
      frame_type: metaWindow.get_frame_type(),
      window_type: metaWindow.get_window_type(),
      layer: metaWindow.get_layer(),
      monitor: metaWindow.get_monitor(),
      role: metaWindow.get_role(),
      width: frame.width,
      height: frame.height,
      x: frame.x,
      y: frame.y,
      in_current_workspace: metaWindow.located_on_workspace?.(
        workspaceManager.get_active_workspace?.()
      ),
      canclose: metaWindow.can_close(),
      canmaximize: metaWindow.can_maximize(),
      canminimize: metaWindow.can_minimize(),
      canshade: false,
      // can_shade() is not in the type definitions
      moveable: metaWindow.allows_move(),
      resizeable: metaWindow.allows_resize(),
      has_focus: metaWindow.has_focus(),
      workspace: windowWorkspace ? windowWorkspace.index() : -1
    };
    return win;
  }
  getTitle(winid) {
    const w = getWindowById(winid);
    if (w) {
      return w.meta_window.get_title();
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameRect(winid) {
    const w = getWindowById(winid);
    if (w) {
      const frame = w.meta_window.get_frame_rect();
      return {
        x: frame.x,
        y: frame.y,
        width: frame.width,
        height: frame.height
      };
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameBounds(winid) {
    const w = getWindowById(winid);
    if (w) {
      return {
        frame_bounds: w.meta_window.get_frame_bounds()
      };
    } else {
      throw new Error("Window not found");
    }
  }
  moveToWorkspace(winid, workspaceNum) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.change_workspace_by_index(workspaceNum, false);
    } else {
      throw new Error("Window not found");
    }
  }
  moveResize(winid, x, y, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(true, x, y, width, height);
    } else {
      throw new Error("Window not found");
    }
  }
  resize(winid, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(
        true,
        win.get_x(),
        win.get_y(),
        width,
        height
      );
    } else {
      throw new Error("Window not found");
    }
  }
  move(winid, x, y) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_frame(true, x, y);
    } else {
      throw new Error("Window not found");
    }
  }
  maximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.maximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  minimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.minimize();
    } else {
      throw new Error("Window not found");
    }
  }
  unmaximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unmaximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  unminimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unminimize();
    } else {
      throw new Error("Window not found");
    }
  }
  activate(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      const currentTime = getCurrentTime();
      const workspace = win.get_workspace();
      if (workspace) {
        workspace.activate_with_focus(win, currentTime);
      } else {
        win.activate(currentTime);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  close(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      try {
        if (win.get_id() === winid) {
          win.delete(getCurrentTime());
        } else {
          throw new Error(
            "Window ID mismatch - window may be destroyed"
          );
        }
      } catch (error2) {
        throw new Error(`Failed to close window ${winid}: ${error2}`);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  listWorkspaces() {
    const workspaceManager = new WorkspaceManager();
    return workspaceManager.getAllWorkspaces();
  }
  getActiveWorkspace() {
    const workspaceManager = new WorkspaceManager();
    const currentIndex = workspaceManager.getCurrentWorkspaceIndex();
    const workspace = workspaceManager.getWorkspaceInfo(currentIndex);
    if (!workspace) {
      throw new Error("No active workspace found");
    }
    return workspace;
  }
  pasteToFocusedWindow(winid, key, modifiers) {
    if (!this.clipboardManager) {
      logger.error("No clipboard manager available for paste operation");
      return false;
    }
    GLib.timeout_add(GLib.PRIORITY_DEFAULT, this.PASTE_DELAY, () => {
      logger.debug(
        `Sending keyboard paste for window ${winid} ${key} ${modifiers} after ${this.PASTE_DELAY}ms delay`
      );
      this.clipboardManager.triggerKeyboardPaste();
      return false;
    });
    logger.debug(
      `Triggered keyboard paste for window ${winid} ${key} ${modifiers}`
    );
    return true;
  }
  sendShortcut(winid, key, modifiers) {
    this.activate(winid);
    const modifiersArray = modifiers.split("|");
    const isNormalPaste = key === "v" && modifiersArray.length === 1 && modifiersArray[0] === "CONTROL";
    const isShiftPaste = key === "v" && modifiersArray.length === 2 && modifiersArray[0] === "CONTROL" && modifiersArray[1] === "SHIFT";
    if (isNormalPaste || isShiftPaste) {
      return this.pasteToFocusedWindow(winid, key, modifiers);
    } else {
      logger.warn(`Unhandled shortcut: ${key} ${modifiers}`);
      return false;
    }
  }
};
export {
  VicinaeWindowManager
};
