// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};
var logger = {
  debug,
  info,
  warn,
  error
};

// src/utils/window-utils.ts
var isMaximized = (win) => {
  if (win.is_maximized !== void 0) {
    return win.is_maximized();
  }
  return win.get_maximized();
};

// src/core/windows/workspace-manager.ts
var WorkspaceManager = class {
  getWorkspaceCount() {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_n_workspaces();
    } catch (error2) {
      logger.error("Error getting workspace count", error2);
      return 0;
    }
  }
  getCurrentWorkspaceIndex() {
    try {
      const workspaceManager = global.workspace_manager;
      const currentWorkspace = workspaceManager.get_active_workspace();
      return currentWorkspace.index();
    } catch (error2) {
      logger.error("Error getting current workspace index", error2);
      return 0;
    }
  }
  getWorkspaceByIndex(index) {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_workspace_by_index(index);
    } catch (error2) {
      logger.error("Error getting workspace by index", error2);
      return null;
    }
  }
  switchToWorkspace(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        workspace.activate(global.get_current_time());
      }
    } catch (error2) {
      logger.error("Error switching to workspace", error2);
      throw error2;
    }
  }
  getWorkspaceInfo(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        const windows = workspace.list_windows();
        const hasFullscreen = windows.some(
          (win) => isMaximized(win) === 3
        );
        let monitor = 0;
        if (windows.length > 0) {
          monitor = windows[0].get_monitor();
        }
        return {
          index: workspace.index(),
          name: `Workspace ${workspace.index() + 1}`,
          isActive: workspace === global.workspace_manager.get_active_workspace(),
          windowCount: windows.length,
          monitor,
          hasfullscreen: hasFullscreen
        };
      }
      return null;
    } catch (error2) {
      logger.error("Error getting workspace info", error2);
      return null;
    }
  }
  getAllWorkspaces() {
    try {
      const workspaceManager = global.workspace_manager;
      const workspaces = [];
      for (let i = 0; i < workspaceManager.get_n_workspaces(); i++) {
        const workspace = workspaceManager.get_workspace_by_index(i);
        if (workspace) {
          const workspaceInfo = this.getWorkspaceInfo(i);
          if (workspaceInfo) {
            workspaces.push(workspaceInfo);
          }
        }
      }
      return workspaces;
    } catch (error2) {
      logger.error("Error getting all workspaces", error2);
      return [];
    }
  }
};
export {
  WorkspaceManager
};
