// src/extension.ts
import Clutter3 from "gi://Clutter";
import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";
import * as Main from "resource:///org/gnome/shell/ui/main.js";

// src/components/indicator.ts
import St from "gi://St";
import * as PanelMenu from "resource:///org/gnome/shell/ui/panelMenu.js";
import * as PopupMenu from "resource:///org/gnome/shell/ui/popupMenu.js";

// src/lib/icons.ts
import Gio from "gi://Gio";
var ICONS = ["vicinae", "github"];
var Icons = class _Icons {
  static #icons = /* @__PURE__ */ new Map();
  constructor(extPath) {
    for (const name of ICONS) {
      const iconPath = `${extPath}/assets/icons/${name}.svg`;
      const icon = Gio.icon_new_for_string(iconPath);
      _Icons.#icons.set(name, icon);
    }
  }
  static get(name) {
    return _Icons.#icons.get(name);
  }
};

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var stringToLogLevel = (level) => {
  switch (level.toLowerCase()) {
    case "error":
      return 0 /* ERROR */;
    case "warn":
      return 1 /* WARN */;
    case "info":
      return 2 /* INFO */;
    case "debug":
      return 3 /* DEBUG */;
    default:
      return 2 /* INFO */;
  }
};
var currentLogLevel = 2 /* INFO */;
var initializeLogger = (settings) => {
  const levelString = settings.get_string("logging-level");
  currentLogLevel = stringToLogLevel(levelString);
  settings.connect("changed::logging-level", () => {
    const newLevelString = settings.get_string("logging-level");
    currentLogLevel = stringToLogLevel(newLevelString);
    log(2 /* INFO */, `Log level changed to: ${newLevelString}`);
  });
  log(2 /* INFO */, `Logger initialized with level: ${levelString}`);
};
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};
var logger = {
  debug,
  info,
  warn,
  error
};

// src/components/indicator.ts
var VicinaeIndicator = class {
  indicator;
  extension;
  constructor(extension) {
    this.extension = extension;
    this.indicator = new PanelMenu.Button(0, "Vicinae Gnome Extension");
    this.setupUI();
    this.setupMenu();
  }
  setupUI() {
    new Icons(this.extension.path);
    const vicinaeIcon = Icons.get("vicinae");
    const icon = new St.Icon({
      gicon: vicinaeIcon,
      style_class: "system-status-icon"
    });
    this.indicator.add_child(icon);
  }
  setupMenu() {
    const settingsItem = new PopupMenu.PopupMenuItem("Settings");
    settingsItem.connect("activate", () => {
      logger.debug("Opening Vicinae settings");
      this.extension.openPreferences();
    });
    if (this.indicator.menu && "addMenuItem" in this.indicator.menu) {
      this.indicator.menu.addMenuItem(settingsItem);
    }
  }
  getButton() {
    return this.indicator;
  }
  destroy() {
    if (this.indicator) {
      this.indicator.destroy();
    }
  }
};

// src/core/clipboard/clipboard-manager.ts
import Clutter from "gi://Clutter";
import GLib from "gi://GLib";
import Meta from "gi://Meta";
import Shell from "gi://Shell";
import St2 from "gi://St";

// src/utils/window-utils.ts
var getWindowById = (winid) => {
  if (!winid || winid <= 0) return null;
  try {
    const windowActors = global.get_window_actors();
    return windowActors.find((w) => {
      try {
        return w.meta_window && w.meta_window.get_id() === winid;
      } catch {
        return false;
      }
    });
  } catch (_error) {
    return null;
  }
};
var getCurrentTime = () => {
  return global.get_current_time();
};
var getFocusedWindow = () => {
  const windowActors = global.get_window_actors();
  return windowActors.find((w) => w.meta_window.has_focus());
};
var getFocusedWindowApp = () => {
  const focusedWindow = getFocusedWindow();
  if (focusedWindow) {
    const wmClass = focusedWindow.meta_window.get_wm_class();
    const title = focusedWindow.meta_window.get_title();
    return wmClass || title || "unknown";
  }
  return "gnome-shell";
};
var isMaximized = (win) => {
  if (win.is_maximized !== void 0) {
    return win.is_maximized();
  }
  return win.get_maximized();
};

// src/utils/clipboard-utils.ts
var CLIPBOARD_CONFIG = {
  MAX_CLIPBOARD_SIZE: 10 * 1024 * 1024
  // 10MB - reasonable clipboard limit
};
function isValidImageBuffer(buffer) {
  if (!buffer) return false;
  let uint8Array;
  if (buffer instanceof ArrayBuffer) {
    if (buffer.byteLength < 8) return false;
    uint8Array = new Uint8Array(buffer);
  } else if (buffer instanceof Uint8Array) {
    if (buffer.length < 8) return false;
    uint8Array = buffer;
  } else if ("length" in buffer && buffer.length >= 8) {
    uint8Array = new Uint8Array(buffer.slice(0, 8));
  } else {
    return false;
  }
  if (uint8Array[0] === 137 && uint8Array[1] === 80 && uint8Array[2] === 78 && uint8Array[3] === 71) {
    return true;
  }
  if (uint8Array[0] === 255 && uint8Array[1] === 216 && uint8Array[2] === 255) {
    return true;
  }
  if (uint8Array[0] === 71 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 56) {
    return true;
  }
  if (uint8Array[0] === 82 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 70) {
    let webpCheck;
    if (buffer instanceof Uint8Array && buffer.length >= 12) {
      webpCheck = buffer.slice(8, 12);
    } else if (buffer instanceof ArrayBuffer && buffer.byteLength >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else if ("length" in buffer && buffer.length >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else {
      return false;
    }
    if (webpCheck[0] === 87 && webpCheck[1] === 69 && webpCheck[2] === 66 && webpCheck[3] === 80) {
      return true;
    }
  }
  return false;
}
function getImageMimeType(buffer) {
  if (!buffer) return "application/octet-stream";
  let uint8Array;
  if (buffer instanceof ArrayBuffer) {
    if (buffer.byteLength < 8) return "application/octet-stream";
    uint8Array = new Uint8Array(buffer.slice(0, 12));
  } else if (buffer instanceof Uint8Array) {
    if (buffer.length < 8) return "application/octet-stream";
    uint8Array = buffer.slice(0, 12);
  } else if ("length" in buffer && buffer.length >= 8) {
    uint8Array = new Uint8Array(buffer.slice(0, 12));
  } else {
    return "application/octet-stream";
  }
  if (uint8Array[0] === 137 && uint8Array[1] === 80 && uint8Array[2] === 78 && uint8Array[3] === 71) {
    return "image/png";
  }
  if (uint8Array[0] === 255 && uint8Array[1] === 216 && uint8Array[2] === 255) {
    return "image/jpeg";
  }
  if (uint8Array[0] === 71 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 56) {
    return "image/gif";
  }
  if (uint8Array[0] === 82 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 70) {
    let webpCheck;
    if (buffer instanceof Uint8Array && buffer.length >= 12) {
      webpCheck = buffer.slice(8, 12);
    } else if (buffer instanceof ArrayBuffer && buffer.byteLength >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else if ("length" in buffer && buffer.length >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else {
      return "application/octet-stream";
    }
    if (webpCheck[0] === 87 && webpCheck[1] === 69 && webpCheck[2] === 66 && webpCheck[3] === 80) {
      return "image/webp";
    }
  }
  return "application/octet-stream";
}
function calculateClipboardMetadata(event) {
  const content = event.content;
  let mimeType = "text/plain";
  const sourceApp = getFocusedWindowApp();
  if (event.source === "image") {
    if (content.startsWith("data:image/")) {
      const match = content.match(/^data:(image\/[^;]+);/);
      mimeType = match ? match[1] : "image/png";
    }
  } else if (content.startsWith("data:")) {
    const match = content.match(/^data:([^;]+);/);
    mimeType = match ? match[1] : "application/octet-stream";
  } else {
    mimeType = "text/plain";
    if (content.includes("<") && content.includes(">") && (content.includes("<html") || content.includes("<div") || content.includes("<p"))) {
      mimeType = "text/html";
    }
  }
  return {
    mimeType,
    sourceApp
  };
}

// src/core/clipboard/clipboard-manager.ts
var VicinaeClipboardManager = class {
  eventListeners = [];
  virtualKeyboard;
  currentContent = "";
  clipboard = null;
  selection = null;
  _selectionOwnerChangedId = null;
  _debouncing = 0;
  settings = null;
  pasteHackCallbackId = null;
  constructor(virtualKeyboard) {
    this.virtualKeyboard = virtualKeyboard;
    this.setupClipboardMonitoring();
  }
  setSettings(settings) {
    this.settings = settings;
    logger.info("Settings set in clipboard manager from external source");
    try {
      const blockedApps = this.settings.get_strv("blocked-applications");
      logger.debug(
        `Current blocked applications: [${blockedApps.join(", ")}]`
      );
    } catch (err) {
      logger.error(
        "Error reading blocked applications from settings",
        err
      );
    }
  }
  updateSettings(settings) {
    this.settings = settings;
    logger.info("Settings updated in clipboard manager");
    try {
      const blockedApps = this.settings.get_strv("blocked-applications");
      logger.debug(
        `Updated blocked applications: [${blockedApps.join(", ")}]`
      );
    } catch (err) {
      logger.error(
        "Error reading updated blocked applications from settings",
        err
      );
    }
  }
  isApplicationBlocked(sourceApp) {
    if (!this.settings) {
      logger.warn(
        "No settings available in clipboard manager - blocking logic disabled"
      );
      return false;
    }
    try {
      const blockedApps = this.settings.get_strv("blocked-applications");
      logger.debug(
        `Checking if ${sourceApp} is blocked. Blocked apps list: [${blockedApps.join(
          ", "
        )}]`
      );
      const isBlocked = blockedApps.some(
        (blockedApp) => sourceApp.toLowerCase().includes(blockedApp.toLowerCase()) || blockedApp.toLowerCase().includes(sourceApp.toLowerCase())
      );
      if (isBlocked) {
        logger.debug(
          `Application ${sourceApp} is blocked from clipboard access`
        );
      } else {
        logger.debug(
          `Application ${sourceApp} is NOT blocked (not in blocked apps list)`
        );
      }
      return isBlocked;
    } catch (error2) {
      logger.error(
        "Error checking blocked applications in clipboard manager",
        error2
      );
      return false;
    }
  }
  shouldBlockContentType(contentType, mimeType) {
    return contentType === "text" || mimeType.startsWith("text/");
  }
  setupClipboardMonitoring() {
    try {
      this.clipboard = St2.Clipboard.get_default();
      this.selection = Shell.Global.get().get_display().get_selection();
      if (this.selection) {
        this._selectionOwnerChangedId = this.selection.connect(
          "owner-changed",
          this.onSelectionOwnerChanged.bind(this)
        );
        this.queryClipboard();
        logger.info(
          "Clipboard monitoring set up successfully using selection listener"
        );
      } else {
        logger.error("Failed to get selection instance");
      }
    } catch (error2) {
      logger.error("Error setting up clipboard monitoring", error2);
    }
  }
  onSelectionOwnerChanged(_, selectionType) {
    if (selectionType === Meta.SelectionType.SELECTION_CLIPBOARD) {
      this.queryClipboard();
    }
  }
  queryClipboard() {
    if (!this.clipboard) return;
    try {
      this.clipboard.get_text(St2.ClipboardType.CLIPBOARD, (_, text) => {
        if (text) {
          this.processClipboardContent(text, "system");
        }
      });
      this.clipboard.get_text(St2.ClipboardType.PRIMARY, (_, text) => {
        if (text) {
          this.processClipboardContent(text, "system");
        }
      });
      const mimeTypes = this.clipboard.get_mimetypes(
        St2.ClipboardType.CLIPBOARD
      );
      if (mimeTypes.length > 0) {
        if (mimeTypes.some((type) => type.startsWith("image/"))) {
          this.captureImageData();
        }
      }
    } catch (error2) {
      logger.error("Error querying clipboard", error2);
    }
  }
  captureImageData() {
    if (!this.clipboard) return;
    try {
      try {
        this.clipboard.get_content(
          St2.ClipboardType.CLIPBOARD,
          "image/png",
          (_, content) => {
            if (content) {
              if (content && typeof content === "object") {
                if (content.constructor && (content.constructor.name === "GLib.Bytes" || content.constructor.name.includes(
                  "GLib.Bytes"
                ) || content.constructor.name.includes(
                  "Bytes"
                ))) {
                  this.extractGLibBytesData(
                    content
                  );
                } else {
                  if (content && typeof content === "object" && "data" in content) {
                    this.processImageContent(
                      content
                    );
                  } else {
                    this.processClipboardContent(
                      "[IMAGE_DATA_AVAILABLE]",
                      "system"
                    );
                  }
                }
              }
            }
          }
        );
      } catch (_contentError) {
        this.processClipboardContent(
          "[IMAGE_DATA_AVAILABLE]",
          "system"
        );
      }
    } catch (error2) {
      logger.error("Error capturing image data", error2);
    }
  }
  processImageContent(content) {
    try {
      if (content.data && content.data.length > 0) {
        const isValid = isValidImageBuffer(content.data);
        const detectedMimeType = getImageMimeType(content.data);
        const finalMimeType = content.mimeType || detectedMimeType;
        if (isValid) {
          logger.debug(
            `Processed image content: ${finalMimeType}, ${content.data.length} bytes`
          );
          const binaryMarker = `[BINARY_IMAGE:${finalMimeType}:${content.data.length}]`;
          this.storeBinaryData(
            binaryMarker,
            content.data,
            finalMimeType
          );
          this.processClipboardContent(binaryMarker, "image");
        } else {
          logger.warn(
            `Invalid image buffer detected: ${content.data.length} bytes`
          );
          this.processClipboardContent(
            "[BINARY_DATA_AVAILABLE]",
            "system"
          );
        }
      } else {
        logger.debug("No image data available in content object");
        this.processClipboardContent(
          "[IMAGE_DATA_AVAILABLE]",
          "system"
        );
      }
    } catch (error2) {
      logger.error("Error processing image content", error2);
      this.processClipboardContent("[IMAGE_DATA_AVAILABLE]", "system");
    }
  }
  binaryDataStore = /* @__PURE__ */ new Map();
  storeBinaryData(marker, data, mimeType) {
    this.binaryDataStore.set(marker, { data, mimeType });
    logger.debug(`Stored binary data: ${marker}`);
  }
  getBinaryData(marker) {
    return this.binaryDataStore.get(marker) || null;
  }
  extractGLibBytesData(bytes) {
    try {
      if (typeof bytes.get_data === "function") {
        try {
          const data = bytes.get_data();
          if (data && data.length > 0) {
            const isValid = isValidImageBuffer(data);
            const mimeType = getImageMimeType(data);
            if (isValid) {
              const binaryMarker = `[BINARY_IMAGE:${mimeType}:${data.length}]`;
              this.storeBinaryData(binaryMarker, data, mimeType);
              logger.debug(
                `Extracted GLib.Bytes data: ${mimeType}, ${data.length} bytes`
              );
              this.processClipboardContent(binaryMarker, "image");
            } else {
              logger.warn(
                `Invalid image buffer from get_data: ${data.length} bytes`
              );
              this.processClipboardContent(
                "[BINARY_DATA_AVAILABLE]",
                "system"
              );
            }
          }
        } catch (getDataError) {
          logger.error("get_data failed", getDataError);
        }
      }
      if (typeof bytes.toArray === "function") {
        try {
          const array = bytes.toArray();
          if (array && array.length > 0) {
            const isValid = isValidImageBuffer(array);
            const mimeType = getImageMimeType(array);
            if (isValid) {
              const binaryMarker = `[BINARY_IMAGE:${mimeType}:${array.length}]`;
              this.storeBinaryData(binaryMarker, array, mimeType);
              logger.debug(
                `Extracted GLib.Bytes array data: ${mimeType}, ${array.length} bytes`
              );
              this.processClipboardContent(binaryMarker, "image");
            }
          }
        } catch (toArrayError) {
          logger.error("to_array failed", toArrayError);
        }
      }
    } catch (error2) {
      logger.error("Error extracting GLib.Bytes data", error2);
    }
  }
  processClipboardContent(text, source) {
    if (this._debouncing > 0) {
      this._debouncing--;
      return;
    }
    if (!text || text === this.currentContent) {
      return;
    }
    this.currentContent = text;
    this.emitClipboardEvent(text, source);
  }
  // Method to emit clipboard change events
  emitClipboardEvent(content, source = "user", mimeType) {
    let contentString;
    let contentType = "text";
    if (content instanceof Uint8Array) {
      if (mimeType?.startsWith("image/")) {
        contentType = "image";
        contentString = `[BINARY_IMAGE:${mimeType}:${content.length}]`;
      } else {
        contentString = `[BINARY_DATA:${mimeType}:${content.length}]`;
      }
    } else {
      contentString = content;
      contentType = source === "image" ? "image" : "text";
    }
    const event = {
      type: "clipboard-changed",
      content: contentString,
      timestamp: Date.now(),
      source,
      contentType
    };
    const metadata = calculateClipboardMetadata(event);
    const isBlocked = this.isApplicationBlocked(metadata.sourceApp);
    const shouldBlock = isBlocked && this.shouldBlockContentType(event.contentType, metadata.mimeType);
    logger.debug("\u{1F3AF} CLIPBOARD EVENT EMITTED", {
      type: event.type,
      content: contentString.length > 100 ? `${contentString.substring(0, 100)}...` : contentString,
      contentLength: content instanceof Uint8Array ? content.length : contentString.length,
      originalContentType: content instanceof Uint8Array ? "binary" : "string",
      timestamp: new Date(event.timestamp).toISOString(),
      source: event.source,
      listeners: this.eventListeners.length,
      mimeType: metadata.mimeType,
      contentType: event.contentType,
      sourceApp: metadata.sourceApp,
      isBlocked,
      shouldBlock,
      note: shouldBlock ? "\u26A0\uFE0F This event will be blocked by clipboard manager" : "\u2705 Event will be processed normally"
    });
    if (shouldBlock) {
      logger.debug(
        `\u{1F6AB} Clipboard access blocked for application: ${metadata.sourceApp} (${event.contentType}) - Event not forwarded to listeners`
      );
      return;
    }
    this.eventListeners.forEach((listener) => {
      try {
        listener(event);
      } catch (error2) {
        logger.error("\u274C Error in clipboard event listener", error2);
      }
    });
  }
  onClipboardChange(listener) {
    this.eventListeners.push(listener);
    logger.debug("\u{1F442} Clipboard change listener added");
  }
  removeClipboardListener(listener) {
    const index = this.eventListeners.indexOf(listener);
    if (index > -1) {
      this.eventListeners.splice(index, 1);
      logger.debug("Clipboard change listener removed");
    }
  }
  getCurrentContent() {
    return this.currentContent;
  }
  setContent(content) {
    if (this.clipboard) {
      try {
        this.clipboard.set_text(St2.ClipboardType.CLIPBOARD, content);
        this.clipboard.set_text(St2.ClipboardType.PRIMARY, content);
        this.currentContent = content;
        this.emitClipboardEvent(content, "user");
      } catch (error2) {
        logger.error("Error setting clipboard content", error2);
      }
    }
  }
  setContentBinary(data, mimeType) {
    if (this.clipboard) {
      try {
        logger.debug(
          `Setting binary clipboard content: ${mimeType}, ${data.length} bytes`
        );
        this.clipboard.set_content(
          St2.ClipboardType.CLIPBOARD,
          mimeType,
          data
        );
        this.clipboard.set_content(
          St2.ClipboardType.PRIMARY,
          mimeType,
          data
        );
        if (mimeType.startsWith("text/")) {
          const text = new TextDecoder().decode(data);
          this.clipboard.set_content(
            St2.ClipboardType.PRIMARY,
            "text/plain",
            new TextEncoder().encode(text)
          );
        }
        this.emitClipboardEvent(data, "user", mimeType);
        logger.debug(
          `Binary clipboard content set successfully for ${mimeType}`
        );
      } catch (error2) {
        logger.error("Error setting binary clipboard content", error2);
      }
    } else {
      logger.error("Clipboard not available for setContentBinary");
    }
  }
  triggerClipboardChange(content, source = "user") {
    this.emitClipboardEvent(content, source);
  }
  triggerKeyboardPaste() {
    logger.debug("Trigger keyboard paste called");
    this.pasteHackCallbackId = GLib.timeout_add(
      GLib.PRIORITY_DEFAULT,
      1,
      // Just post to the end of the event loop
      () => {
        const SHIFT_L = 42;
        const INSERT = 110;
        const eventTime = Clutter.get_current_event_time() * 1e3;
        this.virtualKeyboard.notify_key(
          eventTime,
          SHIFT_L,
          Clutter.KeyState.PRESSED
        );
        this.virtualKeyboard.notify_key(
          eventTime,
          INSERT,
          Clutter.KeyState.PRESSED
        );
        this.virtualKeyboard.notify_key(
          eventTime,
          INSERT,
          Clutter.KeyState.RELEASED
        );
        this.virtualKeyboard.notify_key(
          eventTime,
          SHIFT_L,
          Clutter.KeyState.RELEASED
        );
        this.pasteHackCallbackId = null;
        return false;
      }
    );
  }
  destroy() {
    if (this.selection && this._selectionOwnerChangedId) {
      try {
        this.selection.disconnect(this._selectionOwnerChangedId);
        this._selectionOwnerChangedId = null;
      } catch (edit_error) {
        logger.error(
          "Error disconnecting selection listener",
          edit_error
        );
      }
    }
    this.eventListeners = [];
    this.currentContent = "";
    this.clipboard = null;
    this.selection = null;
    logger.info("Clipboard manager destroyed");
  }
};

// src/utils/dbus-utils.ts
import Gio2 from "gi://Gio";
var createDBusService = (interfaceDefinition, serviceObject) => {
  try {
    const dbus = Gio2.DBusExportedObject.wrapJSObject(
      interfaceDefinition,
      serviceObject
    );
    return dbus;
  } catch (error2) {
    throw new Error(`Failed to create D-Bus service: ${error2}`);
  }
};
var exportDBusService = (dbus, path) => {
  try {
    dbus.export(Gio2.DBus.session, path);
    return true;
  } catch (error2) {
    throw new Error(`Failed to export D-Bus service at ${path}: ${error2}`);
  }
};
var unexportDBusService = (dbus) => {
  try {
    dbus.flush();
    dbus.unexport();
    return true;
  } catch (error2) {
    throw new Error(`Failed to unexport D-Bus service: ${error2}`);
  }
};

// src/core/dbus/interfaces/clipboard.ts
var CLIPBOARD_DBUS_IFACE = `
<node>
   <interface name="org.gnome.Shell.Extensions.Clipboard">
      <method name="ListenToClipboardChanges">
      </method>
      <method name="GetClipboardMimeTypes">
         <arg type="as" direction="out" name="mimeTypes" />
      </method>
      <method name="StopListening">
      </method>
      <method name="TriggerClipboardChange">
         <arg type="s" direction="in" name="content" />
         <arg type="s" direction="in" name="source" />
      </method>
      <method name="GetCurrentContent">
         <arg type="s" direction="out" name="content" />
      </method>
      <method name="SetContent">
         <arg type="s" direction="in" name="content" />
      </method>
      <method name="SetContentBinary">
         <arg type="ay" direction="in" name="content" />
         <arg type="s" direction="in" name="mimeType" />
      </method>
      <signal name="ClipboardChanged">
         <arg type="ay" name="content" />
         <arg type="s" name="mimeType" />
         <arg type="s" name="sourceApp" />
      </signal>
   </interface>
</node>`;

// src/core/dbus/interfaces/windows.ts
var WINDOWS_DBUS_IFACE = `
<node>
   <interface name="org.gnome.Shell.Extensions.Windows">
      <method name="List">
         <arg type="s" direction="out" name="win" />
      </method>
      <method name="Details">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="out" name="win" />
      </method>
      <method name="GetTitle">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="out" name="win" />
      </method>
      <method name="GetFrameRect">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="out" name="frameRect" />
      </method>
      <method name="GetFrameBounds">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="out" name="frameBounds" />
      </method>
      <method name="MoveToWorkspace">
         <arg type="u" direction="in" name="winid" />
         <arg type="u" direction="in" name="workspaceNum" />
      </method>
      <method name="MoveResize">
         <arg type="u" direction="in" name="winid" />
         <arg type="i" direction="in" name="x" />
         <arg type="i" direction="in" name="y" />
         <arg type="u" direction="in" name="width" />
         <arg type="u" direction="in" name="height" />
      </method>
      <method name="Resize">
         <arg type="u" direction="in" name="winid" />
         <arg type="u" direction="in" name="width" />
         <arg type="u" direction="in" name="height" />
      </method>
      <method name="Move">
         <arg type="u" direction="in" name="winid" />
         <arg type="i" direction="in" name="x" />
         <arg type="i" direction="in" name="y" />
      </method>
      <method name="Maximize">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Minimize">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Unmaximize">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Unminimize">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Activate">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="Close">
         <arg type="u" direction="in" name="winid" />
      </method>
      <method name="ListWorkspaces">
         <arg type="s" direction="out" name="workspaces" />
      </method>
      <method name="GetActiveWorkspace">
         <arg type="s" direction="out" name="workspace" />
      </method>
      <method name="GetWorkspaceWindows">
         <arg type="u" direction="in" name="workspaceIndex" />
         <arg type="s" direction="out" name="windows" />
      </method>
      <method name="SendShortcut">
         <arg type="u" direction="in" name="winid" />
         <arg type="s" direction="in" name="key" />
         <arg type="s" direction="in" name="modifiers" />
         <arg type="b" direction="out" name="success" />
      </method>
      <method name="GetFocusedWindowSync">
         <arg type="s" direction="out" name="window" />
      </method>
      <signal name="openwindow">
         <arg type="s" name="windowAddress" />
         <arg type="s" name="workspaceName" />
         <arg type="s" name="wmClass" />
         <arg type="s" name="title" />
      </signal>
      <signal name="closewindow">
         <arg type="s" name="windowAddress" />
      </signal>
      <signal name="focuswindow">
         <arg type="s" name="windowAddress" />
      </signal>
      <signal name="movewindow">
         <arg type="s" name="windowAddress" />
         <arg type="i" name="x" />
         <arg type="i" name="y" />
         <arg type="u" name="width" />
         <arg type="u" name="height" />
      </signal>
      <signal name="statewindow">
         <arg type="s" name="windowAddress" />
         <arg type="s" name="state" />
      </signal>
      <signal name="workspacechanged">
         <arg type="s" name="workspaceId" />
      </signal>
      <signal name="monitorlayoutchanged">
      </signal>
   </interface>
</node>`;

// src/core/dbus/services/clipboard-service.ts
import GLib2 from "gi://GLib";
var ClipboardService = class {
  clipboardManager;
  dbusObject = null;
  clipboardListener = null;
  isListening = false;
  constructor(clipboardManager, _extension) {
    this.clipboardManager = clipboardManager;
    logger.info("ClipboardService initialized with binary-only protocol");
  }
  // Method to set the D-Bus exported object (called by DBusManager)
  setDBusObject(dbusObject) {
    this.dbusObject = dbusObject;
    this.clipboardListener = (event) => {
      try {
        if (!event.content || event.content.length === 0) {
          logger.debug("Skipping empty clipboard content");
          return;
        }
        const metadata = calculateClipboardMetadata(event);
        let content;
        let finalMimeType = metadata.mimeType;
        if (event.content.startsWith("[BINARY_IMAGE:")) {
          const binaryInfo = this.clipboardManager.getBinaryData(
            event.content
          );
          if (binaryInfo) {
            content = this.bufferLikeToUint8Array(binaryInfo.data);
            finalMimeType = binaryInfo.mimeType;
            logger.debug(
              `Using direct binary data: ${finalMimeType}, ${content.length} bytes`
            );
          } else {
            logger.warn(
              `Binary data not found for marker: ${event.content}`
            );
            return;
          }
        } else {
          content = new TextEncoder().encode(event.content);
        }
        if (content.length > CLIPBOARD_CONFIG.MAX_CLIPBOARD_SIZE) {
          logger.warn(
            `Clipboard data too large: ${content.length} bytes, skipping`
          );
          return;
        }
        logger.debug(
          `Processing clipboard: ${finalMimeType}, ${content.length} bytes from ${metadata.sourceApp}`
        );
        this.emitBinarySignal(content, {
          mimeType: finalMimeType,
          sourceApp: metadata.sourceApp
        });
      } catch (signalError) {
        logger.error("Error processing clipboard event", {
          error: signalError,
          errorType: typeof signalError,
          errorMessage: signalError instanceof Error ? signalError.message : String(signalError),
          stack: signalError instanceof Error ? signalError.stack : void 0
        });
      }
    };
    this.clipboardManager.onClipboardChange(this.clipboardListener);
    this.isListening = true;
    logger.info("\u{1F4E1} Binary-only D-Bus clipboard listener activated");
  }
  bufferLikeToUint8Array(buffer) {
    if (buffer instanceof Uint8Array) {
      return buffer;
    }
    if (buffer instanceof ArrayBuffer) {
      return new Uint8Array(buffer);
    }
    if (buffer && typeof buffer === "object" && "length" in buffer) {
      const uint8Array = new Uint8Array(buffer.length);
      for (let i = 0; i < buffer.length; i++) {
        uint8Array[i] = buffer[i];
      }
      return uint8Array;
    }
    throw new Error(`Unsupported buffer type: ${typeof buffer}`);
  }
  emitBinarySignal(content, metadata) {
    try {
      logger.debug(
        `Emitting binary signal: ${metadata.mimeType}, ${content.length} bytes`
      );
      this.dbusObject?.emit_signal(
        "ClipboardChanged",
        GLib2.Variant.new("(ayss)", [
          content,
          String(metadata.mimeType),
          String(metadata.sourceApp)
        ])
      );
      logger.info(
        `Binary signal emitted: ${metadata.mimeType}, ${content.length} bytes`
      );
    } catch (error2) {
      logger.error("Failed to emit binary signal", error2);
      throw error2;
    }
  }
  // Method to start listening to clipboard changes
  ListenToClipboardChanges() {
    try {
      if (!this.isListening && this.clipboardListener) {
        logger.debug("D-Bus: Starting clipboard listener...");
        this.clipboardManager.onClipboardChange(this.clipboardListener);
        this.isListening = true;
        logger.info(
          "\u{1F4E1} Binary D-Bus clipboard listener activated via method call"
        );
      } else if (this.isListening) {
        logger.debug("D-Bus: Clipboard listener already active");
      } else {
        logger.warn("D-Bus: No clipboard listener available");
      }
    } catch (error2) {
      logger.error("D-Bus: Error starting clipboard listener", error2);
      throw error2;
    }
  }
  // Method to manually trigger a clipboard change (for testing)
  TriggerClipboardChange(content, source = "user") {
    try {
      this.clipboardManager.triggerClipboardChange(
        content,
        source
      );
    } catch (error2) {
      logger.error("D-Bus: Error triggering clipboard change", error2);
      throw error2;
    }
  }
  // Method to get current clipboard content
  GetCurrentContent() {
    try {
      return this.clipboardManager.getCurrentContent();
    } catch (error2) {
      logger.error(
        "D-Bus: Error getting current clipboard content",
        error2
      );
      throw error2;
    }
  }
  // Method to set clipboard content
  SetContent(content) {
    try {
      this.clipboardManager.setContent(content);
    } catch (error2) {
      logger.error("D-Bus: Error setting clipboard content", error2);
      throw error2;
    }
  }
  // Method to set binary clipboard content
  SetContentBinary(content, mimeType) {
    try {
      this.clipboardManager.setContentBinary(content, mimeType);
    } catch (error2) {
      logger.error(
        "D-Bus: Error setting binary clipboard content",
        error2
      );
      throw error2;
    }
  }
  // Method to get available MIME types
  GetClipboardMimeTypes() {
    try {
      return [
        "text/plain",
        "text/html",
        "image/png",
        "image/jpeg",
        "image/gif",
        "image/webp",
        "application/x-vicinae-concealed"
      ];
    } catch (error2) {
      logger.error("D-Bus: Error getting clipboard MIME types", error2);
      return [];
    }
  }
  // Method to stop listening to clipboard changes
  StopListening() {
    try {
      if (this.clipboardListener && this.isListening) {
        this.clipboardManager.removeClipboardListener(
          this.clipboardListener
        );
        this.clipboardListener = null;
        this.isListening = false;
        logger.info("\u{1F515} Binary D-Bus clipboard listener deactivated");
      }
    } catch (error2) {
      logger.error("Error stopping clipboard listener", error2);
    }
  }
  // Cleanup method
  destroy() {
    this.StopListening();
    this.dbusObject = null;
    logger.info("ClipboardService destroyed");
  }
};

// src/core/dbus/services/windows-service.ts
import GLib4 from "gi://GLib";

// src/core/windows/window-manager.ts
import GLib3 from "gi://GLib";
import Meta2 from "gi://Meta";

// src/core/windows/workspace-manager.ts
var WorkspaceManager = class {
  getWorkspaceCount() {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_n_workspaces();
    } catch (error2) {
      logger.error("Error getting workspace count", error2);
      return 0;
    }
  }
  getCurrentWorkspaceIndex() {
    try {
      const workspaceManager = global.workspace_manager;
      const currentWorkspace = workspaceManager.get_active_workspace();
      return currentWorkspace.index();
    } catch (error2) {
      logger.error("Error getting current workspace index", error2);
      return 0;
    }
  }
  getWorkspaceByIndex(index) {
    try {
      const workspaceManager = global.workspace_manager;
      return workspaceManager.get_workspace_by_index(index);
    } catch (error2) {
      logger.error("Error getting workspace by index", error2);
      return null;
    }
  }
  switchToWorkspace(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        workspace.activate(global.get_current_time());
      }
    } catch (error2) {
      logger.error("Error switching to workspace", error2);
      throw error2;
    }
  }
  getWorkspaceInfo(index) {
    try {
      const workspace = this.getWorkspaceByIndex(index);
      if (workspace) {
        const windows = workspace.list_windows();
        const hasFullscreen = windows.some(
          (win) => isMaximized(win) === 3
        );
        let monitor = 0;
        if (windows.length > 0) {
          monitor = windows[0].get_monitor();
        }
        return {
          index: workspace.index(),
          name: `Workspace ${workspace.index() + 1}`,
          isActive: workspace === global.workspace_manager.get_active_workspace(),
          windowCount: windows.length,
          monitor,
          hasfullscreen: hasFullscreen
        };
      }
      return null;
    } catch (error2) {
      logger.error("Error getting workspace info", error2);
      return null;
    }
  }
  getAllWorkspaces() {
    try {
      const workspaceManager = global.workspace_manager;
      const workspaces = [];
      for (let i = 0; i < workspaceManager.get_n_workspaces(); i++) {
        const workspace = workspaceManager.get_workspace_by_index(i);
        if (workspace) {
          const workspaceInfo = this.getWorkspaceInfo(i);
          if (workspaceInfo) {
            workspaces.push(workspaceInfo);
          }
        }
      }
      return workspaces;
    } catch (error2) {
      logger.error("Error getting all workspaces", error2);
      return [];
    }
  }
};

// src/core/windows/window-manager.ts
var VicinaeWindowManager = class {
  clipboardManager;
  PASTE_DELAY = 100;
  appClass;
  constructor(clipboardManager, appClass) {
    this.appClass = appClass;
    this.clipboardManager = clipboardManager;
  }
  isTargetWindow(wmClass) {
    return wmClass.toLowerCase().includes(this.appClass.toLowerCase()) || this.appClass.toLowerCase().includes(wmClass.toLowerCase());
  }
  list() {
    const windowActors = global.get_window_actors();
    const workspaceManager = global.workspace_manager;
    const windows = windowActors.map((w) => {
      const metaWindow = w.meta_window;
      const windowWorkspace = metaWindow.get_workspace();
      const frame = metaWindow.get_frame_rect();
      return {
        id: metaWindow.get_id(),
        title: metaWindow.get_title(),
        wm_class: metaWindow.get_wm_class(),
        wm_class_instance: metaWindow.get_wm_class_instance(),
        pid: metaWindow.get_pid(),
        maximized: isMaximized(metaWindow) !== 0,
        // 0 means not maximized
        display: metaWindow.get_display(),
        frame_type: metaWindow.get_frame_type(),
        window_type: metaWindow.get_window_type(),
        layer: metaWindow.get_layer(),
        monitor: metaWindow.get_monitor(),
        role: metaWindow.get_role(),
        width: frame.width,
        height: frame.height,
        x: frame.x,
        y: frame.y,
        in_current_workspace: metaWindow.located_on_workspace?.(
          workspaceManager.get_active_workspace?.()
        ),
        canclose: metaWindow.can_close(),
        canmaximize: metaWindow.can_maximize(),
        canminimize: metaWindow.can_minimize(),
        canshade: false,
        // can_shade() is not in the type definitions
        moveable: metaWindow.allows_move(),
        resizeable: metaWindow.allows_resize(),
        has_focus: metaWindow.has_focus(),
        workspace: windowWorkspace ? windowWorkspace.index() : -1
      };
    });
    return windows;
  }
  details(winid) {
    const w = getWindowById(winid);
    if (!w) {
      throw new Error("Window not found");
    }
    const metaWindow = w.meta_window;
    const workspaceManager = global.workspace_manager;
    const windowWorkspace = metaWindow.get_workspace();
    const frame = metaWindow.get_frame_rect();
    const win = {
      id: metaWindow.get_id(),
      title: metaWindow.get_title(),
      wm_class: metaWindow.get_wm_class(),
      wm_class_instance: metaWindow.get_wm_class_instance(),
      pid: metaWindow.get_pid(),
      maximized: isMaximized(metaWindow) !== 0,
      // 0 means not maximized
      display: metaWindow.get_display(),
      frame_type: metaWindow.get_frame_type(),
      window_type: metaWindow.get_window_type(),
      layer: metaWindow.get_layer(),
      monitor: metaWindow.get_monitor(),
      role: metaWindow.get_role(),
      width: frame.width,
      height: frame.height,
      x: frame.x,
      y: frame.y,
      in_current_workspace: metaWindow.located_on_workspace?.(
        workspaceManager.get_active_workspace?.()
      ),
      canclose: metaWindow.can_close(),
      canmaximize: metaWindow.can_maximize(),
      canminimize: metaWindow.can_minimize(),
      canshade: false,
      // can_shade() is not in the type definitions
      moveable: metaWindow.allows_move(),
      resizeable: metaWindow.allows_resize(),
      has_focus: metaWindow.has_focus(),
      workspace: windowWorkspace ? windowWorkspace.index() : -1
    };
    return win;
  }
  getTitle(winid) {
    const w = getWindowById(winid);
    if (w) {
      return w.meta_window.get_title();
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameRect(winid) {
    const w = getWindowById(winid);
    if (w) {
      const frame = w.meta_window.get_frame_rect();
      return {
        x: frame.x,
        y: frame.y,
        width: frame.width,
        height: frame.height
      };
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameBounds(winid) {
    const w = getWindowById(winid);
    if (w) {
      return {
        frame_bounds: w.meta_window.get_frame_bounds()
      };
    } else {
      throw new Error("Window not found");
    }
  }
  moveToWorkspace(winid, workspaceNum) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.change_workspace_by_index(workspaceNum, false);
    } else {
      throw new Error("Window not found");
    }
  }
  moveResize(winid, x, y, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta2.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(true, x, y, width, height);
    } else {
      throw new Error("Window not found");
    }
  }
  resize(winid, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta2.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(
        true,
        win.get_x(),
        win.get_y(),
        width,
        height
      );
    } else {
      throw new Error("Window not found");
    }
  }
  move(winid, x, y) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta2.MaximizeFlags.BOTH);
      }
      win.meta_window.move_frame(true, x, y);
    } else {
      throw new Error("Window not found");
    }
  }
  maximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.maximize(Meta2.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  minimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.minimize();
    } else {
      throw new Error("Window not found");
    }
  }
  unmaximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unmaximize(Meta2.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  unminimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unminimize();
    } else {
      throw new Error("Window not found");
    }
  }
  activate(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      const currentTime = getCurrentTime();
      const workspace = win.get_workspace();
      if (workspace) {
        workspace.activate_with_focus(win, currentTime);
      } else {
        win.activate(currentTime);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  close(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      try {
        if (win.get_id() === winid) {
          win.delete(getCurrentTime());
        } else {
          throw new Error(
            "Window ID mismatch - window may be destroyed"
          );
        }
      } catch (error2) {
        throw new Error(`Failed to close window ${winid}: ${error2}`);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  listWorkspaces() {
    const workspaceManager = new WorkspaceManager();
    return workspaceManager.getAllWorkspaces();
  }
  getActiveWorkspace() {
    const workspaceManager = new WorkspaceManager();
    const currentIndex = workspaceManager.getCurrentWorkspaceIndex();
    const workspace = workspaceManager.getWorkspaceInfo(currentIndex);
    if (!workspace) {
      throw new Error("No active workspace found");
    }
    return workspace;
  }
  pasteToFocusedWindow(winid, key, modifiers) {
    if (!this.clipboardManager) {
      logger.error("No clipboard manager available for paste operation");
      return false;
    }
    GLib3.timeout_add(GLib3.PRIORITY_DEFAULT, this.PASTE_DELAY, () => {
      logger.debug(
        `Sending keyboard paste for window ${winid} ${key} ${modifiers} after ${this.PASTE_DELAY}ms delay`
      );
      this.clipboardManager.triggerKeyboardPaste();
      return false;
    });
    logger.debug(
      `Triggered keyboard paste for window ${winid} ${key} ${modifiers}`
    );
    return true;
  }
  sendShortcut(winid, key, modifiers) {
    this.activate(winid);
    const modifiersArray = modifiers.split("|");
    const isNormalPaste = key === "v" && modifiersArray.length === 1 && modifiersArray[0] === "CONTROL";
    const isShiftPaste = key === "v" && modifiersArray.length === 2 && modifiersArray[0] === "CONTROL" && modifiersArray[1] === "SHIFT";
    if (isNormalPaste || isShiftPaste) {
      return this.pasteToFocusedWindow(winid, key, modifiers);
    } else {
      logger.warn(`Unhandled shortcut: ${key} ${modifiers}`);
      return false;
    }
  }
};

// src/core/dbus/services/windows-service.ts
var WindowsService = class {
  windowManager;
  dbusObject = null;
  // Signal connection IDs for cleanup
  windowOpenedSignalId = 0;
  windowFocusSignalId = 0;
  workspaceChangedSignalId = 0;
  // Track individual window signal IDs
  windowDestroySignalIds = /* @__PURE__ */ new Map();
  windowSizeSignalIds = /* @__PURE__ */ new Map();
  // Track previous focused window for paste-to-active-app functionality
  previousFocusedWindow = null;
  constructor(clipboardManager, appClass) {
    this.windowManager = new VicinaeWindowManager(
      clipboardManager,
      appClass
    );
  }
  // Method to set the D-Bus exported object (called by DBusManager)
  setDBusObject(dbusObject) {
    this.dbusObject = dbusObject;
    logger.debug("WindowsService: D-Bus object set for signal emission");
    this.setupWindowEventListeners();
  }
  setupWindowEventListeners() {
    logger.debug("WindowsService: Setting up GNOME window event listeners");
    const _display = global.display;
    this.windowOpenedSignalId = global.display.connect(
      "window-created",
      (_display2, window) => {
        try {
          const windowInfo = this.getWindowInfo(window);
          this.emitOpenWindow(
            windowInfo.id.toString(),
            windowInfo.workspace.toString(),
            windowInfo.wm_class,
            windowInfo.title
          );
          this.connectToWindowDestroy(window);
          this.connectToWindowSizeChanges(window);
        } catch (error2) {
          logger.debug(
            `Error handling window opened event: ${error2}`
          );
        }
      }
    );
    this.connectToExistingWindows();
    this.windowFocusSignalId = global.display.connect(
      "notify::focus-window",
      () => {
        try {
          GLib4.idle_add(GLib4.PRIORITY_DEFAULT_IDLE, () => {
            const focusWindow = global.display.focus_window;
            if (focusWindow) {
              const currentWmClass = focusWindow.get_wm_class() || "";
              if (!this.windowManager.isTargetWindow(
                currentWmClass
              )) {
                this.previousFocusedWindow = {
                  id: focusWindow.get_id(),
                  wmClass: currentWmClass
                };
              }
              this.emitFocusWindow(
                focusWindow.get_id().toString()
              );
            }
            return GLib4.SOURCE_REMOVE;
          });
        } catch (error2) {
          logger.debug(`Error handling window focus event: ${error2}`);
        }
      }
    );
    this.workspaceChangedSignalId = global.workspace_manager?.connect(
      "notify::active-workspace",
      () => {
        try {
          const activeWorkspace = global.workspace_manager?.get_active_workspace();
          if (activeWorkspace) {
            this.emitWorkspaceChanged(
              activeWorkspace.index().toString()
            );
          }
        } catch (error2) {
          logger.debug(
            `Error handling workspace changed event: ${error2}`
          );
        }
      }
    );
    logger.debug(
      "WindowsService: GNOME window event listeners set up successfully"
    );
  }
  connectToWindowDestroy(window) {
    const windowId = window.get_id();
    try {
      let signalId;
      let connectedSignal = "";
      try {
        logger.debug(
          `Attempting to connect 'destroy' signal for window ${windowId}`
        );
        signalId = window.connect("destroy", () => {
          try {
            logger.debug(
              `Window ${windowId} destroy signal triggered - emitting closewindow`
            );
            this.emitCloseWindow(windowId.toString());
            this.windowDestroySignalIds.delete(windowId);
          } catch (error2) {
            logger.debug(
              `Error emitting closewindow for ${windowId}: ${error2}`
            );
          }
        });
        connectedSignal = "destroy";
        logger.debug(
          `Successfully connected to 'destroy' signal for window ${windowId}`
        );
      } catch (_destroyError) {
        logger.debug(
          `'destroy' signal not available for window ${windowId}, trying 'unmanaged'`
        );
        try {
          signalId = window.connect("unmanaged", () => {
            try {
              logger.debug(
                `Window ${windowId} unmanaged signal triggered - emitting closewindow`
              );
              this.emitCloseWindow(windowId.toString());
              this.windowDestroySignalIds.delete(windowId);
            } catch (error2) {
              logger.debug(
                `Error emitting closewindow for ${windowId}: ${error2}`
              );
            }
          });
          connectedSignal = "unmanaged";
          logger.debug(
            `Successfully connected to 'unmanaged' signal for window ${windowId}`
          );
        } catch (_unmanagedError) {
          logger.debug(
            `No suitable destroy signal for window ${windowId}, skipping signal connection`
          );
          return;
        }
      }
      if (signalId !== void 0) {
        this.windowDestroySignalIds.set(windowId, signalId);
        logger.debug(
          `Successfully connected ${connectedSignal} signal for window ${windowId} (signal ID: ${signalId})`
        );
      }
    } catch (error2) {
      logger.debug(
        `Failed to connect any destroy signal for window ${windowId}: ${error2}`
      );
    }
  }
  connectToWindowSizeChanges(window) {
    const windowId = window.get_id();
    try {
      const signalId = window.connect("size-changed", () => {
        try {
          const windowInfo = this.getWindowInfo(window);
          logger.debug(
            `Window ${windowId} size changed - emitting movewindow`
          );
          this.emitMoveWindow(
            windowInfo.id.toString(),
            windowInfo.x,
            windowInfo.y,
            windowInfo.width,
            windowInfo.height
          );
        } catch (error2) {
          logger.debug(
            `Error handling size change for window ${windowId}: ${error2}`
          );
        }
      });
      this.windowSizeSignalIds.set(windowId, signalId);
      logger.debug(
        `Connected size-changed signal for window ${windowId}`
      );
    } catch (error2) {
      logger.debug(
        `Failed to connect size-changed signal for window ${windowId}: ${error2}`
      );
    }
  }
  connectToExistingWindows() {
    try {
      const windowActors = global.get_window_actors();
      logger.debug(
        `WindowsService: Connecting to ${windowActors.length} existing windows`
      );
      for (const actor of windowActors) {
        if (actor.meta_window) {
          this.connectToWindowDestroy(actor.meta_window);
          this.connectToWindowSizeChanges(actor.meta_window);
        }
      }
    } catch (error2) {
      logger.debug(`Error connecting to existing windows: ${error2}`);
    }
  }
  getWindowInfo(window) {
    let x = 0, y = 0, width = 0, height = 0;
    try {
      if (typeof window.get_frame_rect === "function") {
        const frame = window.get_frame_rect();
        x = frame.x;
        y = frame.y;
        width = frame.width;
        height = frame.height;
      } else {
        logger.debug(
          `Window ${window.get_id()} does not have get_frame_rect method`
        );
      }
    } catch (error2) {
      logger.debug(
        `Error getting frame rect for window ${window.get_id()}: ${error2}`
      );
    }
    const workspace = window.get_workspace();
    return {
      id: window.get_id(),
      title: window.get_title(),
      wm_class: window.get_wm_class() || "",
      workspace: workspace ? workspace.index() : -1,
      x,
      y,
      width,
      height
    };
  }
  // Cleanup method to disconnect signals
  destroy() {
    logger.debug("WindowsService: Cleaning up window event listeners");
    if (this.windowOpenedSignalId) {
      global.display.disconnect(this.windowOpenedSignalId);
    }
    if (this.windowFocusSignalId) {
      global.display.disconnect(this.windowFocusSignalId);
    }
    if (this.workspaceChangedSignalId && global.workspace_manager) {
      global.workspace_manager.disconnect(this.workspaceChangedSignalId);
    }
    const allWindowIds = /* @__PURE__ */ new Set([
      ...this.windowDestroySignalIds.keys(),
      ...this.windowSizeSignalIds.keys()
    ]);
    for (const windowId of allWindowIds) {
      try {
        const windowActors = global.get_window_actors();
        const windowActor = windowActors.find(
          (actor) => actor.meta_window?.get_id() === windowId
        );
        if (windowActor?.meta_window) {
          const destroySignalId = this.windowDestroySignalIds.get(windowId);
          if (destroySignalId) {
            windowActor.meta_window.disconnect(destroySignalId);
          }
          const sizeSignalId = this.windowSizeSignalIds.get(windowId);
          if (sizeSignalId) {
            windowActor.meta_window.disconnect(sizeSignalId);
          }
        }
      } catch (error2) {
        logger.debug(
          `Error disconnecting signals for window ${windowId}: ${error2}`
        );
      }
    }
    this.windowDestroySignalIds.clear();
    this.windowSizeSignalIds.clear();
    logger.debug("WindowsService: Window event listeners cleaned up");
  }
  List() {
    try {
      GLib4.usleep(1e3);
      const windows = this.windowManager.list();
      const filteredWindows = windows.filter(
        (window) => !this.windowManager.isTargetWindow(window.wm_class)
      );
      return JSON.stringify(filteredWindows);
    } catch (error2) {
      logger.error("D-Bus: Error listing windows", error2);
      throw error2;
    }
  }
  Details(winid) {
    try {
      const details = this.windowManager.details(winid);
      return JSON.stringify(details);
    } catch (error2) {
      logger.error("D-Bus: Error getting window details", error2);
      throw error2;
    }
  }
  GetTitle(winid) {
    try {
      return this.windowManager.getTitle(winid);
    } catch (error2) {
      logger.error("D-Bus: Error getting window title", error2);
      throw error2;
    }
  }
  GetFrameRect(winid) {
    try {
      const frameRect = this.windowManager.getFrameRect(winid);
      return JSON.stringify(frameRect);
    } catch (error2) {
      logger.error("D-Bus: Error getting window frame rect", error2);
      throw error2;
    }
  }
  GetFrameBounds(winid) {
    try {
      const frameBounds = this.windowManager.getFrameBounds(winid);
      return JSON.stringify(frameBounds);
    } catch (error2) {
      logger.error("D-Bus: Error getting window frame bounds", error2);
      throw error2;
    }
  }
  MoveToWorkspace(winid, workspaceNum) {
    try {
      this.windowManager.moveToWorkspace(winid, workspaceNum);
    } catch (error2) {
      logger.error("D-Bus: Error moving window to workspace", error2);
      throw error2;
    }
  }
  MoveResize(winid, x, y, width, height) {
    try {
      this.windowManager.moveResize(winid, x, y, width, height);
    } catch (error2) {
      logger.error("D-Bus: Error move resizing window", error2);
      throw error2;
    }
  }
  Resize(winid, width, height) {
    try {
      this.windowManager.resize(winid, width, height);
    } catch (error2) {
      logger.error("D-Bus: Error resizing window", error2);
      throw error2;
    }
  }
  Move(winid, x, y) {
    try {
      this.windowManager.move(winid, x, y);
    } catch (error2) {
      logger.error("D-Bus: Error moving window", error2);
      throw error2;
    }
  }
  Maximize(winid) {
    try {
      this.windowManager.maximize(winid);
    } catch (error2) {
      logger.error("D-Bus: Error maximizing window", error2);
      throw error2;
    }
  }
  Minimize(winid) {
    try {
      this.windowManager.minimize(winid);
    } catch (error2) {
      logger.error("D-Bus: Error minimizing window", error2);
      throw error2;
    }
  }
  Unmaximize(winid) {
    try {
      this.windowManager.unmaximize(winid);
    } catch (error2) {
      logger.error("D-Bus: Error unmaximizing window", error2);
      throw error2;
    }
  }
  Unminimize(winid) {
    try {
      this.windowManager.unminimize(winid);
    } catch (error2) {
      logger.error("D-Bus: Error unminimizing window", error2);
      throw error2;
    }
  }
  Activate(winid) {
    try {
      this.windowManager.activate(winid);
    } catch (error2) {
      logger.error("D-Bus: Error activating window", error2);
      throw error2;
    }
  }
  Close(winid) {
    try {
      this.windowManager.close(winid);
    } catch (error2) {
      logger.error("D-Bus: Error closing window", error2);
      throw error2;
    }
  }
  ListWorkspaces() {
    try {
      const workspaces = this.windowManager.listWorkspaces();
      return JSON.stringify(workspaces);
    } catch (error2) {
      logger.error("D-Bus: Error listing workspaces", error2);
      throw error2;
    }
  }
  GetActiveWorkspace() {
    try {
      const workspace = this.windowManager.getActiveWorkspace();
      return JSON.stringify(workspace);
    } catch (error2) {
      logger.error("D-Bus: Error getting active workspace", error2);
      throw error2;
    }
  }
  GetWorkspaceWindows(workspaceIndex) {
    try {
      const windows = this.windowManager.list();
      const workspaceWindows = windows.filter(
        (win) => win.workspace === workspaceIndex
      );
      return JSON.stringify(workspaceWindows);
    } catch (error2) {
      logger.error("D-Bus: Error getting workspace windows", error2);
      throw error2;
    }
  }
  SendShortcut(winid, key, modifiers) {
    let success = false;
    try {
      success = this.windowManager.sendShortcut(winid, key, modifiers);
    } catch (error2) {
      logger.error("D-Bus: Error sending shortcut", error2);
      return false;
    }
    return success;
  }
  GetFocusedWindowSync() {
    try {
      const focusedWindow = getFocusedWindow();
      if (!focusedWindow) {
        logger.debug("GetFocusedWindowSync: No focused window");
        return JSON.stringify(null);
      }
      const currentWmClass = focusedWindow.meta_window.get_wm_class() || "";
      if (this.windowManager.isTargetWindow(currentWmClass)) {
        if (this.previousFocusedWindow) {
          logger.debug(
            `GetFocusedWindowSync: Vicinae focused, returning previous window: ${this.previousFocusedWindow.wmClass}`
          );
          const windowDetails2 = this.windowManager.details(
            this.previousFocusedWindow.id
          );
          return JSON.stringify(windowDetails2);
        } else {
          logger.debug(
            "GetFocusedWindowSync: Vicinae focused but no previous window"
          );
          return JSON.stringify(null);
        }
      }
      logger.debug(
        `GetFocusedWindowSync: Returning current focused window: ${currentWmClass}`
      );
      const windowDetails = this.windowManager.details(
        focusedWindow.meta_window.get_id()
      );
      return JSON.stringify(windowDetails);
    } catch (error2) {
      logger.error("D-Bus: Error in GetFocusedWindowSync", error2);
      return JSON.stringify(null);
    }
  }
  // Signal emission methods - called by window manager when events occur
  emitOpenWindow(windowAddress, workspaceName, wmClass, title) {
    try {
      logger.debug(
        `Emitting openwindow signal for ${title} (${wmClass})`
      );
      this.dbusObject?.emit_signal(
        "openwindow",
        GLib4.Variant.new("(ssss)", [
          String(windowAddress),
          String(workspaceName),
          String(wmClass),
          String(title)
        ])
      );
    } catch (signalError) {
      logger.error("Error emitting openwindow signal", signalError);
    }
  }
  emitCloseWindow(windowAddress) {
    try {
      logger.debug(`Emitting closewindow signal for ${windowAddress}`);
      this.dbusObject?.emit_signal(
        "closewindow",
        GLib4.Variant.new("(s)", [String(windowAddress)])
      );
    } catch (signalError) {
      logger.error("Error emitting closewindow signal", signalError);
    }
  }
  emitFocusWindow(windowAddress) {
    try {
      logger.debug(`Emitting focuswindow signal for ${windowAddress}`);
      this.dbusObject?.emit_signal(
        "focuswindow",
        GLib4.Variant.new("(s)", [String(windowAddress)])
      );
    } catch (signalError) {
      logger.error("Error emitting focuswindow signal", signalError);
    }
  }
  emitMoveWindow(windowAddress, x, y, width, height) {
    try {
      logger.debug(`Emitting movewindow signal for ${windowAddress}`);
      this.dbusObject?.emit_signal(
        "movewindow",
        GLib4.Variant.new("(siiuu)", [
          String(windowAddress),
          x,
          y,
          width,
          height
        ])
      );
    } catch (signalError) {
      logger.error("Error emitting movewindow signal", signalError);
    }
  }
  emitStateWindow(windowAddress, state) {
    try {
      logger.debug(
        `Emitting statewindow signal for ${windowAddress}: ${state}`
      );
      this.dbusObject?.emit_signal(
        "statewindow",
        GLib4.Variant.new("(ss)", [
          String(windowAddress),
          String(state)
        ])
      );
    } catch (signalError) {
      logger.error("Error emitting statewindow signal", signalError);
    }
  }
  emitWorkspaceChanged(workspaceId) {
    try {
      logger.debug(`Emitting workspacechanged signal for ${workspaceId}`);
      this.dbusObject?.emit_signal(
        "workspacechanged",
        GLib4.Variant.new("(s)", [String(workspaceId)])
      );
    } catch (signalError) {
      logger.error("Error emitting workspacechanged signal", signalError);
    }
  }
  emitMonitorLayoutChanged() {
    try {
      logger.debug("Emitting monitorlayoutchanged signal");
      this.dbusObject?.emit_signal(
        "monitorlayoutchanged",
        GLib4.Variant.new("()", [])
      );
    } catch (signalError) {
      logger.error(
        "Error emitting monitorlayoutchanged signal",
        signalError
      );
    }
  }
};

// src/core/dbus/manager.ts
var DBusManager = class {
  clipboardService;
  windowsService;
  clipboardServiceInstance;
  windowsServiceInstance;
  constructor(appClass, extension, clipboardManager) {
    if (!clipboardManager) {
      throw new Error(
        "ClipboardManager instance is required for DBusManager"
      );
    }
    this.clipboardServiceInstance = new ClipboardService(
      clipboardManager,
      extension
    );
    this.windowsServiceInstance = new WindowsService(
      clipboardManager,
      appClass
    );
    this.clipboardService = createDBusService(
      CLIPBOARD_DBUS_IFACE,
      // biome-ignore lint/suspicious/noExplicitAny: we need to cast the instance to any to avoid type errors
      this.clipboardServiceInstance
    );
    this.windowsService = createDBusService(
      WINDOWS_DBUS_IFACE,
      // biome-ignore lint/suspicious/noExplicitAny: we need to cast the instance to any to avoid type errors
      this.windowsServiceInstance
    );
    this.clipboardServiceInstance.setDBusObject(this.clipboardService);
    this.windowsServiceInstance.setDBusObject(this.windowsService);
  }
  exportServices() {
    try {
      exportDBusService(
        this.clipboardService,
        "/org/gnome/Shell/Extensions/Clipboard"
      );
      exportDBusService(
        this.windowsService,
        "/org/gnome/Shell/Extensions/Windows"
      );
      logger.info("D-Bus services exported successfully");
    } catch (_error) {
      logger.error("Failed to export D-Bus services", _error);
      throw _error;
    }
  }
  unexportServices() {
    try {
      this.clipboardServiceInstance.destroy();
      this.windowsServiceInstance.destroy();
      unexportDBusService(this.clipboardService);
      unexportDBusService(this.windowsService);
      logger.info("D-Bus services unexported successfully");
    } catch (_error) {
      logger.error("Failed to unexport D-Bus services", _error);
      throw _error;
    }
  }
  getClipboardService() {
    return this.clipboardServiceInstance;
  }
  getWindowsService() {
    return this.windowsServiceInstance;
  }
};

// src/core/launcher/click-handler.ts
import Clutter2 from "gi://Clutter";
var ClickHandler = class {
  constructor(appClass, onClickOutside) {
    this.appClass = appClass;
    this.onClickOutside = onClickOutside;
  }
  buttonPressHandler;
  enable() {
    try {
      this.buttonPressHandler = global.stage.connect(
        "captured-event",
        (_stage, event) => {
          if (event.type() === Clutter2.EventType.BUTTON_PRESS) {
            this.handleClick();
          }
          return Clutter2.EVENT_PROPAGATE;
        }
      );
      logger.info("ClickHandler: Click tracking enabled");
    } catch (error2) {
      logger.error("ClickHandler: Error enabling click tracking", error2);
      throw error2;
    }
  }
  disable() {
    if (this.buttonPressHandler) {
      global.stage.disconnect(this.buttonPressHandler);
      this.buttonPressHandler = void 0;
    }
    logger.info("ClickHandler: Click tracking disabled");
  }
  handleClick() {
    const [x, y] = global.get_pointer();
    const windows = global.get_window_actors();
    const window = windows.find((actor) => {
      const win = actor.meta_window;
      if (!win) return false;
      const rect = win.get_frame_rect();
      return x >= rect.x && x <= rect.x + rect.width && y >= rect.y && y <= rect.y + rect.height;
    });
    const clickedWindow = window?.meta_window;
    const isOurWindow = clickedWindow?.get_wm_class()?.toLowerCase().includes(this.appClass.toLowerCase());
    if (!clickedWindow || !isOurWindow) {
      this.onClickOutside();
    }
  }
};

// src/core/launcher/focus-tracker.ts
var FocusTracker = class {
  constructor(onFocusChange) {
    this.onFocusChange = onFocusChange;
  }
  focusHandler;
  enable() {
    try {
      this.focusHandler = global.display.connect(
        "notify::focus-window",
        (_display, _window) => {
          this.onFocusChange();
        }
      );
      logger.info("FocusTracker: Focus tracking enabled");
    } catch (error2) {
      logger.error("FocusTracker: Error enabling focus tracking", error2);
      throw error2;
    }
  }
  disable() {
    if (this.focusHandler) {
      global.display.disconnect(this.focusHandler);
      this.focusHandler = void 0;
    }
    logger.info("FocusTracker: Focus tracking disabled");
  }
};

// src/core/launcher/window-tracker.ts
import GLib5 from "gi://GLib";
var WindowTracker = class {
  constructor(appClass, onWindowTracked, onWindowUntracked) {
    this.appClass = appClass;
    this.onWindowTracked = onWindowTracked;
    this.onWindowUntracked = onWindowUntracked;
  }
  trackedWindows = /* @__PURE__ */ new Set();
  windowCreatedHandler;
  windowDestroySignalIds = /* @__PURE__ */ new Map();
  windowValidators = /* @__PURE__ */ new Map();
  isDestroying = false;
  enable() {
    try {
      this.windowCreatedHandler = global.display.connect(
        "window-created",
        (_display, window) => {
          GLib5.idle_add(GLib5.PRIORITY_DEFAULT, () => {
            this.handleNewWindow(window);
            return GLib5.SOURCE_REMOVE;
          });
        }
      );
      this.scanExistingWindows();
      logger.info("WindowTracker: Window tracking enabled");
    } catch (error2) {
      logger.error(
        "WindowTracker: Error enabling window tracking",
        error2
      );
      throw error2;
    }
  }
  disable() {
    this.isDestroying = true;
    if (this.windowCreatedHandler) {
      global.display.disconnect(this.windowCreatedHandler);
      this.windowCreatedHandler = void 0;
    }
    for (const [windowId, signalId] of this.windowDestroySignalIds) {
      if (signalId) {
        try {
          const window = this.getWindowById(windowId);
          if (window && this.isWindowValid(window)) {
            window.disconnect(signalId);
          }
        } catch (_error) {
          logger.debug(
            `WindowTracker: Signal already disconnected for window ${windowId}`
          );
        }
      }
    }
    this.windowDestroySignalIds.clear();
    this.windowValidators.clear();
    this.trackedWindows.clear();
    this.isDestroying = false;
    logger.info("WindowTracker: Window tracking disabled");
  }
  scanExistingWindows() {
    try {
      const windowActors = global.get_window_actors();
      logger.debug(
        `WindowTracker: Scanning ${windowActors.length} existing windows`
      );
      windowActors.forEach((actor) => {
        if (actor?.meta_window) {
          this.handleNewWindow(actor.meta_window);
        } else {
          logger.debug(
            "WindowTracker: Skipping invalid window actor"
          );
        }
      });
    } catch (error2) {
      logger.error(
        "WindowTracker: Error scanning existing windows",
        error2
      );
    }
  }
  handleNewWindow(window) {
    if (this.isDestroying || !this.isValidWindow(window)) {
      logger.debug(
        "WindowTracker: Invalid window object or destroying, skipping"
      );
      return;
    }
    try {
      const wmClass = window.get_wm_class();
      const title = window.get_title();
      const windowId = window.get_id();
      if (!wmClass || !title || windowId <= 0) {
        logger.debug("WindowTracker: Invalid window properties", {
          wmClass,
          title,
          windowId
        });
        return;
      }
      if (wmClass.toLowerCase().includes(this.appClass.toLowerCase())) {
        if (!this.trackedWindows.has(windowId)) {
          this.trackedWindows.add(windowId);
          const validator = () => this.isWindowValid(window);
          this.windowValidators.set(windowId, validator);
          this.onWindowTracked(windowId);
          this.centerWindow(window);
          let signalId;
          try {
            signalId = window.connect("destroy", () => {
              this.handleWindowDestroyed(window);
            });
          } catch (_error) {
            try {
              signalId = window.connect("unmanaged", () => {
                this.handleWindowDestroyed(window);
              });
            } catch (_unmanagedError) {
              logger.debug(
                `WindowTracker: No suitable destroy signal for window ${windowId}, skipping signal connection`
              );
            }
          }
          if (signalId) {
            this.windowDestroySignalIds.set(windowId, signalId);
          }
          logger.debug(
            `WindowTracker: Tracking new window ${windowId} (${wmClass})`
          );
        }
      }
    } catch (error2) {
      logger.error("WindowTracker: Error handling new window", {
        error: error2 instanceof Error ? error2.message : String(error2),
        stack: error2 instanceof Error ? error2.stack : void 0,
        windowId: window?.get_id?.() || "unknown"
      });
    }
  }
  isValidWindow(window) {
    if (!window) return false;
    if (typeof window.get_wm_class !== "function") return false;
    if (typeof window.get_id !== "function") return false;
    try {
      const windowId = window.get_id();
      return windowId > 0 && windowId !== void 0;
    } catch {
      return false;
    }
  }
  isWindowValid(window) {
    if (!window) return false;
    try {
      const windowActors = global.get_window_actors();
      const stillExists = windowActors.some(
        (actor) => actor.meta_window && actor.meta_window.get_id() === window.get_id()
      );
      if (!stillExists) return false;
      window.get_id();
      window.get_wm_class();
      return true;
    } catch {
      return false;
    }
  }
  handleWindowDestroyed(window) {
    if (this.isDestroying || !this.isValidWindow(window)) return;
    const windowId = window.get_id();
    if (this.trackedWindows.has(windowId)) {
      this.trackedWindows.delete(windowId);
      this.windowValidators.delete(windowId);
      this.onWindowUntracked(windowId);
      this.windowDestroySignalIds.delete(windowId);
      logger.debug(
        `WindowTracker: Untracking destroyed window ${windowId}`
      );
    }
  }
  getWindowById(windowId) {
    try {
      const actors = global.get_window_actors();
      for (const actor of actors) {
        const window = actor.meta_window;
        if (window && window.get_id() === windowId) {
          return window;
        }
      }
    } catch (error2) {
      logger.error(
        `WindowTracker: Error finding window ${windowId}`,
        error2
      );
    }
    return null;
  }
  // Public methods for external access
  getTrackedWindows() {
    return Array.from(this.trackedWindows);
  }
  getTrackedWindowsCount() {
    return this.trackedWindows.size;
  }
  /**
   * Centers a window on the current monitor
   */
  centerWindow(window) {
    if (this.isDestroying || !this.isWindowValid(window)) {
      logger.debug(
        "WindowTracker: Skipping center window - window invalid or destroying"
      );
      return;
    }
    try {
      const { x, y } = this.getCenterPosition(window);
      window.move_frame(true, x, y);
      logger.debug(
        `WindowTracker: Centered window ${window.get_id()} at (${Math.round(
          x
        )}, ${Math.round(y)})`
      );
    } catch (error2) {
      logger.error("WindowTracker: Error centering window", error2);
    }
  }
  /**
   * Gets the center position of a window
   */
  getCenterPosition(window) {
    const monitor = window.get_monitor();
    const display = global.display;
    const monitorGeometry = display.get_monitor_geometry(monitor);
    const frame = window.get_frame_rect();
    const centerX = monitorGeometry.x + (monitorGeometry.width - frame.width) / 2;
    const centerY = monitorGeometry.y + (monitorGeometry.height - frame.height) / 2;
    return { x: centerX, y: centerY };
  }
};

// src/core/launcher/launcher-manager.ts
var LauncherManager = class {
  windowManager;
  windowTracker;
  focusTracker;
  clickHandler;
  config;
  isEnabled = false;
  trackedWindows = /* @__PURE__ */ new Set();
  constructor(config, clipboardManager) {
    this.windowManager = new VicinaeWindowManager(clipboardManager);
    this.config = config;
    this.windowTracker = new WindowTracker(
      config.appClass,
      this.handleWindowTracked.bind(this),
      this.handleWindowUntracked.bind(this)
    );
  }
  async enable() {
    if (this.isEnabled) {
      logger.debug("LauncherManager: Already enabled, skipping");
      return;
    }
    try {
      await this.windowTracker.enable();
      if (this.config.autoCloseOnFocusLoss) {
        this.setupFocusTracking();
        this.setupClickHandling();
      }
      this.isEnabled = true;
      logger.info("LauncherManager: Successfully enabled");
    } catch (error2) {
      logger.error("LauncherManager: Error during enable", error2);
      this.cleanup();
      throw error2;
    }
  }
  handleWindowTracked(windowId) {
    this.trackedWindows.add(windowId);
    logger.debug(`LauncherManager: Window ${windowId} is now tracked`);
  }
  handleWindowUntracked(windowId) {
    this.trackedWindows.delete(windowId);
    logger.debug(
      `LauncherManager: Window ${windowId} is no longer tracked`
    );
  }
  disable() {
    logger.info("LauncherManager: Disabling");
    this.isEnabled = false;
    this.cleanup();
  }
  setupFocusTracking() {
    this.focusTracker = new FocusTracker(() => this.handleFocusChange());
    this.focusTracker.enable();
  }
  setupClickHandling() {
    this.clickHandler = new ClickHandler(
      this.config.appClass,
      () => this.closeTrackedWindows()
    );
    this.clickHandler.enable();
  }
  handleFocusChange() {
    if (!this.isEnabled) return;
    const focusedWindow = global.display.get_focus_window();
    if (!focusedWindow) {
      this.closeTrackedWindows();
      return;
    }
    const focusedWmClass = focusedWindow.get_wm_class()?.toLowerCase() || "";
    if (!focusedWmClass.includes(this.config.appClass.toLowerCase())) {
      this.closeTrackedWindows();
    }
  }
  closeTrackedWindows() {
    if (this.trackedWindows.size === 0) return;
    logger.debug(
      `LauncherManager: Closing ${this.trackedWindows.size} launcher windows due to focus loss`
    );
    const windowsToClose = Array.from(this.trackedWindows);
    this.trackedWindows.clear();
    windowsToClose.forEach((windowId) => {
      try {
        if (this.isValidWindowId(windowId)) {
          this.windowManager.close(windowId);
          logger.debug(
            `LauncherManager: Successfully closed window ${windowId}`
          );
        } else {
          logger.debug(
            `LauncherManager: Window ${windowId} no longer valid, skipping close`
          );
        }
      } catch (error2) {
        logger.error(
          `LauncherManager: Error closing window ${windowId}`,
          error2
        );
      }
    });
  }
  isValidWindowId(windowId) {
    if (!windowId || windowId <= 0) return false;
    try {
      const windowActors = global.get_window_actors();
      const windowExists = windowActors.some((actor) => {
        try {
          return actor.meta_window && actor.meta_window.get_id() === windowId;
        } catch {
          return false;
        }
      });
      if (!windowExists) return false;
      const details = this.windowManager.details(windowId);
      return details && details.id === windowId;
    } catch {
      return false;
    }
  }
  cleanup() {
    try {
      this.windowTracker.disable();
      this.focusTracker?.disable();
      this.clickHandler?.disable();
      this.trackedWindows.clear();
    } catch (error2) {
      logger.error("LauncherManager: Error during cleanup", error2);
    }
  }
  // Recovery method for error situations
  recover() {
    logger.warn("LauncherManager: Attempting recovery from errors");
    try {
      this.cleanup();
      this.isEnabled = false;
      setTimeout(() => {
        this.enable();
      }, 500);
      logger.info("LauncherManager: Recovery initiated");
    } catch (error2) {
      logger.error("LauncherManager: Recovery failed", error2);
    }
  }
  // Public methods for dynamic configuration
  updateConfig(newConfig) {
    const oldConfig = { ...this.config };
    this.config = { ...this.config, ...newConfig };
    logger.debug("LauncherManager: Configuration updated", {
      old: oldConfig,
      new: this.config
    });
    if (this.isEnabled) {
      logger.info("LauncherManager: Re-enabling with new configuration");
      this.disable();
      setTimeout(() => this.enable(), 100);
    }
  }
  getTrackedWindows() {
    return Array.from(this.trackedWindows);
  }
  getStatus() {
    return {
      isEnabled: this.isEnabled,
      trackedWindowsCount: this.trackedWindows.size,
      config: this.config,
      hasFocusTracker: !!this.focusTracker,
      hasClickHandler: !!this.clickHandler,
      hasWindowTracker: !!this.windowTracker
    };
  }
  // Force refresh of tracked windows
  refresh() {
    logger.debug("LauncherManager: Refreshing tracked windows");
    this.trackedWindows.clear();
  }
};

// src/extension.ts
var getVirtualKeyboard = /* @__PURE__ */ (() => {
  let virtualKeyboard;
  return () => {
    if (!virtualKeyboard) {
      virtualKeyboard = Clutter3.get_default_backend().get_default_seat().create_virtual_device(Clutter3.InputDeviceType.KEYBOARD_DEVICE);
    }
    return virtualKeyboard;
  };
})();
var Vicinae = class extends Extension {
  indicator;
  dbusManager;
  clipboardManager;
  launcherManager;
  settings;
  settingsConnection;
  launcherSettingsConnection;
  async enable() {
    logger.info("Vicinae extension enabled");
    try {
      this.settings = this.getSettings();
      initializeLogger(this.settings);
      this.clipboardManager = new VicinaeClipboardManager(
        getVirtualKeyboard()
      );
      this.clipboardManager.setSettings(this.settings);
      const appClass = this.settings.get_string("launcher-app-class") || "vicinae";
      this.dbusManager = new DBusManager(
        appClass,
        this,
        this.clipboardManager
      );
      this.dbusManager.exportServices();
      logger.debug("Extension: Initializing launcher manager...");
      await this.initializeLauncherManager();
      this.updateIndicatorVisibility();
      this.settingsConnection = this.settings.connect(
        "changed::show-status-indicator",
        () => {
          this.updateIndicatorVisibility();
        }
      );
      this.launcherSettingsConnection = this.settings.connect(
        "changed::launcher-auto-close-focus-loss",
        () => {
          this.updateLauncherManager();
        }
      );
      this.settings.connect("changed::blocked-applications", () => {
        if (this.clipboardManager && this.settings) {
          this.clipboardManager.updateSettings(this.settings);
          logger.debug(
            "Updated clipboard manager with new blocked applications list"
          );
        }
      });
      logger.info("Vicinae extension initialized successfully");
    } catch (error2) {
      logger.error(
        "Failed to initialize Vicinae extension",
        error2
      );
      throw error2;
    }
  }
  updateIndicatorVisibility() {
    const shouldShow = this.settings?.get_boolean("show-status-indicator");
    if (shouldShow && !this.indicator) {
      this.indicator = new VicinaeIndicator(this);
      Main.panel.addToStatusArea(
        "vicinae-gnome-extension",
        this.indicator.getButton(),
        0,
        "right"
      );
      logger.debug("Vicinae indicator shown");
    } else if (!shouldShow && this.indicator) {
      this.indicator.destroy();
      this.indicator = null;
      logger.debug("Vicinae indicator hidden");
    }
  }
  async initializeLauncherManager() {
    if (!this.settings) return;
    const autoClose = this.settings.get_boolean(
      "launcher-auto-close-focus-loss"
    );
    const appClass = this.settings.get_string("launcher-app-class") || "vicinae";
    if (autoClose) {
      if (!this.clipboardManager) {
        throw new Error("Clipboard manager is not initialized");
      }
      this.launcherManager = new LauncherManager(
        {
          appClass,
          autoCloseOnFocusLoss: autoClose
        },
        this.clipboardManager
      );
      await this.launcherManager.enable();
      logger.info("Launcher manager initialized and enabled");
    }
  }
  async updateLauncherManager() {
    if (!this.settings) return;
    const autoClose = this.settings.get_boolean(
      "launcher-auto-close-focus-loss"
    );
    if (autoClose && !this.launcherManager) {
      await this.initializeLauncherManager();
    } else if (!autoClose && this.launcherManager) {
      this.launcherManager.disable();
      this.launcherManager = null;
      logger.debug("Launcher manager disabled");
    } else if (autoClose && this.launcherManager) {
      const appClass = this.settings.get_string("launcher-app-class") || "vicinae";
      logger.debug(
        "initializeShellIntegrationManager: Using app class",
        appClass
      );
      this.launcherManager.updateConfig({
        appClass,
        autoCloseOnFocusLoss: autoClose
      });
    }
  }
  disable() {
    logger.info("Vicinae extension disabled");
    try {
      if (this.settingsConnection) {
        this.settings?.disconnect(this.settingsConnection);
        this.settingsConnection = 0;
      }
      if (this.launcherSettingsConnection) {
        this.settings?.disconnect(this.launcherSettingsConnection);
        this.launcherSettingsConnection = 0;
      }
      if (this.launcherManager) {
        this.launcherManager.disable();
        this.launcherManager = null;
      }
      if (this.indicator) {
        this.indicator.destroy();
        this.indicator = null;
      }
      if (this.dbusManager) {
        this.dbusManager.unexportServices();
        this.dbusManager = null;
      }
      if (this.clipboardManager) {
        this.clipboardManager.destroy();
        this.clipboardManager = null;
      }
      this.settings = null;
      logger.info("Vicinae extension cleaned up successfully");
    } catch (error2) {
      logger.error("Error during extension cleanup", error2);
    }
  }
};
export {
  Vicinae as default
};
