import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";

export default class InPicture extends Extension {
  constructor(metadata) {
    super(metadata);
    this.windowCreatedSignal = null;
    this.settings = null;
  }

  enable() {
    this.settings = this.getSettings();
    this.windowCreatedSignal = global.display.connect(
      "window-created",
      (display, metaWindow) => {
        const wrapper = metaWindow.get_compositor_private();
        const windowFirstSignal = wrapper.connect("first-frame", () => {
          this.handleWindow(metaWindow);
          wrapper.disconnect(windowFirstSignal);
        });
      }
    );
  }

  disable() {
    this.settings = null;
    if (this.windowCreatedSignal) {
      global.display.disconnect(this.windowCreatedSignal);
      this.windowCreatedSignal = null;
    }
  }

  handleWindow(metaWindow) {
    // Check if the title contains "Picture-in-Picture"
    if (metaWindow.get_title() === null) return false;

    const titles = this.settings.get_strv("titles");
    let matched = false;
    for (const title of titles) {
      if (
        metaWindow
          .get_title()
          .replace(/\u00A0/g, " ")
          .normalize("NFC")
          .includes(title)
      ) {
        matched = true;
        break;
      }
    }
    if (!matched) return false;

    // Get monitor work area
    const workspace = global.workspace_manager.get_active_workspace();
    const monitorIndex = metaWindow.get_monitor();
    const workArea = workspace.get_work_area_for_monitor(monitorIndex);

    const rectangle = metaWindow.get_frame_rect();
    const diagonal = Math.sqrt(
      rectangle.width * rectangle.width + rectangle.height * rectangle.height
    );
    const coef = this.settings.get_int("diagonal") / diagonal;
    const width = rectangle.width * coef;
    const height = rectangle.height * coef;
    const marginX = this.settings.get_int("margin-x");
    const marginY = this.settings.get_int("margin-y");

    // Calculate window position
    const corner = this.settings.get_int("corner");
    let x, y;
    switch (corner) {
      case 0:
        x = workArea.x + marginX;
        y = workArea.y + marginY;
        break;
      case 1:
        x = workArea.x + workArea.width - width - marginX;
        y = workArea.y + marginY;
        break;
      case 2:
        x = workArea.x + marginX;
        y = workArea.y + workArea.height - height - marginY;
        break;
      default:
        x = workArea.x + workArea.width - width - marginX;
        y = workArea.y + workArea.height - height - marginY;
        break;
    }

    // Move and resize
    metaWindow.move_resize_frame(true, x, y, width, height);

    // Make window always above
    if (this.settings.get_boolean("top")) {
      metaWindow.make_above();
    }

    return true;
  }
}
