import Clutter from 'gi://Clutter';
import St from 'gi://St';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
const CLOCK_SCHEMA = 'org.gnome.desktop.interface';
const CLOCK_FORMAT_KEY = 'clock-format';
/**
 * Returns minutes until a given time (defaults to midnight).
 * @param until - Target time as a string (e.g. "23:00", "07:30"), or undefined for midnight.
 */
function getMinutesUntil(hours = 24, minutes = 0, stopUntilMidnight = false) {
    if (hours === 0)
        hours = 24;
    const now = new Date();
    const target = new Date(now);
    target.setHours(hours, minutes, 0, 0);
    // If target time has already passed today, use tomorrow’s target time
    if (target <= now) {
        if (stopUntilMidnight)
            return 0;
        target.setDate(target.getDate() + 1);
    }
    const diffMs = target.getTime() - now.getTime();
    return Math.ceil(diffMs / 1000 / 60); // round up for human-friendly "minutes left"
}
export default class OneFourFourZero extends Extension {
    _indicator = null;
    _task = null;
    _timeout = null;
    _settings = null;
    _clockSettings = null;
    _menuLabel = null;
    enable() {
        // Load settings.
        this._settings = this.getSettings();
        this._clockSettings = this.getSettings(CLOCK_SCHEMA);
        // Create a panel button
        this._indicator = new PanelMenu.Button(0.0, this.metadata.name, false);
        this._buildMenu();
        // Create the minute counter label
        const text = new St.Label({
            text: this.getMinutes().toString(),
            yAlign: Clutter.ActorAlign.CENTER
        });
        this._indicator.add_child(text);
        // Add the indicator to the panel
        Main.panel.addToStatusArea(this.uuid, this._indicator, 1, this._settings.get_string('panel-position'));
        // Reset indicator if position changed
        this._settings.connect('changed::panel-position', () => {
            // TODO: See if we can reposition the indicator without having to restart the extension like that
            // this failed because the uuid (role) is still active and I cannot re-add the indicator.
            // if (this._indicator) {
            //     this._indicator.container.remove_child(this._indicator);
            //     Main.panel.addToStatusArea(this.uuid, this._indicator, 1, this._settings?.get_string('panel-position'));
            // }
            this.disable();
            this.enable();
        });
        // Recalculate minutes if any of the settings change.
        ['countdown-hours', 'countdown-minutes', 'stop-until-midnight'].forEach(key => {
            this._settings.connect(`changed::${key}`, () => {
                text.set_text(this.getMinutes().toString());
                this._setMenuLabel();
            });
        });
        // Align with the next full minute
        const now = new Date();
        const msUntilNextMinute = (60 - now.getSeconds()) * 1000 - now.getMilliseconds();
        // Create a timer to trigger on the next minute.
        // then we can start an every minute interval task after that.
        this._timeout = setTimeout(() => {
            text.text = this.getMinutes().toString();
            this._timeout = null;
            this._task = setInterval(() => {
                text.text = this.getMinutes().toString();
            }, 60 * 1000); // every minute
        }, msUntilNextMinute);
    }
    getMinutes() {
        if (!this._settings)
            return 0;
        return getMinutesUntil(this._settings.get_int('countdown-hours') || 24, this._settings.get_int('countdown-minutes') || 0, this._settings.get_boolean('stop-until-midnight'));
    }
    disable() {
        this._indicator?.destroy();
        this._indicator = null;
        this._settings = null;
        this._clockSettings = null;
        this._menuLabel = null;
        if (this._timeout) {
            clearTimeout(this._timeout);
            this._timeout = null;
        }
        if (this._task) {
            clearInterval(this._task);
            this._task = null;
        }
    }
    _setMenuLabel() {
        if (!this._settings || !this._clockSettings || !this._menuLabel)
            return;
        const hours = this._settings.get_int('countdown-hours') || 24;
        const minutes = this._settings.get_int('countdown-minutes') || 0;
        const isAmPm = this._clockSettings.get_string(CLOCK_FORMAT_KEY) === '12h';
        const formattedTime = this._formatTime(hours, minutes, isAmPm);
        this._menuLabel.label.text = `...minutes until ${formattedTime}.`;
    }
    _buildMenu() {
        if (!this._indicator)
            return;
        const menu = this._indicator.menu;
        this._menuLabel = new PopupMenu.PopupMenuItem('');
        this._menuLabel.sensitive = false;
        this._setMenuLabel();
        menu.addMenuItem(this._menuLabel);
        menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        menu.addAction('Preferences', () => this.openPreferences());
    }
    _formatTime(hours, minutes, isAmPm) {
        if (isAmPm) {
            const ampm = hours >= 12 && hours !== 24 ? 'PM' : 'AM';
            const hours12 = hours % 12 || 12; // converts 0 to 12
            return `${hours12}:${minutes.toString().padStart(2, '0')} ${ampm}`;
        }
        else {
            return `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}`;
        }
    }
}
