import Gtk from 'gi://Gtk';
import Adw from 'gi://Adw';
import Gio from 'gi://Gio';
import { ExtensionPreferences, gettext as _ } from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';
const CLOCK_SCHEMA = 'org.gnome.desktop.interface';
const CLOCK_FORMAT_KEY = 'clock-format';
function to24Hour(hour12, ampm) {
    let hour24 = hour12 % 12; // ensures 12 becomes 0
    if (ampm.toUpperCase() === "PM")
        hour24 += 12;
    return hour24;
}
function formatTime(hours, minutes, ampm) {
    return `${ampm ? hours : hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}${ampm ? ' ' + ampm : ''}`;
}
export default class OneFourFourZeroPreferences extends ExtensionPreferences {
    _settings;
    _clockSettings;
    async fillPreferencesWindow(window) {
        // Load settings
        this._settings = this.getSettings();
        this._clockSettings = this.getSettings(CLOCK_SCHEMA);
        const page = new Adw.PreferencesPage();
        // Add preference groups
        page.add(this._buildTimeGroup());
        page.add(this._buildDisplayGroup());
        window.add(page);
    }
    /**
     * Settings related to the countdown timer
     */
    _buildTimeGroup() {
        const isAmPm = this._clockSettings?.get_string(CLOCK_FORMAT_KEY) === '12h';
        // Current settings.
        const hours24 = this._settings?.get_int('countdown-hours'); // 0–23
        const minutes = this._settings?.get_int('countdown-minutes'); // 0–59
        const hours12 = hours24 % 12 || 12; // converts 0 to 12
        const ampm = hours24 >= 12 ? 'PM' : 'AM';
        const timeGroup = new Adw.PreferencesGroup();
        const countdownRow = new Adw.ActionRow({ title: 'Countdown to' });
        const box = new Gtk.Box({ orientation: Gtk.Orientation.HORIZONTAL, marginTop: 10, marginBottom: 10, spacing: 6 });
        const hourAdjustment = new Gtk.Adjustment({ lower: 0, upper: isAmPm ? 12 : 23, value: isAmPm ? hours12 : hours24, stepIncrement: 1, pageIncrement: 5, pageSize: 0 });
        const hourSpin = new Gtk.SpinButton({ orientation: Gtk.Orientation.VERTICAL, adjustment: hourAdjustment, digits: 0 });
        const label = new Gtk.Label({ label: ':' });
        const minuteAdjustment = new Gtk.Adjustment({ lower: 0, upper: 59, value: minutes, stepIncrement: 1, pageIncrement: 5, pageSize: 0 });
        const minuteSpin = new Gtk.SpinButton({ orientation: Gtk.Orientation.VERTICAL, adjustment: minuteAdjustment, digits: 0 });
        const ampmButton = new Gtk.Button({ label: ampm, valign: Gtk.Align.CENTER, vexpand: false, hexpand: false });
        box.append(hourSpin);
        box.append(label);
        box.append(minuteSpin);
        if (isAmPm)
            box.append(ampmButton);
        countdownRow.add_suffix(box);
        const stopUntilMidnight = new Adw.SwitchRow({
            title: 'Stop at 0 until midnight',
            subtitle: `After ${formatTime(isAmPm ? hours12 : hours24, minutes, isAmPm ? ampm : '')}, the counter will stay at 0 until midnight.`
        });
        this._settings?.bind('stop-until-midnight', stopUntilMidnight, 'active', Gio.SettingsBindFlags.DEFAULT);
        ampmButton.connect('clicked', () => {
            ampmButton.label = ampmButton.label === 'AM' ? 'PM' : 'AM';
            const hour = hourSpin.get_value_as_int();
            const currentMinutes = minuteSpin.get_value_as_int();
            this._settings?.set_int('countdown-hours', to24Hour(hour, ampmButton.label));
            stopUntilMidnight.set_subtitle(`After ${formatTime(hour, currentMinutes, ampm)}, the counter will stay at 0 until midnight.`);
        });
        hourSpin.connect('value-changed', () => {
            const hour = hourSpin.get_value_as_int();
            const currentAmpm = ampmButton.get_label();
            const currentHours24 = isAmPm && ampm ? to24Hour(hour, ampm) : hour;
            const currentHours12 = hours24 % 12 || 12;
            const currentMinutes = minuteSpin.get_value_as_int();
            this._settings?.set_int('countdown-hours', hours24);
            stopUntilMidnight.set_subtitle(`After ${formatTime(isAmPm ? currentHours12 : currentHours24, currentMinutes, isAmPm && currentAmpm ? currentAmpm : '')}, the counter will stay at 0 until midnight.`);
        });
        minuteSpin.connect('value-changed', () => {
            const hours = hourSpin.get_value_as_int();
            const currentMinutes = minuteSpin.get_value_as_int();
            const currentAmpm = hours24 >= 12 ? 'PM' : 'AM';
            this._settings?.set_int('countdown-minutes', minutes);
            stopUntilMidnight.set_subtitle(`After ${formatTime(hours, currentMinutes, isAmPm ? currentAmpm : '')}, the counter will stay at 0 until midnight.`);
        });
        timeGroup.add(countdownRow);
        timeGroup.add(stopUntilMidnight);
        return timeGroup;
    }
    /**
     * Settings related to how the extension is displayed
     */
    _buildDisplayGroup() {
        const displayGroup = new Adw.PreferencesGroup();
        const alignmentList = new Gtk.StringList();
        const comboItems = [
            { position: 'center', label: _('Center') },
            { position: 'left', label: _('Left') },
            { position: 'right', label: _('Right') }
        ];
        comboItems.forEach(item => alignmentList.append(item.label));
        const combo = new Adw.ComboRow({
            title: _('Position in Panel'),
            model: alignmentList
        });
        const panelPosition = this._settings.get_string('panel-position');
        combo.set_selected(comboItems.findIndex(item => panelPosition === item.position));
        combo.connect('notify::selected', (widget) => {
            this._settings.set_string('panel-position', comboItems[widget.get_selected()].position);
        });
        displayGroup.add(combo);
        return displayGroup;
    }
}
