// extension.js - Tasks in Bottom Panel
// Combines panel positioning from Just Perfection with Tasks in Panel functionality
// Based on Just Perfection by Javad Rahmatzadeh (GPL-3.0-only)
// https://gitlab.gnome.org/jrahmatzadeh/just-perfection

import St from 'gi://St';
import Clutter from 'gi://Clutter';
import Shell from 'gi://Shell';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import {Extension} from 'resource:///org/gnome/shell/extensions/extension.js';

const PANEL_POSITION = {
    TOP: 0,
    BOTTOM: 1,
};

export default class TasksBottomPanelExtension extends Extension {
    enable() {
        this._taskButtons = new Map();
        this._signals = [];
        this._panelPosition = PANEL_POSITION.TOP;
        this._workareasChangedSignal = null;
        this._panelHeightSignal = null;
        
        // Move panel to bottom (using Just Perfection's approach)
        this._panelSetPosition(PANEL_POSITION.BOTTOM);
        
        // Create task container in panel
        this._createTaskContainer();
        
        // Start tracking windows
        this._connectSignals();
        this._updateTasks();
    }
    
    /**
     * Move panel position (from Just Perfection API.js)
     *
     * @param {number} position see PANEL_POSITION
     * @param {boolean} force allow to set even when the current position is the same
     */
    _panelSetPosition(position, force = false) {
        let monitorInfo = this._monitorGetInfo();
        let panelBox = Main.layoutManager.panelBox;

        if (!force && position === this._panelPosition) {
            return;
        }

        if (position === PANEL_POSITION.TOP) {
            this._panelPosition = PANEL_POSITION.TOP;
            if (this._workareasChangedSignal) {
                global.display.disconnect(this._workareasChangedSignal);
                this._workareasChangedSignal = null;
            }
            if (this._panelHeightSignal) {
                panelBox.disconnect(this._panelHeightSignal);
                this._panelHeightSignal = null;
            }
            let topX = (monitorInfo) ? monitorInfo.x : 0;
            let topY = (monitorInfo) ? monitorInfo.y : 0;
            panelBox.set_position(topX, topY);
            this._UIStyleClassRemove('just-perfection-api-bottom-panel');
            this._fixPanelMenuSide(St.Side.TOP);
            return;
        }

        this._panelPosition = PANEL_POSITION.BOTTOM;

        // only change it when a monitor detected
        if (monitorInfo) {
            let BottomX = monitorInfo.x;
            let BottomY = monitorInfo.y + monitorInfo.height - Main.panel.height;

            panelBox.set_position(BottomX, BottomY);
            this._UIStyleClassAdd('just-perfection-api-bottom-panel');
        }

        if (!this._workareasChangedSignal) {
            this._workareasChangedSignal = global.display.connect('workareas-changed', () => {
                this._panelSetPosition(PANEL_POSITION.BOTTOM, true);
            });
        }

        if (!this._panelHeightSignal) {
            this._panelHeightSignal = panelBox.connect('notify::height', () => {
                this._panelSetPosition(PANEL_POSITION.BOTTOM, true);
            });
        }

        this._fixPanelMenuSide(St.Side.BOTTOM);
    }

    /**
     * Get monitor info (from Just Perfection API.js)
     */
    _monitorGetInfo() {
        let pMonitor = Main.layoutManager.primaryMonitor;

        if (!pMonitor) {
            return false;
        }

        return {
            'x': pMonitor.x,
            'y': pMonitor.y,
            'width': pMonitor.width,
            'height': pMonitor.height,
            'geometryScale': pMonitor.geometry_scale,
        };
    }

    /**
     * Fix panel menu opening side based on panel position (from Just Perfection API.js)
     */
    _fixPanelMenuSide(position) {
        let PanelMenuButton = PanelMenu.Button;
        let PanelMenuButtonProto = PanelMenuButton.prototype;

        // Set Instances
        let findPanelMenus = (widget) => {
            if (widget instanceof PanelMenuButton && widget.menu?._boxPointer) {
                widget.menu._boxPointer._userArrowSide = position;
            }
            widget.get_children().forEach(subWidget => {
                findPanelMenus(subWidget);
            });
        };

        let panelBoxes = [
            Main.panel._centerBox,
            Main.panel._rightBox,
            Main.panel._leftBox,
        ];
        panelBoxes.forEach(panelBox => findPanelMenus(panelBox));

        // Set Prototypes
        if (position === St.Side.TOP) {
            if (PanelMenuButtonProto._setMenuOld) {
                PanelMenuButtonProto.setMenu = PanelMenuButtonProto._setMenuOld;
            }
            return;
        }

        if (!PanelMenuButtonProto._setMenuOld) {
            PanelMenuButtonProto._setMenuOld = PanelMenuButtonProto.setMenu;
        }

        PanelMenuButtonProto.setMenu = function (menu) {
            this._setMenuOld(menu);
            if (menu) {
                menu._boxPointer._userArrowSide = position;
            }
        };
    }

    /**
     * Add class name to UI group (from Just Perfection API.js)
     */
    _UIStyleClassAdd(classname) {
        Main.layoutManager.uiGroup.add_style_class_name(classname);
    }

    /**
     * Remove class name from UI group (from Just Perfection API.js)
     */
    _UIStyleClassRemove(classname) {
        Main.layoutManager.uiGroup.remove_style_class_name(classname);
    }

    // ========== Tasks in Panel functionality ==========
    
    _createTaskContainer() {
        this._taskContainer = new St.BoxLayout({
            style_class: 'panel-button',
            x_expand: true,
            y_expand: false,
            x_align: Clutter.ActorAlign.START,
            style: 'spacing: 4px;'
        });
        
        // Add to left side of panel, after activities button
        Main.panel._leftBox.insert_child_at_index(this._taskContainer, 1);
    }
    
    _connectSignals() {
        const windowTracker = Shell.WindowTracker.get_default();
        const display = global.display;
        const workspaceManager = global.workspace_manager;
        
        // Window tracking
        this._addSignal(
            display,
            'window-created',
            (_, window) => {
                this._addSignal(
                    window,
                    'workspace-changed',
                    () => this._updateTasks()
                );
                this._updateTasks();
            }
        );
        
        this._addSignal(
            display,
            'window-demands-attention',
            (_, window) => {
                const button = this._taskButtons.get(window);
                if (button) {
                    button.add_style_class_name('window-attention');
                }
            }
        );
        
        // Workspace changes
        this._addSignal(
            workspaceManager,
            'active-workspace-changed',
            () => this._updateTasks()
        );
        
        this._addSignal(
            windowTracker,
            'tracked-windows-changed',
            () => this._updateTasks()
        );
        
        // Scroll on panel to switch workspaces
        this._addSignal(
            Main.panel,
            'scroll-event',
            (actor, event) => {
                const direction = event.get_scroll_direction();
                const activeIndex = workspaceManager.get_active_workspace_index();
                let newIndex = activeIndex;
                
                if (direction === Clutter.ScrollDirection.UP) {
                    newIndex = Math.max(0, activeIndex - 1);
                } else if (direction === Clutter.ScrollDirection.DOWN) {
                    newIndex = Math.min(
                        workspaceManager.n_workspaces - 1,
                        activeIndex + 1
                    );
                }
                
                if (newIndex !== activeIndex) {
                    workspaceManager.get_workspace_by_index(newIndex).activate(
                        global.get_current_time()
                    );
                }
                
                return Clutter.EVENT_STOP;
            }
        );
    }
    
    _updateTasks() {
        // Clear existing
        this._taskContainer.remove_all_children();
        this._taskButtons.clear();
        
        // Get windows on current workspace only
        const workspace = global.workspace_manager.get_active_workspace();
        const windows = workspace.list_windows().filter(w => 
            !w.is_skip_taskbar() && w.get_window_type() === 0
        );
        
        // Create button for each window
        windows.forEach(window => {
            const button = this._createTaskButton(window);
            this._taskContainer.add_child(button);
            this._taskButtons.set(window, button);
        });
    }
    
    _createTaskButton(window) {
        const app = Shell.WindowTracker.get_default().get_window_app(window);
        const button = new St.Button({
            style_class: 'task-button panel-button',
            x_expand: false,
            y_expand: true
        });
        
        // Icon
        const icon = app ? app.create_icon_texture(24) : new St.Icon({
            icon_name: 'application-x-executable',
            icon_size: 24
        });
        
        button.set_child(icon);
        
        // Highlight if focused
        if (window.has_focus()) {
            button.add_style_pseudo_class('checked');
        }
        
        // Click to activate/minimize
        button.connect('clicked', () => {
            if (window.has_focus()) {
                window.minimize();
            } else {
                window.activate(global.get_current_time());
            }
        });
        
        // Right click for app menu
        button.connect('button-press-event', (actor, event) => {
            if (event.get_button() === 3 && app) {
                const appMenu = app.get_app_menu();
                if (appMenu) {
                    appMenu.popup_at_pointer(event);
                }
                return Clutter.EVENT_STOP;
            }
            return Clutter.EVENT_PROPAGATE;
        });
        
        // Middle click for new window
        button.connect('button-press-event', (actor, event) => {
            if (event.get_button() === 2 && app) {
                app.open_new_window(-1);
                return Clutter.EVENT_STOP;
            }
            return Clutter.EVENT_PROPAGATE;
        });
        
        // Hover to raise window
        button.connect('enter-event', () => {
            window.raise();
        });
        
        // Track focus changes
        const focusId = window.connect('notify::appears-focused', () => {
            if (window.has_focus()) {
                button.add_style_pseudo_class('checked');
            } else {
                button.remove_style_pseudo_class('checked');
            }
        });
        
        button.connect('destroy', () => {
            window.disconnect(focusId);
        });
        
        return button;
    }
    
    _addSignal(obj, signal, callback) {
        const id = obj.connect(signal, callback);
        this._signals.push({ obj, id });
    }
    
    disable() {
        // Disconnect all signals
        this._signals.forEach(({ obj, id }) => {
            obj.disconnect(id);
        });
        this._signals = [];
        
        // Remove task container
        if (this._taskContainer) {
            this._taskContainer.destroy();
            this._taskContainer = null;
        }
        
        // Restore panel position to top
        this._panelSetPosition(PANEL_POSITION.TOP);
        
        // Disconnect panel signals
        if (this._workareasChangedSignal) {
            global.display.disconnect(this._workareasChangedSignal);
            this._workareasChangedSignal = null;
        }
        if (this._panelHeightSignal) {
            Main.layoutManager.panelBox.disconnect(this._panelHeightSignal);
            this._panelHeightSignal = null;
        }
        
        // Clear maps
        this._taskButtons.clear();
    }
}