// src/components/indicator.ts
import St from "gi://St";
import * as PanelMenu from "resource:///org/gnome/shell/ui/panelMenu.js";
import * as PopupMenu from "resource:///org/gnome/shell/ui/popupMenu.js";

// src/lib/icons.ts
import Gio from "gi://Gio";
var ICONS = ["vicinae", "github"];
var Icons = class _Icons {
  static #icons = /* @__PURE__ */ new Map();
  constructor(extPath) {
    for (const name of ICONS) {
      const iconPath = `${extPath}/assets/icons/${name}.svg`;
      const icon = Gio.icon_new_for_string(iconPath);
      _Icons.#icons.set(name, icon);
    }
  }
  static get(name) {
    return _Icons.#icons.get(name);
  }
};

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};
var logger = {
  debug,
  info,
  warn,
  error
};

// src/components/indicator.ts
var VicinaeIndicator = class {
  indicator;
  extension;
  constructor(extension) {
    this.extension = extension;
    this.indicator = new PanelMenu.Button(0, "Vicinae Gnome Extension");
    this.setupUI();
    this.setupMenu();
  }
  setupUI() {
    new Icons(this.extension.path);
    const vicinaeIcon = Icons.get("vicinae");
    const icon = new St.Icon({
      gicon: vicinaeIcon,
      style_class: "system-status-icon"
    });
    this.indicator.add_child(icon);
  }
  setupMenu() {
    const settingsItem = new PopupMenu.PopupMenuItem("Settings");
    settingsItem.connect("activate", () => {
      logger.debug("Opening Vicinae settings");
      this.extension.openPreferences();
    });
    if (this.indicator.menu && "addMenuItem" in this.indicator.menu) {
      this.indicator.menu.addMenuItem(settingsItem);
    }
  }
  getButton() {
    return this.indicator;
  }
  destroy() {
    if (this.indicator) {
      this.indicator.destroy();
    }
  }
};
export {
  VicinaeIndicator
};
