// src/core/clipboard/clipboard-manager.ts
import Clutter from "gi://Clutter";
import GLib from "gi://GLib";
import Meta from "gi://Meta";
import Shell from "gi://Shell";
import St from "gi://St";

// src/utils/window-utils.ts
var getFocusedWindow = () => {
  const windowActors = global.get_window_actors();
  return windowActors.find((w) => w.meta_window.has_focus());
};
var getFocusedWindowApp = () => {
  const focusedWindow = getFocusedWindow();
  if (focusedWindow) {
    const wmClass = focusedWindow.meta_window.get_wm_class();
    const title = focusedWindow.meta_window.get_title();
    return wmClass || title || "unknown";
  }
  return "gnome-shell";
};

// src/utils/clipboard-utils.ts
var CLIPBOARD_CONFIG = {
  MAX_CLIPBOARD_SIZE: 10 * 1024 * 1024
  // 10MB - reasonable clipboard limit
};
function isValidImageBuffer(buffer) {
  if (!buffer) return false;
  let uint8Array;
  if (buffer instanceof ArrayBuffer) {
    if (buffer.byteLength < 8) return false;
    uint8Array = new Uint8Array(buffer);
  } else if (buffer instanceof Uint8Array) {
    if (buffer.length < 8) return false;
    uint8Array = buffer;
  } else if ("length" in buffer && buffer.length >= 8) {
    uint8Array = new Uint8Array(buffer.slice(0, 8));
  } else {
    return false;
  }
  if (uint8Array[0] === 137 && uint8Array[1] === 80 && uint8Array[2] === 78 && uint8Array[3] === 71) {
    return true;
  }
  if (uint8Array[0] === 255 && uint8Array[1] === 216 && uint8Array[2] === 255) {
    return true;
  }
  if (uint8Array[0] === 71 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 56) {
    return true;
  }
  if (uint8Array[0] === 82 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 70) {
    let webpCheck;
    if (buffer instanceof Uint8Array && buffer.length >= 12) {
      webpCheck = buffer.slice(8, 12);
    } else if (buffer instanceof ArrayBuffer && buffer.byteLength >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else if ("length" in buffer && buffer.length >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else {
      return false;
    }
    if (webpCheck[0] === 87 && webpCheck[1] === 69 && webpCheck[2] === 66 && webpCheck[3] === 80) {
      return true;
    }
  }
  return false;
}
function getImageMimeType(buffer) {
  if (!buffer) return "application/octet-stream";
  let uint8Array;
  if (buffer instanceof ArrayBuffer) {
    if (buffer.byteLength < 8) return "application/octet-stream";
    uint8Array = new Uint8Array(buffer.slice(0, 12));
  } else if (buffer instanceof Uint8Array) {
    if (buffer.length < 8) return "application/octet-stream";
    uint8Array = buffer.slice(0, 12);
  } else if ("length" in buffer && buffer.length >= 8) {
    uint8Array = new Uint8Array(buffer.slice(0, 12));
  } else {
    return "application/octet-stream";
  }
  if (uint8Array[0] === 137 && uint8Array[1] === 80 && uint8Array[2] === 78 && uint8Array[3] === 71) {
    return "image/png";
  }
  if (uint8Array[0] === 255 && uint8Array[1] === 216 && uint8Array[2] === 255) {
    return "image/jpeg";
  }
  if (uint8Array[0] === 71 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 56) {
    return "image/gif";
  }
  if (uint8Array[0] === 82 && uint8Array[1] === 73 && uint8Array[2] === 70 && uint8Array[3] === 70) {
    let webpCheck;
    if (buffer instanceof Uint8Array && buffer.length >= 12) {
      webpCheck = buffer.slice(8, 12);
    } else if (buffer instanceof ArrayBuffer && buffer.byteLength >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else if ("length" in buffer && buffer.length >= 12) {
      webpCheck = new Uint8Array(buffer.slice(8, 12));
    } else {
      return "application/octet-stream";
    }
    if (webpCheck[0] === 87 && webpCheck[1] === 69 && webpCheck[2] === 66 && webpCheck[3] === 80) {
      return "image/webp";
    }
  }
  return "application/octet-stream";
}
function calculateClipboardMetadata(event) {
  const content = event.content;
  let mimeType = "text/plain";
  const sourceApp = getFocusedWindowApp();
  if (event.source === "image") {
    if (content.startsWith("data:image/")) {
      const match = content.match(/^data:(image\/[^;]+);/);
      mimeType = match ? match[1] : "image/png";
    }
  } else if (content.startsWith("data:")) {
    const match = content.match(/^data:([^;]+);/);
    mimeType = match ? match[1] : "application/octet-stream";
  } else {
    mimeType = "text/plain";
    if (content.includes("<") && content.includes(">") && (content.includes("<html") || content.includes("<div") || content.includes("<p"))) {
      mimeType = "text/html";
    }
  }
  return {
    mimeType,
    sourceApp
  };
}

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};
var logger = {
  debug,
  info,
  warn,
  error
};

// src/core/clipboard/clipboard-manager.ts
var VicinaeClipboardManager = class {
  eventListeners = [];
  virtualKeyboard;
  currentContent = "";
  clipboard = null;
  selection = null;
  _selectionOwnerChangedId = null;
  _debouncing = 0;
  settings = null;
  pasteHackCallbackId = null;
  constructor(virtualKeyboard) {
    this.virtualKeyboard = virtualKeyboard;
    this.setupClipboardMonitoring();
  }
  setSettings(settings) {
    this.settings = settings;
    logger.info("Settings set in clipboard manager from external source");
    try {
      const blockedApps = this.settings.get_strv("blocked-applications");
      logger.debug(
        `Current blocked applications: [${blockedApps.join(", ")}]`
      );
    } catch (err) {
      logger.error(
        "Error reading blocked applications from settings",
        err
      );
    }
  }
  updateSettings(settings) {
    this.settings = settings;
    logger.info("Settings updated in clipboard manager");
    try {
      const blockedApps = this.settings.get_strv("blocked-applications");
      logger.debug(
        `Updated blocked applications: [${blockedApps.join(", ")}]`
      );
    } catch (err) {
      logger.error(
        "Error reading updated blocked applications from settings",
        err
      );
    }
  }
  isApplicationBlocked(sourceApp) {
    if (!this.settings) {
      logger.warn(
        "No settings available in clipboard manager - blocking logic disabled"
      );
      return false;
    }
    try {
      const blockedApps = this.settings.get_strv("blocked-applications");
      logger.debug(
        `Checking if ${sourceApp} is blocked. Blocked apps list: [${blockedApps.join(
          ", "
        )}]`
      );
      const isBlocked = blockedApps.some(
        (blockedApp) => sourceApp.toLowerCase().includes(blockedApp.toLowerCase()) || blockedApp.toLowerCase().includes(sourceApp.toLowerCase())
      );
      if (isBlocked) {
        logger.debug(
          `Application ${sourceApp} is blocked from clipboard access`
        );
      } else {
        logger.debug(
          `Application ${sourceApp} is NOT blocked (not in blocked apps list)`
        );
      }
      return isBlocked;
    } catch (error2) {
      logger.error(
        "Error checking blocked applications in clipboard manager",
        error2
      );
      return false;
    }
  }
  shouldBlockContentType(contentType, mimeType) {
    return contentType === "text" || mimeType.startsWith("text/");
  }
  setupClipboardMonitoring() {
    try {
      this.clipboard = St.Clipboard.get_default();
      this.selection = Shell.Global.get().get_display().get_selection();
      if (this.selection) {
        this._selectionOwnerChangedId = this.selection.connect(
          "owner-changed",
          this.onSelectionOwnerChanged.bind(this)
        );
        this.queryClipboard();
        logger.info(
          "Clipboard monitoring set up successfully using selection listener"
        );
      } else {
        logger.error("Failed to get selection instance");
      }
    } catch (error2) {
      logger.error("Error setting up clipboard monitoring", error2);
    }
  }
  onSelectionOwnerChanged(_, selectionType) {
    if (selectionType === Meta.SelectionType.SELECTION_CLIPBOARD) {
      this.queryClipboard();
    }
  }
  queryClipboard() {
    if (!this.clipboard) return;
    try {
      this.clipboard.get_text(St.ClipboardType.CLIPBOARD, (_, text) => {
        if (text) {
          this.processClipboardContent(text, "system");
        }
      });
      this.clipboard.get_text(St.ClipboardType.PRIMARY, (_, text) => {
        if (text) {
          this.processClipboardContent(text, "system");
        }
      });
      const mimeTypes = this.clipboard.get_mimetypes(
        St.ClipboardType.CLIPBOARD
      );
      if (mimeTypes.length > 0) {
        if (mimeTypes.some((type) => type.startsWith("image/"))) {
          this.captureImageData();
        }
      }
    } catch (error2) {
      logger.error("Error querying clipboard", error2);
    }
  }
  captureImageData() {
    if (!this.clipboard) return;
    try {
      try {
        this.clipboard.get_content(
          St.ClipboardType.CLIPBOARD,
          "image/png",
          (_, content) => {
            if (content) {
              if (content && typeof content === "object") {
                if (content.constructor && (content.constructor.name === "GLib.Bytes" || content.constructor.name.includes(
                  "GLib.Bytes"
                ) || content.constructor.name.includes(
                  "Bytes"
                ))) {
                  this.extractGLibBytesData(
                    content
                  );
                } else {
                  if (content && typeof content === "object" && "data" in content) {
                    this.processImageContent(
                      content
                    );
                  } else {
                    this.processClipboardContent(
                      "[IMAGE_DATA_AVAILABLE]",
                      "system"
                    );
                  }
                }
              }
            }
          }
        );
      } catch (_contentError) {
        this.processClipboardContent(
          "[IMAGE_DATA_AVAILABLE]",
          "system"
        );
      }
    } catch (error2) {
      logger.error("Error capturing image data", error2);
    }
  }
  processImageContent(content) {
    try {
      if (content.data && content.data.length > 0) {
        const isValid = isValidImageBuffer(content.data);
        const detectedMimeType = getImageMimeType(content.data);
        const finalMimeType = content.mimeType || detectedMimeType;
        if (isValid) {
          logger.debug(
            `Processed image content: ${finalMimeType}, ${content.data.length} bytes`
          );
          const binaryMarker = `[BINARY_IMAGE:${finalMimeType}:${content.data.length}]`;
          this.storeBinaryData(
            binaryMarker,
            content.data,
            finalMimeType
          );
          this.processClipboardContent(binaryMarker, "image");
        } else {
          logger.warn(
            `Invalid image buffer detected: ${content.data.length} bytes`
          );
          this.processClipboardContent(
            "[BINARY_DATA_AVAILABLE]",
            "system"
          );
        }
      } else {
        logger.debug("No image data available in content object");
        this.processClipboardContent(
          "[IMAGE_DATA_AVAILABLE]",
          "system"
        );
      }
    } catch (error2) {
      logger.error("Error processing image content", error2);
      this.processClipboardContent("[IMAGE_DATA_AVAILABLE]", "system");
    }
  }
  binaryDataStore = /* @__PURE__ */ new Map();
  storeBinaryData(marker, data, mimeType) {
    this.binaryDataStore.set(marker, { data, mimeType });
    logger.debug(`Stored binary data: ${marker}`);
  }
  getBinaryData(marker) {
    return this.binaryDataStore.get(marker) || null;
  }
  extractGLibBytesData(bytes) {
    try {
      if (typeof bytes.get_data === "function") {
        try {
          const data = bytes.get_data();
          if (data && data.length > 0) {
            const isValid = isValidImageBuffer(data);
            const mimeType = getImageMimeType(data);
            if (isValid) {
              const binaryMarker = `[BINARY_IMAGE:${mimeType}:${data.length}]`;
              this.storeBinaryData(binaryMarker, data, mimeType);
              logger.debug(
                `Extracted GLib.Bytes data: ${mimeType}, ${data.length} bytes`
              );
              this.processClipboardContent(binaryMarker, "image");
            } else {
              logger.warn(
                `Invalid image buffer from get_data: ${data.length} bytes`
              );
              this.processClipboardContent(
                "[BINARY_DATA_AVAILABLE]",
                "system"
              );
            }
          }
        } catch (getDataError) {
          logger.error("get_data failed", getDataError);
        }
      }
      if (typeof bytes.toArray === "function") {
        try {
          const array = bytes.toArray();
          if (array && array.length > 0) {
            const isValid = isValidImageBuffer(array);
            const mimeType = getImageMimeType(array);
            if (isValid) {
              const binaryMarker = `[BINARY_IMAGE:${mimeType}:${array.length}]`;
              this.storeBinaryData(binaryMarker, array, mimeType);
              logger.debug(
                `Extracted GLib.Bytes array data: ${mimeType}, ${array.length} bytes`
              );
              this.processClipboardContent(binaryMarker, "image");
            }
          }
        } catch (toArrayError) {
          logger.error("to_array failed", toArrayError);
        }
      }
    } catch (error2) {
      logger.error("Error extracting GLib.Bytes data", error2);
    }
  }
  processClipboardContent(text, source) {
    if (this._debouncing > 0) {
      this._debouncing--;
      return;
    }
    if (!text || text === this.currentContent) {
      return;
    }
    this.currentContent = text;
    this.emitClipboardEvent(text, source);
  }
  // Method to emit clipboard change events
  emitClipboardEvent(content, source = "user", mimeType) {
    let contentString;
    let contentType = "text";
    if (content instanceof Uint8Array) {
      if (mimeType?.startsWith("image/")) {
        contentType = "image";
        contentString = `[BINARY_IMAGE:${mimeType}:${content.length}]`;
      } else {
        contentString = `[BINARY_DATA:${mimeType}:${content.length}]`;
      }
    } else {
      contentString = content;
      contentType = source === "image" ? "image" : "text";
    }
    const event = {
      type: "clipboard-changed",
      content: contentString,
      timestamp: Date.now(),
      source,
      contentType
    };
    const metadata = calculateClipboardMetadata(event);
    const isBlocked = this.isApplicationBlocked(metadata.sourceApp);
    const shouldBlock = isBlocked && this.shouldBlockContentType(event.contentType, metadata.mimeType);
    logger.debug("\u{1F3AF} CLIPBOARD EVENT EMITTED", {
      type: event.type,
      content: contentString.length > 100 ? `${contentString.substring(0, 100)}...` : contentString,
      contentLength: content instanceof Uint8Array ? content.length : contentString.length,
      originalContentType: content instanceof Uint8Array ? "binary" : "string",
      timestamp: new Date(event.timestamp).toISOString(),
      source: event.source,
      listeners: this.eventListeners.length,
      mimeType: metadata.mimeType,
      contentType: event.contentType,
      sourceApp: metadata.sourceApp,
      isBlocked,
      shouldBlock,
      note: shouldBlock ? "\u26A0\uFE0F This event will be blocked by clipboard manager" : "\u2705 Event will be processed normally"
    });
    if (shouldBlock) {
      logger.debug(
        `\u{1F6AB} Clipboard access blocked for application: ${metadata.sourceApp} (${event.contentType}) - Event not forwarded to listeners`
      );
      return;
    }
    this.eventListeners.forEach((listener) => {
      try {
        listener(event);
      } catch (error2) {
        logger.error("\u274C Error in clipboard event listener", error2);
      }
    });
  }
  onClipboardChange(listener) {
    this.eventListeners.push(listener);
    logger.debug("\u{1F442} Clipboard change listener added");
  }
  removeClipboardListener(listener) {
    const index = this.eventListeners.indexOf(listener);
    if (index > -1) {
      this.eventListeners.splice(index, 1);
      logger.debug("Clipboard change listener removed");
    }
  }
  getCurrentContent() {
    return this.currentContent;
  }
  setContent(content) {
    if (this.clipboard) {
      try {
        this.clipboard.set_text(St.ClipboardType.CLIPBOARD, content);
        this.clipboard.set_text(St.ClipboardType.PRIMARY, content);
        this.currentContent = content;
        this.emitClipboardEvent(content, "user");
      } catch (error2) {
        logger.error("Error setting clipboard content", error2);
      }
    }
  }
  setContentBinary(data, mimeType) {
    if (this.clipboard) {
      try {
        logger.debug(
          `Setting binary clipboard content: ${mimeType}, ${data.length} bytes`
        );
        this.clipboard.set_content(
          St.ClipboardType.CLIPBOARD,
          mimeType,
          data
        );
        this.clipboard.set_content(
          St.ClipboardType.PRIMARY,
          mimeType,
          data
        );
        if (mimeType.startsWith("text/")) {
          const text = new TextDecoder().decode(data);
          this.clipboard.set_content(
            St.ClipboardType.PRIMARY,
            "text/plain",
            new TextEncoder().encode(text)
          );
        }
        this.emitClipboardEvent(data, "user", mimeType);
        logger.debug(
          `Binary clipboard content set successfully for ${mimeType}`
        );
      } catch (error2) {
        logger.error("Error setting binary clipboard content", error2);
      }
    } else {
      logger.error("Clipboard not available for setContentBinary");
    }
  }
  triggerClipboardChange(content, source = "user") {
    this.emitClipboardEvent(content, source);
  }
  triggerKeyboardPaste() {
    logger.debug("Trigger keyboard paste called");
    this.pasteHackCallbackId = GLib.timeout_add(
      GLib.PRIORITY_DEFAULT,
      1,
      // Just post to the end of the event loop
      () => {
        const SHIFT_L = 42;
        const INSERT = 110;
        const eventTime = Clutter.get_current_event_time() * 1e3;
        this.virtualKeyboard.notify_key(
          eventTime,
          SHIFT_L,
          Clutter.KeyState.PRESSED
        );
        this.virtualKeyboard.notify_key(
          eventTime,
          INSERT,
          Clutter.KeyState.PRESSED
        );
        this.virtualKeyboard.notify_key(
          eventTime,
          INSERT,
          Clutter.KeyState.RELEASED
        );
        this.virtualKeyboard.notify_key(
          eventTime,
          SHIFT_L,
          Clutter.KeyState.RELEASED
        );
        this.pasteHackCallbackId = null;
        return false;
      }
    );
  }
  destroy() {
    if (this.selection && this._selectionOwnerChangedId) {
      try {
        this.selection.disconnect(this._selectionOwnerChangedId);
        this._selectionOwnerChangedId = null;
      } catch (edit_error) {
        logger.error(
          "Error disconnecting selection listener",
          edit_error
        );
      }
    }
    this.eventListeners = [];
    this.currentContent = "";
    this.clipboard = null;
    this.selection = null;
    logger.info("Clipboard manager destroyed");
  }
};
export {
  VicinaeClipboardManager
};
