// src/core/launcher/window-tracker.ts
import GLib from "gi://GLib";

// src/utils/logger.ts
var PROJECT_NAME = "Vicinae";
var LogLevel = /* @__PURE__ */ ((LogLevel2) => {
  LogLevel2[LogLevel2["ERROR"] = 0] = "ERROR";
  LogLevel2[LogLevel2["WARN"] = 1] = "WARN";
  LogLevel2[LogLevel2["INFO"] = 2] = "INFO";
  LogLevel2[LogLevel2["DEBUG"] = 3] = "DEBUG";
  return LogLevel2;
})(LogLevel || {});
var currentLogLevel = 2 /* INFO */;
var log = (level, message, data) => {
  if (level > currentLogLevel) {
    return;
  }
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const levelName = LogLevel[level];
  const prefix = `[${PROJECT_NAME}] ${timestamp} ${levelName}`;
  if (data) {
    console.log(`${prefix}: ${message}`);
    if (typeof data === "object" && data !== null) {
      Object.entries(data).forEach(([key, value]) => {
        console.log(`${prefix}:   ${key}: ${value}`);
      });
    } else {
      console.log(`${prefix}: ${data}`);
    }
  } else {
    console.log(`${prefix}: ${message}`);
  }
};
var debug = (message, data) => {
  log(3 /* DEBUG */, message, data);
};
var info = (message, data) => {
  log(2 /* INFO */, message, data);
};
var warn = (message, data) => {
  log(1 /* WARN */, message, data);
};
var error = (message, error2) => {
  const timestamp = (/* @__PURE__ */ new Date()).toISOString();
  const prefix = `[${PROJECT_NAME}] ${timestamp} ERROR`;
  if (error2) {
    console.error(`${prefix}: ${message}`);
    console.error(`${prefix}: ${String(error2)}`);
  } else {
    console.error(`${prefix}: ${message}`);
  }
};
var logger = {
  debug,
  info,
  warn,
  error
};

// src/core/launcher/window-tracker.ts
var WindowTracker = class {
  constructor(appClass, onWindowTracked, onWindowUntracked) {
    this.appClass = appClass;
    this.onWindowTracked = onWindowTracked;
    this.onWindowUntracked = onWindowUntracked;
  }
  trackedWindows = /* @__PURE__ */ new Set();
  windowCreatedHandler;
  windowDestroySignalIds = /* @__PURE__ */ new Map();
  windowValidators = /* @__PURE__ */ new Map();
  isDestroying = false;
  enable() {
    try {
      this.windowCreatedHandler = global.display.connect(
        "window-created",
        (_display, window) => {
          GLib.idle_add(GLib.PRIORITY_DEFAULT, () => {
            this.handleNewWindow(window);
            return GLib.SOURCE_REMOVE;
          });
        }
      );
      this.scanExistingWindows();
      logger.info("WindowTracker: Window tracking enabled");
    } catch (error2) {
      logger.error(
        "WindowTracker: Error enabling window tracking",
        error2
      );
      throw error2;
    }
  }
  disable() {
    this.isDestroying = true;
    if (this.windowCreatedHandler) {
      global.display.disconnect(this.windowCreatedHandler);
      this.windowCreatedHandler = void 0;
    }
    for (const [windowId, signalId] of this.windowDestroySignalIds) {
      if (signalId) {
        try {
          const window = this.getWindowById(windowId);
          if (window && this.isWindowValid(window)) {
            window.disconnect(signalId);
          }
        } catch (_error) {
          logger.debug(
            `WindowTracker: Signal already disconnected for window ${windowId}`
          );
        }
      }
    }
    this.windowDestroySignalIds.clear();
    this.windowValidators.clear();
    this.trackedWindows.clear();
    this.isDestroying = false;
    logger.info("WindowTracker: Window tracking disabled");
  }
  scanExistingWindows() {
    try {
      const windowActors = global.get_window_actors();
      logger.debug(
        `WindowTracker: Scanning ${windowActors.length} existing windows`
      );
      windowActors.forEach((actor) => {
        if (actor?.meta_window) {
          this.handleNewWindow(actor.meta_window);
        } else {
          logger.debug(
            "WindowTracker: Skipping invalid window actor"
          );
        }
      });
    } catch (error2) {
      logger.error(
        "WindowTracker: Error scanning existing windows",
        error2
      );
    }
  }
  handleNewWindow(window) {
    if (this.isDestroying || !this.isValidWindow(window)) {
      logger.debug(
        "WindowTracker: Invalid window object or destroying, skipping"
      );
      return;
    }
    try {
      const wmClass = window.get_wm_class();
      const title = window.get_title();
      const windowId = window.get_id();
      if (!wmClass || !title || windowId <= 0) {
        logger.debug("WindowTracker: Invalid window properties", {
          wmClass,
          title,
          windowId
        });
        return;
      }
      if (wmClass.toLowerCase().includes(this.appClass.toLowerCase())) {
        if (!this.trackedWindows.has(windowId)) {
          this.trackedWindows.add(windowId);
          const validator = () => this.isWindowValid(window);
          this.windowValidators.set(windowId, validator);
          this.onWindowTracked(windowId);
          this.centerWindow(window);
          let signalId;
          try {
            signalId = window.connect("destroy", () => {
              this.handleWindowDestroyed(window);
            });
          } catch (_error) {
            try {
              signalId = window.connect("unmanaged", () => {
                this.handleWindowDestroyed(window);
              });
            } catch (_unmanagedError) {
              logger.debug(
                `WindowTracker: No suitable destroy signal for window ${windowId}, skipping signal connection`
              );
            }
          }
          if (signalId) {
            this.windowDestroySignalIds.set(windowId, signalId);
          }
          logger.debug(
            `WindowTracker: Tracking new window ${windowId} (${wmClass})`
          );
        }
      }
    } catch (error2) {
      logger.error("WindowTracker: Error handling new window", {
        error: error2 instanceof Error ? error2.message : String(error2),
        stack: error2 instanceof Error ? error2.stack : void 0,
        windowId: window?.get_id?.() || "unknown"
      });
    }
  }
  isValidWindow(window) {
    if (!window) return false;
    if (typeof window.get_wm_class !== "function") return false;
    if (typeof window.get_id !== "function") return false;
    try {
      const windowId = window.get_id();
      return windowId > 0 && windowId !== void 0;
    } catch {
      return false;
    }
  }
  isWindowValid(window) {
    if (!window) return false;
    try {
      const windowActors = global.get_window_actors();
      const stillExists = windowActors.some(
        (actor) => actor.meta_window && actor.meta_window.get_id() === window.get_id()
      );
      if (!stillExists) return false;
      window.get_id();
      window.get_wm_class();
      return true;
    } catch {
      return false;
    }
  }
  handleWindowDestroyed(window) {
    if (this.isDestroying || !this.isValidWindow(window)) return;
    const windowId = window.get_id();
    if (this.trackedWindows.has(windowId)) {
      this.trackedWindows.delete(windowId);
      this.windowValidators.delete(windowId);
      this.onWindowUntracked(windowId);
      this.windowDestroySignalIds.delete(windowId);
      logger.debug(
        `WindowTracker: Untracking destroyed window ${windowId}`
      );
    }
  }
  getWindowById(windowId) {
    try {
      const actors = global.get_window_actors();
      for (const actor of actors) {
        const window = actor.meta_window;
        if (window && window.get_id() === windowId) {
          return window;
        }
      }
    } catch (error2) {
      logger.error(
        `WindowTracker: Error finding window ${windowId}`,
        error2
      );
    }
    return null;
  }
  // Public methods for external access
  getTrackedWindows() {
    return Array.from(this.trackedWindows);
  }
  getTrackedWindowsCount() {
    return this.trackedWindows.size;
  }
  /**
   * Centers a window on the current monitor
   */
  centerWindow(window) {
    if (this.isDestroying || !this.isWindowValid(window)) {
      logger.debug(
        "WindowTracker: Skipping center window - window invalid or destroying"
      );
      return;
    }
    try {
      const { x, y } = this.getCenterPosition(window);
      window.move_frame(true, x, y);
      logger.debug(
        `WindowTracker: Centered window ${window.get_id()} at (${Math.round(
          x
        )}, ${Math.round(y)})`
      );
    } catch (error2) {
      logger.error("WindowTracker: Error centering window", error2);
    }
  }
  /**
   * Gets the center position of a window
   */
  getCenterPosition(window) {
    const monitor = window.get_monitor();
    const display = global.display;
    const monitorGeometry = display.get_monitor_geometry(monitor);
    const frame = window.get_frame_rect();
    const centerX = monitorGeometry.x + (monitorGeometry.width - frame.width) / 2;
    const centerY = monitorGeometry.y + (monitorGeometry.height - frame.height) / 2;
    return { x: centerX, y: centerY };
  }
};
export {
  WindowTracker
};
