/**
 * USD-ARS GNOME Shell Extension
 * 
 * Displays different Argentine Peso (ARS) exchange rates for US Dollar (USD)
 * in the GNOME Shell top panel. The extension rotates through various dollar
 * types (Official, Blue, MEP, CCL, etc.) showing updated rates.
 * 
 * Copyright (C) 2025 Marcos Reuquen
 * Licensed under GPL-3.0 - See LICENSE file
 */

'use strict';

import St from 'gi://St';
import Clutter from 'gi://Clutter';
import Soup from 'gi://Soup';
import GLib from 'gi://GLib';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';

// Constants
const API_URL = 'https://dolarapi.com/v1/dolares';
const API_TIMEOUT = 10; // seconds
const DATA_UPDATE_INTERVAL = 30; // seconds - How often to fetch new data from API
const DISPLAY_ROTATION_INTERVAL = 5; // seconds - How often to rotate between dollar types

// Global state variables
let panelButton = null;
let panelButtonText = null;
let httpSession = null;
let dataUpdateTimerId = null;
let displayRotationTimerId = null;
let currentDollarIndex = 0;
let dollarTypes = [];

/**
 * Fetches dollar exchange rates from the DolarAPI
 * Updates the global dollarTypes array with the response
 */
async function fetchDollarRates() {
    try {
        // Initialize HTTP session if needed
        if (!httpSession) {
            httpSession = new Soup.Session({ timeout: API_TIMEOUT });
        }

        // Create HTTP GET request
        const message = Soup.Message.new_from_encoded_form(
            'GET',
            API_URL,
            Soup.form_encode_hash({})
        );

        // Send async request to API
        await httpSession.send_and_read_async(
            message,
            GLib.PRIORITY_DEFAULT,
            null,
            (_, result) => {
                try {
                    const bytes = httpSession.send_and_read_finish(result);
                    const decoder = new TextDecoder('utf-8');
                    const responseText = decoder.decode(bytes.get_data());

                    // Parse and store dollar types
                    dollarTypes = JSON.parse(responseText);

                    // Update display with new data
                    updateDisplay();

                    // Clean up
                    httpSession.abort();
                } catch (parseError) {
                    console.error(`[USD-ARS] Error parsing API response: ${parseError}`);
                    displayError();
                }
            }
        );
    } catch (error) {
        console.error(`[USD-ARS] Error fetching dollar rates: ${error}`);
        displayError();
        if (httpSession) {
            httpSession.abort();
        }
    }
}

/**
 * Displays an error message in the panel when API request fails
 */
function displayError() {
    if (!panelButtonText) {
        panelButtonText = new St.Label({
            style_class: 'cPanelText',
            text: 'Error loading USD rates',
            y_align: Clutter.ActorAlign.CENTER,
        });
        panelButton.set_child(panelButtonText);
    } else {
        panelButtonText.text = 'Error loading USD rates';
    }
}

/**
 * Updates the panel display with the current dollar type information
 * Shows the name and sell price (venta) of the selected dollar type
 */
function updateDisplay() {
    // Validate data availability
    if (!dollarTypes || dollarTypes.length === 0) {
        return;
    }

    // Get current dollar type and format display text
    const currentDollar = dollarTypes[currentDollarIndex];
    const formattedPrice = currentDollar.venta.toLocaleString('es-AR');
    const displayText = `${currentDollar.nombre}: $${formattedPrice}`;

    // Create label if it doesn't exist
    if (!panelButtonText) {
        panelButtonText = new St.Label({
            style_class: 'cPanelText',
            y_align: Clutter.ActorAlign.CENTER,
        });
        panelButton.set_child(panelButtonText);
    }

    // Update label text
    panelButtonText.text = displayText;
}

/**
 * Rotates to the next dollar type in the list
 * Called periodically by the display rotation timer
 */
function rotateDisplay() {
    if (dollarTypes && dollarTypes.length > 0) {
        currentDollarIndex = (currentDollarIndex + 1) % dollarTypes.length;
        updateDisplay();
    }
}

/**
 * Main extension class
 * Handles extension lifecycle (enable/disable)
 */
export default class UsdArsExtension {
/**
 * Called when the extension is enabled
 * Initializes UI and starts timers
 */
    enable() {
        // Create panel button container
        panelButton = new St.Bin({
            style_class: 'panel-button',
        });

        // Fetch initial data from API
        fetchDollarRates();

        // Add button to center of top panel
        Main.panel._centerBox.insert_child_at_index(panelButton, 0);

        // Set up timer to update data from API every 30 seconds
        dataUpdateTimerId = GLib.timeout_add_seconds(
            GLib.PRIORITY_DEFAULT,
            DATA_UPDATE_INTERVAL,
            () => {
                fetchDollarRates();
                return GLib.SOURCE_CONTINUE;
            }
        );

        // Set up timer to rotate display every 5 seconds
        displayRotationTimerId = GLib.timeout_add_seconds(
            GLib.PRIORITY_DEFAULT,
            DISPLAY_ROTATION_INTERVAL,
            () => {
                rotateDisplay();
                return GLib.SOURCE_CONTINUE;
            }
        );
    }

    /**
     * Called when the extension is disabled
     * Cleans up UI elements and timers
     */
    disable() {
        // Remove button from panel
        if (panelButton) {
            Main.panel._centerBox.remove_child(panelButton);
            panelButton.destroy();
            panelButton = null;
        }

        // Clean up label reference
        panelButtonText = null;

        // Reset state
        dollarTypes = [];
        currentDollarIndex = 0;

        // Remove data update timer
        if (dataUpdateTimerId) {
            GLib.Source.remove(dataUpdateTimerId);
            dataUpdateTimerId = null;
        }

        // Remove display rotation timer
        if (displayRotationTimerId) {
            GLib.Source.remove(displayRotationTimerId);
            displayRotationTimerId = null;
        }

        // Clean up HTTP session
        if (httpSession) {
            httpSession.abort();
            httpSession = null;
        }
    }
}
