import {
  createDBusService,
  exportDBusService,
  unexportDBusService
} from "../../utils/dbus-utils.js";
import { logger } from "../../utils/logger.js";
import { CLIPBOARD_DBUS_IFACE } from "./interfaces/clipboard.js";
import { WINDOWS_DBUS_IFACE } from "./interfaces/windows.js";
import { ClipboardService } from "./services/clipboard-service.js";
import { WindowsService } from "./services/windows-service.js";
class DBusManager {
  clipboardService;
  windowsService;
  clipboardServiceInstance;
  windowsServiceInstance;
  constructor(appClass, extension, clipboardManager) {
    if (!clipboardManager) {
      throw new Error(
        "ClipboardManager instance is required for DBusManager"
      );
    }
    this.clipboardServiceInstance = new ClipboardService(
      clipboardManager,
      extension
    );
    this.windowsServiceInstance = new WindowsService(
      clipboardManager,
      appClass
    );
    this.clipboardService = createDBusService(
      CLIPBOARD_DBUS_IFACE,
      // biome-ignore lint/suspicious/noExplicitAny: we need to cast the instance to any to avoid type errors
      this.clipboardServiceInstance
    );
    this.windowsService = createDBusService(
      WINDOWS_DBUS_IFACE,
      // biome-ignore lint/suspicious/noExplicitAny: we need to cast the instance to any to avoid type errors
      this.windowsServiceInstance
    );
    this.clipboardServiceInstance.setDBusObject(this.clipboardService);
    this.windowsServiceInstance.setDBusObject(this.windowsService);
  }
  exportServices() {
    try {
      exportDBusService(
        this.clipboardService,
        "/org/gnome/Shell/Extensions/Clipboard"
      );
      exportDBusService(
        this.windowsService,
        "/org/gnome/Shell/Extensions/Windows"
      );
      logger.info("D-Bus services exported successfully");
    } catch (_error) {
      logger.error("Failed to export D-Bus services", _error);
      throw _error;
    }
  }
  unexportServices() {
    try {
      this.clipboardServiceInstance.destroy();
      this.windowsServiceInstance.destroy();
      unexportDBusService(this.clipboardService);
      unexportDBusService(this.windowsService);
      logger.info("D-Bus services unexported successfully");
    } catch (_error) {
      logger.error("Failed to unexport D-Bus services", _error);
      throw _error;
    }
  }
  getClipboardService() {
    return this.clipboardServiceInstance;
  }
  getWindowsService() {
    return this.windowsServiceInstance;
  }
}
export {
  DBusManager
};
