import GLib from "gi://GLib";
import { logger } from "../../utils/logger.js";
import { VicinaeWindowManager } from "../windows/window-manager.js";
import { ClickHandler } from "./click-handler.js";
import { FocusTracker } from "./focus-tracker.js";
import { WindowTracker } from "./window-tracker.js";
class LauncherManager {
  windowManager;
  windowTracker;
  focusTracker;
  clickHandler;
  config;
  isEnabled = false;
  trackedWindows = /* @__PURE__ */ new Set();
  enableTimeoutId = null;
  constructor(config, clipboardManager) {
    this.windowManager = new VicinaeWindowManager(
      clipboardManager,
      config.appClass
    );
    this.config = config;
    this.windowTracker = new WindowTracker(
      config.appClass,
      this.handleWindowTracked.bind(this),
      this.handleWindowUntracked.bind(this)
    );
  }
  async enable() {
    if (this.isEnabled) {
      logger.debug("LauncherManager: Already enabled, skipping");
      return;
    }
    try {
      await this.windowTracker.enable();
      if (this.config.autoCloseOnFocusLoss) {
        this.setupFocusTracking();
        this.setupClickHandling();
      }
      this.isEnabled = true;
      logger.info("LauncherManager: Successfully enabled");
    } catch (error) {
      logger.error("LauncherManager: Error during enable", error);
      this.cleanup();
      throw error;
    }
  }
  handleWindowTracked(windowId) {
    this.trackedWindows.add(windowId);
    logger.debug(`LauncherManager: Window ${windowId} is now tracked`);
  }
  handleWindowUntracked(windowId) {
    this.trackedWindows.delete(windowId);
    logger.debug(
      `LauncherManager: Window ${windowId} is no longer tracked`
    );
  }
  disable() {
    logger.info("LauncherManager: Disabling");
    this.isEnabled = false;
    this.cleanup();
  }
  setupFocusTracking() {
    this.focusTracker = new FocusTracker(() => this.handleFocusChange());
    this.focusTracker.enable();
  }
  setupClickHandling() {
    this.clickHandler = new ClickHandler(
      this.windowManager,
      () => this.closeTrackedWindows()
    );
    this.clickHandler.enable();
  }
  handleFocusChange() {
    if (!this.isEnabled) return;
    const focusedWindow = global.display.get_focus_window();
    if (!focusedWindow) {
      this.closeTrackedWindows();
      return;
    }
    const focusedWmClass = focusedWindow.get_wm_class()?.toLowerCase() || "";
    if (!focusedWmClass.includes(this.config.appClass.toLowerCase())) {
      this.closeTrackedWindows();
    }
  }
  closeTrackedWindows() {
    if (this.trackedWindows.size === 0) return;
    logger.debug(
      `LauncherManager: Closing ${this.trackedWindows.size} launcher windows due to focus loss`
    );
    const windowsToClose = Array.from(this.trackedWindows);
    this.trackedWindows.clear();
    windowsToClose.forEach((windowId) => {
      try {
        if (this.isValidWindowId(windowId)) {
          this.windowManager.close(windowId);
          logger.debug(
            `LauncherManager: Successfully closed window ${windowId}`
          );
        } else {
          logger.debug(
            `LauncherManager: Window ${windowId} no longer valid, skipping close`
          );
        }
      } catch (error) {
        logger.error(
          `LauncherManager: Error closing window ${windowId}`,
          error
        );
      }
    });
  }
  isValidWindowId(windowId) {
    if (!windowId || windowId <= 0) return false;
    try {
      const windowActors = global.get_window_actors();
      const windowExists = windowActors.some((actor) => {
        try {
          return actor.meta_window && actor.meta_window.get_id() === windowId;
        } catch {
          return false;
        }
      });
      if (!windowExists) return false;
      const details = this.windowManager.details(windowId);
      return details && details.id === windowId;
    } catch {
      return false;
    }
  }
  cleanup() {
    try {
      if (this.enableTimeoutId) {
        GLib.source_remove(this.enableTimeoutId);
        this.enableTimeoutId = null;
      }
      this.windowTracker.disable();
      this.focusTracker?.disable();
      this.clickHandler?.disable();
      this.windowManager.destroy();
      this.trackedWindows.clear();
    } catch (error) {
      logger.error("LauncherManager: Error during cleanup", error);
    }
  }
  // Recovery method for error situations
  recover() {
    logger.warn("LauncherManager: Attempting recovery from errors");
    try {
      this.cleanup();
      this.isEnabled = false;
      if (this.enableTimeoutId) {
        GLib.source_remove(this.enableTimeoutId);
        this.enableTimeoutId = null;
      }
      this.enableTimeoutId = GLib.timeout_add(
        GLib.PRIORITY_DEFAULT,
        500,
        () => {
          this.enable();
          this.enableTimeoutId = null;
          return false;
        }
      );
      logger.info("LauncherManager: Recovery initiated");
    } catch (error) {
      logger.error("LauncherManager: Recovery failed", error);
    }
  }
  // Public methods for dynamic configuration
  updateConfig(newConfig) {
    const oldConfig = { ...this.config };
    this.config = { ...this.config, ...newConfig };
    logger.debug("LauncherManager: Configuration updated", {
      old: oldConfig,
      new: this.config
    });
    if (this.isEnabled) {
      logger.info("LauncherManager: Re-enabling with new configuration");
      this.disable();
      if (this.enableTimeoutId) {
        GLib.source_remove(this.enableTimeoutId);
        this.enableTimeoutId = null;
      }
      this.enableTimeoutId = GLib.timeout_add(
        GLib.PRIORITY_DEFAULT,
        100,
        () => {
          this.enable();
          this.enableTimeoutId = null;
          return false;
        }
      );
    }
  }
  getTrackedWindows() {
    return Array.from(this.trackedWindows);
  }
  getStatus() {
    return {
      isEnabled: this.isEnabled,
      trackedWindowsCount: this.trackedWindows.size,
      config: this.config,
      hasFocusTracker: !!this.focusTracker,
      hasClickHandler: !!this.clickHandler,
      hasWindowTracker: !!this.windowTracker
    };
  }
  // Force refresh of tracked windows
  refresh() {
    logger.debug("LauncherManager: Refreshing tracked windows");
    this.trackedWindows.clear();
  }
}
export {
  LauncherManager
};
