import GLib from "gi://GLib";
import Meta from "gi://Meta";
import { logger } from "../../utils/logger.js";
import {
  getCurrentTime,
  getWindowById,
  isMaximized
} from "../../utils/window-utils.js";
import { WorkspaceManager } from "./workspace-manager.js";
class VicinaeWindowManager {
  clipboardManager;
  PASTE_DELAY = 100;
  appClass;
  pasteTimeoutId = null;
  constructor(clipboardManager, appClass) {
    this.appClass = appClass;
    this.clipboardManager = clipboardManager;
  }
  /**
   * Checks if the window is a target window.
   *
   * @param wmClass - The window manager class of the window.
   * @returns True if the window is a target window, false otherwise.
   */
  isTargetWindow(wmClass) {
    if (!wmClass) {
      logger.debug("isTargetWindow: No wmClass provided");
      return false;
    }
    return wmClass.toLowerCase().includes(this.appClass.toLowerCase()) || this.appClass.toLowerCase().includes(wmClass.toLowerCase());
  }
  list() {
    const windowActors = global.get_window_actors();
    const workspaceManager = global.workspace_manager;
    const windows = windowActors.map((w) => {
      const metaWindow = w.meta_window;
      const windowWorkspace = metaWindow.get_workspace();
      const frame = metaWindow.get_frame_rect();
      return {
        id: metaWindow.get_id(),
        title: metaWindow.get_title(),
        wm_class: metaWindow.get_wm_class() || "",
        wm_class_instance: metaWindow.get_wm_class_instance() || "",
        pid: metaWindow.get_pid(),
        maximized: isMaximized(metaWindow) !== 0,
        // 0 means not maximized
        display: metaWindow.get_display(),
        frame_type: metaWindow.get_frame_type(),
        window_type: metaWindow.get_window_type(),
        layer: metaWindow.get_layer(),
        monitor: metaWindow.get_monitor(),
        role: metaWindow.get_role(),
        width: frame.width,
        height: frame.height,
        x: frame.x,
        y: frame.y,
        in_current_workspace: metaWindow.located_on_workspace?.(
          workspaceManager.get_active_workspace?.()
        ),
        canclose: metaWindow.can_close(),
        canmaximize: metaWindow.can_maximize(),
        canminimize: metaWindow.can_minimize(),
        canshade: false,
        // can_shade() is not in the type definitions
        moveable: metaWindow.allows_move(),
        resizeable: metaWindow.allows_resize(),
        has_focus: metaWindow.has_focus(),
        workspace: windowWorkspace ? windowWorkspace.index() : -1
      };
    });
    return windows;
  }
  details(winid) {
    const w = getWindowById(winid);
    if (!w) {
      throw new Error("Window not found");
    }
    const metaWindow = w.meta_window;
    const workspaceManager = global.workspace_manager;
    const windowWorkspace = metaWindow.get_workspace();
    const frame = metaWindow.get_frame_rect();
    const win = {
      id: metaWindow.get_id(),
      title: metaWindow.get_title(),
      wm_class: metaWindow.get_wm_class() || "",
      wm_class_instance: metaWindow.get_wm_class_instance() || "",
      pid: metaWindow.get_pid(),
      maximized: isMaximized(metaWindow) !== 0,
      // 0 means not maximized
      display: metaWindow.get_display(),
      frame_type: metaWindow.get_frame_type(),
      window_type: metaWindow.get_window_type(),
      layer: metaWindow.get_layer(),
      monitor: metaWindow.get_monitor(),
      role: metaWindow.get_role(),
      width: frame.width,
      height: frame.height,
      x: frame.x,
      y: frame.y,
      in_current_workspace: metaWindow.located_on_workspace?.(
        workspaceManager.get_active_workspace?.()
      ),
      canclose: metaWindow.can_close(),
      canmaximize: metaWindow.can_maximize(),
      canminimize: metaWindow.can_minimize(),
      canshade: false,
      // can_shade() is not in the type definitions
      moveable: metaWindow.allows_move(),
      resizeable: metaWindow.allows_resize(),
      has_focus: metaWindow.has_focus(),
      workspace: windowWorkspace ? windowWorkspace.index() : -1
    };
    return win;
  }
  getTitle(winid) {
    const w = getWindowById(winid);
    if (w) {
      return w.meta_window.get_title();
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameRect(winid) {
    const w = getWindowById(winid);
    if (w) {
      const frame = w.meta_window.get_frame_rect();
      return {
        x: frame.x,
        y: frame.y,
        width: frame.width,
        height: frame.height
      };
    } else {
      throw new Error("Window not found");
    }
  }
  getFrameBounds(winid) {
    const w = getWindowById(winid);
    if (w) {
      return {
        frame_bounds: w.meta_window.get_frame_bounds()
      };
    } else {
      throw new Error("Window not found");
    }
  }
  moveToWorkspace(winid, workspaceNum) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.change_workspace_by_index(workspaceNum, false);
    } else {
      throw new Error("Window not found");
    }
  }
  moveResize(winid, x, y, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(true, x, y, width, height);
    } else {
      throw new Error("Window not found");
    }
  }
  resize(winid, width, height) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_resize_frame(
        true,
        win.get_x(),
        win.get_y(),
        width,
        height
      );
    } else {
      throw new Error("Window not found");
    }
  }
  move(winid, x, y) {
    const win = getWindowById(winid);
    if (win) {
      if (win.meta_window.maximized_horizontally || win.meta_window.maximized_vertically) {
        win.meta_window.unmaximize(Meta.MaximizeFlags.BOTH);
      }
      win.meta_window.move_frame(true, x, y);
    } else {
      throw new Error("Window not found");
    }
  }
  maximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.maximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  minimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.minimize();
    } else {
      throw new Error("Window not found");
    }
  }
  unmaximize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unmaximize(Meta.MaximizeFlags.BOTH);
    } else {
      throw new Error("Window not found");
    }
  }
  unminimize(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      win.unminimize();
    } else {
      throw new Error("Window not found");
    }
  }
  activate(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      const currentTime = getCurrentTime();
      const workspace = win.get_workspace();
      if (workspace) {
        workspace.activate_with_focus(win, currentTime);
      } else {
        win.activate(currentTime);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  close(winid) {
    const win = getWindowById(winid)?.meta_window;
    if (win) {
      try {
        if (win.get_id() === winid) {
          win.delete(getCurrentTime());
        } else {
          throw new Error(
            "Window ID mismatch - window may be destroyed"
          );
        }
      } catch (error) {
        throw new Error(`Failed to close window ${winid}: ${error}`);
      }
    } else {
      throw new Error("Window not found");
    }
  }
  listWorkspaces() {
    const workspaceManager = new WorkspaceManager();
    return workspaceManager.getAllWorkspaces();
  }
  getActiveWorkspace() {
    const workspaceManager = new WorkspaceManager();
    const currentIndex = workspaceManager.getCurrentWorkspaceIndex();
    const workspace = workspaceManager.getWorkspaceInfo(currentIndex);
    if (!workspace) {
      throw new Error("No active workspace found");
    }
    return workspace;
  }
  pasteToFocusedWindow(winid, key, modifiers) {
    if (!this.clipboardManager) {
      logger.error("No clipboard manager available for paste operation");
      return false;
    }
    if (this.pasteTimeoutId) {
      GLib.source_remove(this.pasteTimeoutId);
      this.pasteTimeoutId = null;
    }
    this.pasteTimeoutId = GLib.timeout_add(
      GLib.PRIORITY_DEFAULT,
      this.PASTE_DELAY,
      () => {
        logger.debug(
          `Sending keyboard paste for window ${winid} ${key} ${modifiers} after ${this.PASTE_DELAY}ms delay`
        );
        this.clipboardManager.triggerKeyboardPaste();
        this.pasteTimeoutId = null;
        return false;
      }
    );
    logger.debug(
      `Triggered keyboard paste for window ${winid} ${key} ${modifiers}`
    );
    return true;
  }
  sendShortcut(winid, key, modifiers) {
    this.activate(winid);
    const modifiersArray = modifiers.split("|");
    const isNormalPaste = key === "v" && modifiersArray.length === 1 && modifiersArray[0] === "CONTROL";
    const isShiftPaste = key === "v" && modifiersArray.length === 2 && modifiersArray[0] === "CONTROL" && modifiersArray[1] === "SHIFT";
    if (isNormalPaste || isShiftPaste) {
      return this.pasteToFocusedWindow(winid, key, modifiers);
    } else {
      logger.warn(`Unhandled shortcut: ${key} ${modifiers}`);
      return false;
    }
  }
  destroy() {
    if (this.pasteTimeoutId) {
      GLib.source_remove(this.pasteTimeoutId);
      this.pasteTimeoutId = null;
    }
    logger.debug("Window manager destroyed");
  }
}
export {
  VicinaeWindowManager
};
