import Clutter from "gi://Clutter";
import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";
import * as Main from "resource:///org/gnome/shell/ui/main.js";
import { VicinaeIndicator } from "./components/indicator.js";
import { VicinaeClipboardManager } from "./core/clipboard/clipboard-manager.js";
import { DBusManager } from "./core/dbus/manager.js";
import { LauncherManager } from "./core/launcher/launcher-manager.js";
import { initializeLogger, logger } from "./utils/logger.js";
const getVirtualKeyboard = /* @__PURE__ */ (() => {
  let virtualKeyboard;
  return () => {
    if (!virtualKeyboard) {
      virtualKeyboard = Clutter.get_default_backend().get_default_seat().create_virtual_device(Clutter.InputDeviceType.KEYBOARD_DEVICE);
    }
    return virtualKeyboard;
  };
})();
class Vicinae extends Extension {
  indicator;
  dbusManager;
  clipboardManager;
  launcherManager;
  settings;
  settingsConnection;
  launcherSettingsConnection;
  async enable() {
    logger.info("Vicinae extension enabled");
    this.settings = this.getSettings();
    initializeLogger(this.settings);
    this.clipboardManager = new VicinaeClipboardManager(
      getVirtualKeyboard()
    );
    this.clipboardManager.setSettings(this.settings);
    const appClass = this.settings.get_string("launcher-app-class") || "vicinae";
    this.dbusManager = new DBusManager(
      appClass,
      this,
      this.clipboardManager
    );
    this.dbusManager.exportServices();
    logger.debug("Extension: Initializing launcher manager...");
    await this.initializeLauncherManager();
    this.updateIndicatorVisibility();
    this.settingsConnection = this.settings.connect(
      "changed::show-status-indicator",
      () => {
        this.updateIndicatorVisibility();
      }
    );
    this.launcherSettingsConnection = this.settings.connect(
      "changed::launcher-auto-close-focus-loss",
      () => {
        this.updateLauncherManager();
      }
    );
    this.settings.connect("changed::blocked-applications", () => {
      if (this.clipboardManager && this.settings) {
        this.clipboardManager.updateSettings(this.settings);
        logger.debug(
          "Updated clipboard manager with new blocked applications list"
        );
      }
    });
    logger.info("Vicinae extension initialized successfully");
  }
  updateIndicatorVisibility() {
    const shouldShow = this.settings?.get_boolean("show-status-indicator");
    if (shouldShow && !this.indicator) {
      this.indicator = new VicinaeIndicator(this);
      Main.panel.addToStatusArea(
        "vicinae-gnome-extension",
        this.indicator.getButton(),
        0,
        "right"
      );
      logger.debug("Vicinae indicator shown");
    } else if (!shouldShow && this.indicator) {
      this.indicator.destroy();
      this.indicator = null;
      logger.debug("Vicinae indicator hidden");
    }
  }
  async initializeLauncherManager() {
    if (!this.settings) return;
    const autoClose = this.settings.get_boolean(
      "launcher-auto-close-focus-loss"
    );
    const appClass = this.settings.get_string("launcher-app-class") || "vicinae";
    if (autoClose) {
      if (!this.clipboardManager) {
        throw new Error("Clipboard manager is not initialized");
      }
      this.launcherManager = new LauncherManager(
        {
          appClass,
          autoCloseOnFocusLoss: autoClose
        },
        this.clipboardManager
      );
      await this.launcherManager.enable();
      logger.info("Launcher manager initialized and enabled");
    }
  }
  async updateLauncherManager() {
    if (!this.settings) return;
    const autoClose = this.settings.get_boolean(
      "launcher-auto-close-focus-loss"
    );
    if (autoClose && !this.launcherManager) {
      await this.initializeLauncherManager();
    } else if (!autoClose && this.launcherManager) {
      this.launcherManager.disable();
      this.launcherManager = null;
      logger.debug("Launcher manager disabled");
    } else if (autoClose && this.launcherManager) {
      const appClass = this.settings.get_string("launcher-app-class") || "vicinae";
      logger.debug(
        "initializeShellIntegrationManager: Using app class",
        appClass
      );
      this.launcherManager.updateConfig({
        appClass,
        autoCloseOnFocusLoss: autoClose
      });
    }
  }
  disable() {
    logger.info("Vicinae extension disabled");
    if (this.settingsConnection) {
      this.settings?.disconnect(this.settingsConnection);
      this.settingsConnection = 0;
    }
    if (this.launcherSettingsConnection) {
      this.settings?.disconnect(this.launcherSettingsConnection);
      this.launcherSettingsConnection = 0;
    }
    if (this.launcherManager) {
      this.launcherManager.disable();
      this.launcherManager = null;
    }
    if (this.indicator) {
      this.indicator.destroy();
      this.indicator = null;
    }
    if (this.dbusManager) {
      this.dbusManager.unexportServices();
      this.dbusManager = null;
    }
    if (this.clipboardManager) {
      this.clipboardManager.destroy();
      this.clipboardManager = null;
    }
    this.settings = null;
    logger.info("Vicinae extension cleaned up successfully");
  }
}
export {
  Vicinae as default
};
