import GLib from 'gi://GLib';
import Gio from 'gi://Gio';
import '../node_modules/@girs/gnome-shell/dist/ui/main.js';
import GObject from 'gi://GObject';
import { SETTINGS_KEYS } from '../utils/constants.js';
import { notifyError } from 'resource:///org/gnome/shell/ui/main.js';

class Player extends GObject.Object {
    _settings;
    static {
        GObject.registerClass({
            Signals: {
                'play-state-changed': { param_types: [GObject.TYPE_BOOLEAN] },
                'playback-stopped': { param_types: [] },
            },
        }, this);
    }
    _mpvSocket = '/tmp/quicklofi-socket';
    _isCommandRunning = false;
    _proc = null;
    _keepReading = true;
    _stdoutStream = null;
    _cancellable = null;
    constructor(_settings) {
        super();
        this._settings = _settings;
    }
    initVolumeControl() {
        this._settings.connect(`changed::${SETTINGS_KEYS.VOLUME}`, (settings, key) => {
            if (this._proc !== null && !this._isCommandRunning) {
                const volume = settings.get_int(key);
                const command = this.createCommand({
                    command: ['set_property', 'volume', volume],
                });
                this.sendCommandToMpvSocket(command);
            }
        });
    }
    async stopPlayer() {
        this._keepReading = false;
        if (this._cancellable) {
            this._cancellable.cancel();
            this._cancellable = null;
        }
        if (this._stdoutStream) {
            try {
                this._stdoutStream.close(null);
            }
            catch (e) { }
            this._stdoutStream = null;
        }
        if (this._proc) {
            const proc = this._proc;
            this._proc = null;
            return new Promise((resolve) => {
                proc.wait_check_async(null, (p, res) => {
                    try {
                        p.wait_check_finish(res);
                    }
                    catch (e) { }
                    GLib.unlink(this._mpvSocket);
                    this.emit('playback-stopped');
                    resolve();
                });
                try {
                    proc.force_exit();
                }
                catch (e) { }
            });
            return;
        }
    }
    isPlaying() {
        return this._proc !== null;
    }
    playPause() {
        const playPauseCommand = this.createCommand({ command: ['cycle', 'pause'] });
        this.sendCommandToMpvSocket(playPauseCommand);
        const result = this.getProperty('pause');
        if (result) {
            const isPaused = result.data;
            this.emit('play-state-changed', isPaused);
        }
    }
    getProperty(prop) {
        if (this._proc) {
            const command = this.createCommand({ command: ['get_property', prop] });
            const output = this.sendCommandToMpvSocket(command);
            return JSON.parse(output) ?? null;
        }
    }
    _monitorPlayerOutput({ stream, onLine, }) {
        this._cancellable = new Gio.Cancellable();
        const readLine = () => {
            if (!this._keepReading || this._proc === null)
                return;
            stream.read_line_async(GLib.PRIORITY_DEFAULT, this._cancellable, (s, res) => {
                if (this._cancellable?.is_cancelled())
                    return;
                let line = null;
                try {
                    [line] = s.read_line_finish_utf8(res);
                }
                catch (e) {
                    return;
                }
                if (line !== null) {
                    const keep = onLine(line);
                    if (keep === false) {
                        this._keepReading = false;
                        return;
                    }
                }
                if (this._keepReading)
                    readLine();
            });
        };
        readLine();
    }
    async startPlayer(radio) {
        await this.stopPlayer();
        const MPV_OPTIONS = [
            `--volume=${this._settings.get_int(SETTINGS_KEYS.VOLUME)}`,
            '--demuxer-lavf-o=extension_picky=0',
            `--input-ipc-server=${this._mpvSocket}`,
            '--loop-playlist=force',
            '--no-video',
            '--ytdl-format=best*[vcodec=none]',
            '--ytdl-raw-options-add=force-ipv4=',
            '--msg-level=all=warn',
            `"${radio.radioUrl}"`,
        ];
        try {
            this._keepReading = true;
            const [, argv] = GLib.shell_parse_argv(`mpv ${MPV_OPTIONS.join(' ')}`);
            this._proc = Gio.Subprocess.new(argv, Gio.SubprocessFlags.STDOUT_PIPE | Gio.SubprocessFlags.STDERR_PIPE);
            this._stdoutStream = new Gio.DataInputStream({
                base_stream: this._proc.get_stdout_pipe(),
            });
            this._monitorPlayerOutput({
                stream: this._stdoutStream,
                onLine: (line) => {
                    if (line.trim().startsWith('Failed')) {
                        this.stopPlayer();
                        notifyError(`Error while playing: ${radio.radioName}`, line.trim());
                        return false;
                    }
                },
            });
        }
        catch (e) {
            this._keepReading = false;
            this.stopPlayer();
            notifyError('MPV not found', 'Did you have mpv installed?\nhttps://github.com/EuCaue/gnome-shell-extension-quick-lofi?tab=readme-ov-file#dependencies');
        }
    }
    createCommand(command) {
        return JSON.stringify(command) + '\n';
    }
    sendCommandToMpvSocket(mpvCommand) {
        let response = null;
        if (this._isCommandRunning) {
            return null;
        }
        this._isCommandRunning = true;
        try {
            const address = Gio.UnixSocketAddress.new(this._mpvSocket);
            const client = new Gio.SocketClient();
            const connection = client.connect(address, null);
            const outputStream = connection.get_output_stream();
            const inputStream = connection.get_input_stream();
            const byteArray = new TextEncoder().encode(mpvCommand);
            outputStream.write(byteArray, null);
            outputStream.flush(null);
            const dataInputStream = new Gio.DataInputStream({ base_stream: inputStream });
            const [res] = dataInputStream.read_line_utf8(null);
            response = res;
            outputStream.close(null);
            inputStream.close(null);
            connection.close(null);
        }
        catch (e) {
            notifyError('Error while connecting to the MPV SOCKET', e.message);
        }
        this._isCommandRunning = false;
        return response;
    }
}

export { Player as default };
