import Adw from 'gi://Adw';
import Gio from 'gi://Gio';
import Gtk4 from 'gi://Gtk';
import Gdk from 'gi://Gdk';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import '../node_modules/@girs/gnome-shell/dist/extensions/prefs.js';
import { SETTINGS_KEYS, ffmpegFormats } from '../utils/constants.js';
import { handleErrorRow, generateNanoIdWithSymbols } from '../utils/helpers.js';
import { gettext } from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';

class RadiosPage extends Adw.PreferencesPage {
    _settings;
    _window;
    _radios = [];
    static {
        GObject.registerClass({
            GTypeName: 'RadiosPage',
            Template: 'resource:///org/gnome/Shell/Extensions/quick-lofi/preferences/RadiosPage.ui',
            InternalChildren: ['radiosGroup', 'nameRadioRow', 'urlRadioRow'],
        }, this);
    }
    _updateRadio(index, field, content) {
        if (index !== -1) {
            const radio = this._radios[index];
            const [radioName, radioUrl, radioID] = radio.split(' - ');
            if (field === 'radioUrl') {
                this._radios[index] = `${radioName} - ${content} - ${radioID}`;
            }
            if (field === 'radioName') {
                this._radios[index] = `${content} - ${radioUrl} - ${radioID}`;
            }
            this._settings.set_strv(SETTINGS_KEYS.RADIOS_LIST, this._radios);
            return true;
        }
        return false;
    }
    _removeRadio(index, radioID) {
        this._radios.splice(index, 1);
        if (radioID === this._settings.get_string(SETTINGS_KEYS.CURRENT_RADIO_PLAYING)) {
            this._settings.set_string(SETTINGS_KEYS.CURRENT_RADIO_PLAYING, '');
        }
        this._settings.set_strv(SETTINGS_KEYS.RADIOS_LIST, this._radios);
    }
    _populateRadios(radiosGroup) {
        const listBox = radiosGroup.get_last_child().get_last_child().get_first_child();
        const dropTarget = Gtk4.DropTarget.new(Gtk4.ListBoxRow.$gtype, Gdk.DragAction.MOVE);
        let dragIndex = -1;
        listBox.add_controller(dropTarget);
        for (let i = 0; i < this._radios.length; i++) {
            const [radioName, radioUrl, radioID] = this._radios[i].split(' - ');
            const radiosExpander = new Adw.ExpanderRow({
                title: gettext(radioName),
                cursor: new Gdk.Cursor({ name: 'pointer' }),
            });
            const nameRadioRow = new Adw.EntryRow({
                title: gettext('Radio Name'),
                text: gettext(radioName),
                showApplyButton: true,
            });
            const urlRadioRow = new Adw.EntryRow({
                title: gettext('Radio URL'),
                text: gettext(radioUrl),
                showApplyButton: true,
                inputPurpose: Gtk4.InputPurpose.URL,
            });
            const removeButton = new Gtk4.Button({
                label: `Remove ${radioName}`,
                iconName: 'user-trash-symbolic',
                cursor: new Gdk.Cursor({ name: 'pointer' }),
                halign: Gtk4.Align.CENTER,
                valign: Gtk4.Align.CENTER,
            });
            removeButton.connect('clicked', () => {
                const dialog = new Adw.AlertDialog({
                    heading: gettext(`Are you sure you want to delete ${radioName} ?`),
                    closeResponse: 'cancel',
                });
                dialog.add_response('cancel', 'Cancel');
                dialog.add_response('ok', 'Ok');
                dialog.set_response_appearance('ok', Adw.ResponseAppearance.DESTRUCTIVE);
                dialog.choose(this._window, null, () => { });
                dialog.connect('response', (dialog, response) => {
                    if (response === 'ok') {
                        this._removeRadio(i, radioID);
                        this._reloadRadios(radiosGroup);
                    }
                    dialog.close();
                });
            });
            nameRadioRow.connect('apply', (w) => {
                const index = this._radios.findIndex((entry) => entry.endsWith(radioID));
                if (w.text.length < 2) {
                    handleErrorRow(w, 'Name must be at least 2 characters');
                    const originalRadioName = this._radios[index].split('-')[0].trim();
                    w.set_text(originalRadioName);
                    return;
                }
                this._updateRadio(index, 'radioName', w.text);
                radiosExpander.set_title(w.text);
            });
            urlRadioRow.connect('apply', (w) => {
                const index = this._radios.findIndex((entry) => entry.endsWith(radioID));
                if (this._isPlayable({ uri: w.text }) === false) {
                    handleErrorRow(urlRadioRow, 'Invalid URL or PATH.');
                    const originalRadioUrl = this._radios[index].split('-')[1].trim();
                    w.set_text(originalRadioUrl);
                    return;
                }
                this._updateRadio(index, 'radioUrl', w.text);
            });
            radiosExpander.add_row(nameRadioRow);
            radiosExpander.add_row(urlRadioRow);
            radiosExpander.add_row(removeButton);
            let dragX;
            let dragY;
            const dropController = new Gtk4.DropControllerMotion();
            const dragSource = new Gtk4.DragSource({
                actions: Gdk.DragAction.MOVE,
            });
            radiosExpander.add_controller(dragSource);
            radiosExpander.add_controller(dropController);
            dragSource.connect('prepare', (_source, x, y) => {
                dragX = x;
                dragY = y;
                const value = new GObject.Value();
                value.init(Gtk4.ListBoxRow);
                value.set_object(radiosExpander);
                dragIndex = radiosExpander.get_index();
                return Gdk.ContentProvider.new_for_value(value);
            });
            dragSource.connect('drag-begin', (_source, drag) => {
                const dragWidget = new Gtk4.ListBox();
                dragWidget.set_size_request(radiosExpander.get_width(), radiosExpander.get_height());
                dragWidget.add_css_class('boxed-list');
                const dragRow = new Adw.ActionRow({ title: radiosExpander.title });
                dragRow.add_prefix(new Gtk4.Image({
                    icon_name: 'list-drag-handle-symbolic',
                    css_classes: ['dim-label'],
                }));
                dragWidget.append(dragRow);
                dragWidget.drag_highlight_row(dragRow);
                const icon = Gtk4.DragIcon.get_for_drag(drag);
                icon.child = dragWidget;
                drag.set_hotspot(dragX, dragY);
            });
            dropController.connect('enter', () => {
                listBox.drag_highlight_row(radiosExpander);
            });
            dropController.connect('leave', () => {
                listBox.drag_unhighlight_row();
            });
            radiosGroup.add(radiosExpander);
        }
        dropTarget.connect('drop', (_drop, dragedExpanderRow, _x, y) => {
            const targetRow = listBox.get_row_at_y(y);
            const targetIndex = targetRow.get_index();
            if (!dragedExpanderRow || !targetRow) {
                return false;
            }
            const [movedRadio] = this._radios.splice(dragIndex, 1);
            this._radios.splice(targetIndex, 0, movedRadio);
            targetRow.set_state_flags(Gtk4.StateFlags.NORMAL, true);
            listBox.remove(dragedExpanderRow);
            listBox.insert(dragedExpanderRow, targetIndex);
            this._settings.set_strv(SETTINGS_KEYS.RADIOS_LIST, this._radios);
            return true;
        });
    }
    _reloadRadios(radiosGroup) {
        for (let i = 0; i <= this._radios.length; i++) {
            const child = radiosGroup
                .get_first_child()
                .get_first_child()
                .get_next_sibling()
                .get_first_child()
                .get_first_child();
            if (child === null)
                break;
            radiosGroup.remove(child);
        }
        this._populateRadios(radiosGroup);
    }
    _addRadio(radioName, radioUrl) {
        const radioID = generateNanoIdWithSymbols(10);
        this._radios.push(`${radioName} - ${radioUrl} - ${radioID}`);
        this._settings.set_strv(SETTINGS_KEYS.RADIOS_LIST, this._radios);
    }
    _isPlayable({ uri }) {
        if (uri.trim() === '')
            return false;
        try {
            const isValidUri = GLib.uri_is_valid(uri, GLib.UriFlags.NONE);
            if (isValidUri) {
                return true;
            }
        }
        catch (e) { }
        let path = uri;
        if (path.startsWith('~')) {
            path = GLib.get_home_dir() + path.slice(1);
        }
        const filepath = Gio.File.new_for_path(path);
        if (!filepath)
            return false;
        const basename = filepath.get_basename().split('.');
        const ext = basename[basename.length - 1];
        const filepathUri = filepath.get_uri();
        try {
            const fileUri = Gio.File.new_for_uri(filepathUri);
            const fileInfo = fileUri.query_info('standard::*,access::*', Gio.FileQueryInfoFlags.NONE, null);
            if (!ffmpegFormats.has(ext) && !fileInfo.get_content_type().match(/^video|^audio/)) {
                return false;
            }
            const fileType = fileInfo.get_file_type();
            const isFileReadable = fileInfo.get_attribute_boolean('access::can-read');
            if (isFileReadable &&
                (fileType === Gio.FileType.REGULAR ||
                    fileType === Gio.FileType.SYMBOLIC_LINK ||
                    fileType === Gio.FileType.SPECIAL)) {
                return true;
            }
        }
        catch (e) {
            return false;
        }
        return false;
    }
    _handleAddRadio() {
        if (this._nameRadioRow.text.length < 2) {
            handleErrorRow(this._nameRadioRow, 'Name must be at least 2 characters');
            return;
        }
        if (this._urlRadioRow.text.length <= 0) {
            handleErrorRow(this._urlRadioRow, 'URL cannot be empty.');
            return;
        }
        if (this._isPlayable({ uri: this._urlRadioRow.text }) === false) {
            handleErrorRow(this._urlRadioRow, 'Invalid URL or PATH.');
            return;
        }
        this._addRadio(this._nameRadioRow.text, this._urlRadioRow.text);
        this._nameRadioRow.set_text('');
        this._urlRadioRow.set_text('');
        this._reloadRadios(this._radiosGroup);
    }
    _enableAddRadioOnEnter() {
        const controllerCallback = (_source, keyVal, _keyCode, _state) => {
            if (keyVal === Gdk.KEY_Return || keyVal === Gdk.KEY_KP_Enter) {
                this._handleAddRadio();
            }
            return Gdk.EVENT_PROPAGATE;
        };
        const nameRadioController = new Gtk4.EventControllerKey({
            propagationPhase: Gtk4.PropagationPhase.CAPTURE,
        });
        const urlRadioController = new Gtk4.EventControllerKey({
            propagationPhase: Gtk4.PropagationPhase.CAPTURE,
        });
        nameRadioController.connect('key-pressed', controllerCallback);
        urlRadioController.connect('key-pressed', controllerCallback);
        this._nameRadioRow.add_controller(nameRadioController);
        this._urlRadioRow.add_controller(urlRadioController);
    }
    constructor(_settings, _window) {
        super();
        this._settings = _settings;
        this._window = _window;
        this._radios = this._settings.get_strv(SETTINGS_KEYS.RADIOS_LIST);
        this._populateRadios(this._radiosGroup);
        this._enableAddRadioOnEnter();
        this._window.connect('close-request', () => {
            this._settings = null;
            this._radios = null;
        });
    }
}

export { RadiosPage };
