import Clutter from "gi://Clutter";
import GLib from "gi://GLib";
import { MainPanel } from "../types/index.js";
import { logger } from "../utils/logger.js";
class PanelManager {
  settings;
  veilIndicator;
  addedHandlerId = null;
  removedHandlerId = null;
  onItemsChangedCallback;
  activeAnimations = /* @__PURE__ */ new Map();
  timeoutIds = /* @__PURE__ */ new Map();
  constructor(settings, veilIndicator) {
    this.settings = settings;
    this.veilIndicator = veilIndicator;
    this.setupListeners();
    this.updateAllItemsList();
  }
  setupListeners() {
    this.addedHandlerId = MainPanel._rightBox.connect(
      "child-added",
      this._onItemAdded.bind(this)
    );
    this.removedHandlerId = MainPanel._rightBox.connect(
      "child-removed",
      this._onItemRemoved.bind(this)
    );
    logger.debug("Panel listeners setup complete");
  }
  _onItemAdded(_container, actor) {
    logger.debug("Panel item added", { actor });
    this.updateAllItemsList();
    this.onItemsChangedCallback?.(this.getAllItemNames());
  }
  _onItemRemoved(_container, actor) {
    logger.debug("Panel item removed", { actor });
    this.updateAllItemsList();
    this.onItemsChangedCallback?.(this.getAllItemNames());
  }
  updateAllItemsList() {
    const itemNames = this.getAllItemNames();
    this.settings.set_strv("all-items", itemNames);
    logger.debug("Updated all-items list", { count: itemNames.length });
  }
  getAllItemNames() {
    const rightBoxItems = MainPanel._rightBox.get_children();
    const itemNames = [];
    rightBoxItems.forEach(
      (item) => {
        const child = item.firstChild;
        if (!child) return;
        if (child === MainPanel.statusArea.quickSettings || child === this.veilIndicator) {
          return;
        }
        const name = this.getItemName(child);
        if (name) {
          itemNames.push(name);
        }
      }
    );
    return itemNames;
  }
  getAllPanelItems() {
    const rightBoxItems = MainPanel._rightBox.get_children();
    const items = [];
    rightBoxItems.forEach(
      (item) => {
        const child = item.firstChild;
        if (!child) return;
        if (child === MainPanel.statusArea.quickSettings || child === this.veilIndicator) {
          return;
        }
        const name = this.getItemName(child);
        if (name) {
          items.push({
            name,
            actor: child,
            container: item
          });
        }
      }
    );
    return items;
  }
  getItemName(item) {
    if (item.accessible_name && item.accessible_name !== "") {
      return item.accessible_name;
    }
    if (item.constructor && "name" in item.constructor) {
      return item.constructor.name;
    }
    return null;
  }
  isItemVisible(item) {
    const visibleItems = this.settings.get_strv("visible-items");
    return visibleItems.includes(item.name);
  }
  setVisibility(visible) {
    const panelItems = this.getAllPanelItems();
    const visibleItems = this.settings.get_strv("visible-items");
    const animationEnabled = this.settings.get_boolean("animation-enabled");
    panelItems.forEach((item) => {
      const shouldBeVisible = visibleItems.includes(item.name);
      const targetVisibility = visible ? true : shouldBeVisible;
      if (animationEnabled) {
        if (targetVisibility) {
          this.fadeIn(item.container);
        } else {
          this.fadeOut(item.container);
        }
      } else {
        item.container.visible = targetVisibility;
        item.container.opacity = 255;
      }
    });
    logger.debug("Set panel visibility", {
      visible,
      totalItems: panelItems.length,
      visibleItemsCount: visibleItems.length,
      animated: animationEnabled
    });
  }
  fadeIn(actor) {
    this.cancelAnimation(actor);
    actor.visible = true;
    const duration = this.settings.get_int("animation-duration");
    const slideOffset = 30;
    actor.opacity = 0;
    actor.set_translation(slideOffset, 0, 0);
    const actorWithTransitions = actor;
    const translationTransition = new Clutter.PropertyTransition({
      property_name: "translation-x"
    });
    translationTransition.set_from(slideOffset);
    translationTransition.set_to(0);
    translationTransition.set_duration(duration);
    translationTransition.set_progress_mode(
      Clutter.AnimationMode.EASE_OUT_QUAD
    );
    const opacityTransition = new Clutter.PropertyTransition({
      property_name: "opacity"
    });
    opacityTransition.set_from(0);
    opacityTransition.set_to(255);
    opacityTransition.set_duration(duration);
    opacityTransition.set_progress_mode(Clutter.AnimationMode.EASE_OUT_QUAD);
    actorWithTransitions.add_transition("veil-slide-in", translationTransition);
    actorWithTransitions.add_transition("veil-fade-in", opacityTransition);
    this.activeAnimations.set(actor, Date.now());
    const handlerId = actorWithTransitions.connect(
      "transitions-completed",
      () => {
        actorWithTransitions.disconnect(handlerId);
        this.activeAnimations.delete(actor);
        logger.debug("Slide in completed");
      }
    );
  }
  fadeOut(actor) {
    this.cancelAnimation(actor);
    const duration = this.settings.get_int("animation-duration");
    const slideOffset = 30;
    const actorWithTransitions = actor;
    const translationTransition = new Clutter.PropertyTransition({
      property_name: "translation-x"
    });
    translationTransition.set_from(0);
    translationTransition.set_to(slideOffset);
    translationTransition.set_duration(duration);
    translationTransition.set_progress_mode(Clutter.AnimationMode.EASE_IN_QUAD);
    const opacityTransition = new Clutter.PropertyTransition({
      property_name: "opacity"
    });
    opacityTransition.set_from(actor.opacity);
    opacityTransition.set_to(0);
    opacityTransition.set_duration(duration);
    opacityTransition.set_progress_mode(Clutter.AnimationMode.EASE_IN_QUAD);
    actorWithTransitions.add_transition(
      "veil-slide-out",
      translationTransition
    );
    actorWithTransitions.add_transition("veil-fade-out", opacityTransition);
    this.activeAnimations.set(actor, Date.now());
    const handlerId = actorWithTransitions.connect(
      "transitions-completed",
      () => {
        actorWithTransitions.disconnect(handlerId);
        const timeoutId2 = this.timeoutIds.get(actor);
        if (timeoutId2 !== void 0) {
          GLib.Source.remove(timeoutId2);
          this.timeoutIds.delete(actor);
        }
        actor.visible = false;
        actor.opacity = 255;
        actor.set_translation(0, 0, 0);
        this.activeAnimations.delete(actor);
        logger.debug("Slide out completed, actor hidden");
      }
    );
    const existingTimeoutId = this.timeoutIds.get(actor);
    if (existingTimeoutId !== void 0) {
      GLib.Source.remove(existingTimeoutId);
    }
    const timeoutId = GLib.timeout_add(
      GLib.PRIORITY_DEFAULT,
      duration + 100,
      () => {
        if (this.activeAnimations.has(actor)) {
          logger.warn("Slide out timeout - forcing completion");
          actor.visible = false;
          actor.opacity = 255;
          actor.set_translation(0, 0, 0);
          this.activeAnimations.delete(actor);
          this.timeoutIds.delete(actor);
        }
        return GLib.SOURCE_REMOVE;
      }
    );
    this.timeoutIds.set(actor, timeoutId);
  }
  cancelAnimation(actor) {
    if (this.activeAnimations.has(actor)) {
      const actorWithTransitions = actor;
      try {
        actorWithTransitions.remove_all_transitions?.();
      } catch {
        actorWithTransitions.remove_transition?.("veil-slide-in");
        actorWithTransitions.remove_transition?.("veil-slide-out");
        actorWithTransitions.remove_transition?.("veil-fade-in");
        actorWithTransitions.remove_transition?.("veil-fade-out");
      }
      const timeoutId = this.timeoutIds.get(actor);
      if (timeoutId !== void 0) {
        GLib.Source.remove(timeoutId);
        this.timeoutIds.delete(actor);
      }
      this.activeAnimations.delete(actor);
    }
  }
  restoreVisibility() {
    const panelItems = this.getAllPanelItems();
    const visibleItems = this.settings.get_strv("visible-items");
    panelItems.forEach((item) => {
      const shouldBeVisible = visibleItems.includes(item.name);
      item.container.visible = shouldBeVisible;
    });
    logger.debug("Restored panel visibility", {
      totalItems: panelItems.length,
      visibleItemsCount: visibleItems.length
    });
  }
  showAllItems() {
    const panelItems = this.getAllPanelItems();
    panelItems.forEach((item) => {
      item.container.visible = true;
    });
    logger.debug("Showed all panel items", { count: panelItems.length });
  }
  setOnItemsChanged(callback) {
    this.onItemsChangedCallback = callback;
  }
  destroy() {
    for (const actor of this.activeAnimations.keys()) {
      this.cancelAnimation(actor);
    }
    for (const timeoutId of this.timeoutIds.values()) {
      GLib.Source.remove(timeoutId);
    }
    this.activeAnimations.clear();
    this.timeoutIds.clear();
    if (this.addedHandlerId !== null) {
      MainPanel._rightBox.disconnect(this.addedHandlerId);
      this.addedHandlerId = null;
    }
    if (this.removedHandlerId !== null) {
      MainPanel._rightBox.disconnect(this.removedHandlerId);
      this.removedHandlerId = null;
    }
    logger.debug("PanelManager destroyed");
  }
}
export {
  PanelManager
};
