//    Tasks in panel
//    GNOME Shell extension
//    @fthx 2025


import Clutter from 'gi://Clutter';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Meta from 'gi://Meta';
import Shell from 'gi://Shell';
import St from 'gi://St';

import { AppMenu } from 'resource:///org/gnome/shell/ui/appMenu.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';


const ICON_SIZE = 18; // px

const TaskButton = GObject.registerClass(
    class TaskButton extends PanelMenu.Button {
        _init(window) {
            super._init();
            this.add_style_class_name('task-button');

            this._window = window;

            this._makeButtonBox();

            this._updateApp();
            this._updateVisibility();

            const windowId = this._window?.get_id();
            const buttonId = `task-button-${windowId}`;
            if (windowId && !Main.panel.statusArea[buttonId])
                Main.panel.addToStatusArea(buttonId, this, 99, 'left');

            this._connectSignals();
        }

        _connectSignals() {
            global.workspace_manager.connectObject('active-workspace-changed', () => this._updateVisibility(), this);

            this._window?.connectObject(
                'notify::appears-focused', () => this._updateFocus(), GObject.ConnectFlags.AFTER,
                'notify::demands-attention', () => this._updateDemandsAttention(), GObject.ConnectFlags.AFTER,
                'notify::gtk-application-id', () => this._updateApp(), GObject.ConnectFlags.AFTER,
                'notify::skip-taskbar', () => this._updateVisibility(), GObject.ConnectFlags.AFTER,
                'notify::urgent', () => this._updateDemandsAttention(), GObject.ConnectFlags.AFTER,
                'notify::wm-class', () => this._updateApp(), GObject.ConnectFlags.AFTER,
                'unmanaging', () => this.destroy(), GObject.ConnectFlags.AFTER,
                'workspace-changed', () => this._updateVisibility(), GObject.ConnectFlags.AFTER,
                this);

            this.connectObject(
                'notify::hover', () => this._onHover(),
                'button-press-event', (actor, event) => this._onClick(event),
                this);
        }

        _disconnectSignals() {
            global.workspace_manager.disconnectObject(this);

            this._window?.disconnectObject(this);
        }

        _makeButtonBox() {
            this._box = new St.BoxLayout({ reactive: true, track_hover: true, style_class: 'task-box' });

            this._icon = new St.Icon({ fallback_gicon: null });
            this._box.add_child(this._icon);

            this.add_child(this._box);

            this.setMenu(new AppMenu(this));
        }

        _toggleWindow() {
            this._windowOnTop = null;

            if (this._window?.appears_focused && this._windowIsOnActiveWorkspace) {
                if (this._window?.can_minimize() && !Main.overview.visible)
                    this._window?.minimize();
            } else {
                this._window?.activate(global.get_current_time());
                this._window?.focus(global.get_current_time());
            }
            Main.overview.hide();
        }

        _onClick(event) {
            const button = event?.get_button();

            if (button === Clutter.BUTTON_PRIMARY) {
                this.menu?.close();

                this._toggleWindow();

                return Clutter.EVENT_STOP;
            }

            if (button === Clutter.BUTTON_MIDDLE) {
                this.menu?.close();

                if (this._app?.can_open_new_window())
                    this._app?.open_new_window(-1);
                Main.overview.hide();

                return Clutter.EVENT_STOP;
            }

            return Clutter.EVENT_PROPAGATE;
        }

        _onHover() {
            if (Main.overview.visible || !Main.wm._canScroll)
                return;

            if (this.hover) {
                const monitorIndex = this._window?.get_monitor();
                const monitorWindows = this._window?.get_workspace()?.list_windows()
                    .filter(w => !w.minimized && w.get_monitor() === monitorIndex);
                this._windowOnTop = global.display.sort_windows_by_stacking(monitorWindows)?.at(-1);

                this._window?.raise();
            }
            else
                this._windowOnTop?.raise();
        }

        _updateWorkspace() {
            this._activeWorkspace = global.workspace_manager.get_active_workspace();
            this._windowIsOnActiveWorkspace = this._window?.located_on_workspace(this._activeWorkspace);
        }

        _updateFocus() {
            if (this._window?.appears_focused)
                this._box.add_style_class_name('task-box-focus');
            else
                this._box.remove_style_class_name('task-box-focus');
        }

        _updateDemandsAttention() {
            if (this._window?.demands_attention) {
                this._box.add_style_class_name('task-box-demands-attention');
                this.visible = true;
            } else {
                this._box.remove_style_class_name('task-box-demands-attention');
                this._updateVisibility();
            }
        }

        _updateApp() {
            if (this._window)
                this._app = Shell.WindowTracker.get_default().get_window_app(this._window);
            if (!this._app)
                return;

            const wmClass = this._window?.wm_class;
            if (wmClass?.startsWith('chrome'))
                this._icon.set_gicon(Gio.Icon.new_for_string(wmClass));
            else
                this._icon.set_gicon(this._app.icon);

            this.menu.setApp(this._app);

            this._icon.set_icon_size(ICON_SIZE);
        }

        _updateVisibility() {
            this._updateFocus();
            this._updateWorkspace();

            this.visible = !this._window?.skip_taskbar && this._windowIsOnActiveWorkspace;
        }

        destroy() {
            this._disconnectSignals();

            super.destroy();
        }
    });

const TaskBar = GObject.registerClass(
    class TaskBar extends GObject.Object {
        _init() {
            super._init();

            this._initTaskbar();
        }

        _connectSignals() {
            global.display.connectObject('window-created', (display, window) => this._makeTaskButton(window), this);

            Main.panel.connectObject('scroll-event', (actor, event) => Main.wm.handleWorkspaceScroll(event), this);
        }

        _disconnectSignals() {
            global.display.disconnectObject(this);

            Main.panel.disconnectObject(this);
        }

        _initTaskbar() {
            this._makeTaskbarTimeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 500, () => {
                this._makeTaskbar();

                this._makeTaskbarTimeout = null;
                return GLib.SOURCE_REMOVE;
            });
        }

        _destroyTaskbar() {
            if (this._makeTaskbarTimeout) {
                GLib.Source.remove(this._makeTaskbarTimeout);
                this._makeTaskbarTimeout = null;
            }

            for (const bin of Main.panel._leftBox.get_children()) {
                const button = bin.child;

                if (button && button instanceof TaskButton)
                    button.destroy();
            }
        }

        _makeTaskbar() {
            const workspacesNumber = global.workspace_manager.n_workspaces;

            for (let workspaceIndex = 0; workspaceIndex < workspacesNumber; workspaceIndex++) {
                const workspace = global.workspace_manager.get_workspace_by_index(workspaceIndex);
                const windowsList = workspace?.list_windows() ?? [];
                const windowsListSorted = global.display.sort_windows_by_stacking(windowsList);

                for (const window of windowsListSorted)
                    this._makeTaskButton(window);
            }

            this._connectSignals();
        }

        _makeTaskButton(window) {
            if (!window || window.skip_taskbar || window.get_window_type() === Meta.WindowType.MODAL_DIALOG)
                return;

            new TaskButton(window);
        }

        destroy() {
            this._disconnectSignals();
            this._destroyTaskbar();
        }
    });

export default class TasksInPanelExtension {
    enable() {
        this._taskbar = new TaskBar();
    }

    disable() {
        this._taskbar?.destroy();
        this._taskbar = null;
    }
}
