import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
import GLib from 'gi://GLib';
import * as LoginManager from 'resource:///org/gnome/shell/misc/loginManager.js';
import { NotificationService } from './lib/NotificationService.js';
import { DisplayBrightnessService } from './lib/DisplayBrightnessService.js';
import { SensorProxyService } from './lib/SensorProxyService.js';
import { BucketMapper } from './lib/BucketMapper.js';
import { KeyboardBacklightService } from './lib/KeyboardBacklightService.js';

export default class AdaptiveBrightnessExtension extends Extension {
  enable() {
    this._osdBaselineBucketIdx = -1;
    this._isPaused = false; // Track if automatic brightness is paused
    this.settings = this.getSettings();

    // Load brightness buckets from settings
    const buckets = this._loadBucketsFromSettings();
    this.bucketMapper = new BucketMapper(buckets);

    this.notifications = new NotificationService();
    this.displayBrightness = new DisplayBrightnessService();
    this.keyboardBacklight = new KeyboardBacklightService(this.settings);

    // Pass bucket boundary filter to sensor service for efficient event filtering
    this.sensorProxy = new SensorProxyService(
      this.bucketMapper.crossesBucketBoundary.bind(this.bucketMapper)
    );

    // Set up sleep/resume handling using GNOME Shell's LoginManager
    // When resuming from sleep, check light level immediately
    // This handles scenarios where we wake up in different lighting conditions
    // and might not receive ALS events (e.g., waking in darkness)
    this.loginManager = LoginManager.getLoginManager();

    Promise.allSettled([
      this.displayBrightness.start(),
      this.keyboardBacklight.start(),
      this.sensorProxy.start(),
    ]).then((results) => {
      if (results.some((r) => r.status === 'rejected')) {
        console.log('Some required services failed to start', results);
        return;
      }
      this.setupHandlers();

      // Set initial brightness based on current light level
      this.adjustBrightnessForLightLevel(this.sensorProxy.dbus.lightLevel, true);
    });

    // Watch for settings changes and reload buckets
    this.bucketSettingsChangedId = this.settings.connect(
      'changed::brightness-buckets',
      () => {
        const buckets = this._loadBucketsFromSettings();
        this.bucketMapper = new BucketMapper(buckets);
        this.adjustBrightnessForLightLevel(this.sensorProxy.dbus.lightLevel, true);
      }
    );

    // Watch for keyboard backlight bucket settings changes
    this.keyboardBucketSettingsChangedId = this.settings.connect(
      'changed::keyboard-backlight-buckets',
      () => {
        // Re-evaluate keyboard backlight with current light level
        this.adjustBrightnessForLightLevel(this.sensorProxy.dbus.lightLevel);
      }
    );
  }

  _loadBucketsFromSettings() {
    const bucketsVariant = this.settings.get_value('brightness-buckets');
    const buckets = [];

    for (let i = 0; i < bucketsVariant.n_children(); i++) {
      const tuple = bucketsVariant.get_child_value(i);
      buckets.push({
        min: tuple.get_child_value(0).get_uint32(),
        max: tuple.get_child_value(1).get_uint32(),
        brightness: tuple.get_child_value(2).get_double(),
      });
    }

    return buckets;
  }

  setupHandlers() {
    this.sleepResumeSignalId = this.loginManager?.connect(
      'prepare-for-sleep',
      (lm, aboutToSuspend) => {
        if (!aboutToSuspend) {
          // Force an update on resume to handle lighting changes during sleep
          this.sensorProxy.forceUpdate();
        }
      }
    );

    this.displayBrightness.onDisplayIsActiveChanged.add(() => {
      this.adjustBrightnessForLightLevel(this.sensorProxy.dbus.lightLevel);
    });
    this.displayBrightness.backend.onUserPreferenceChange.add(
      this.handleManualAdjustment.bind(this)
    );
    this.displayBrightness.onAmbientEnabledChanged.add(this.handleGSDAmbientEnableChanged.bind(this));

    this.sensorProxy.onLightLevelChanged.add((x) => this.adjustBrightnessForLightLevel(x));
    this.sensorProxy.onSensorAvailableChanged.add(this.handleSensorAvailableChanged.bind(this));

    this.handleGSDAmbientEnableChanged(this.displayBrightness.isGSDambientEnabled);
  }

  handleSensorAvailableChanged(val) {
    if (val === false) {
      this.notifications.showNotification(
        'Adaptive Brightness Extension',
        `Ambient Light Sensor is not available. Extension will not function`,
        { transient: false }
      );
    }
  }

  handleGSDAmbientEnableChanged(val) {
    if (val) {
      this.notifications.showNotification(
        'Adaptive Brightness Extension',
        "GNOME's automatic brightness feature is enabled. Press to disable it in Settings → Power, allowing the extension to work properly.",
        {
          transient: true,
          onActivate: () => {
            GLib.spawn_command_line_async('gnome-control-center power');
          },
        }
      );
    }
  }

  handleManualAdjustment(manualBrightness) {
    if (
      !this.displayBrightness.displayIsActive ||
      this.displayBrightness._settingBrightness ||
      manualBrightness === null ||
      this.sensorProxy.dbus.lightLevel === null ||
      this._isPaused
    ) {
      return;
    }

    // Pause automatic brightness management
    this._isPaused = true;

    // Release brightness control to allow manual adjustment
    if (this.displayBrightness.backend.releaseControl) {
      this.displayBrightness.backend.releaseControl();
    }

    // Show notification with resume on dismiss
    this.notifications.showNotification(
      'Adaptive Brightness',
      'Automatic brightness management is paused. Dismiss this notification to resume.',
      {
        transient: false,
        onDestroy: () => {
          this._isPaused = false;
          this.adjustBrightnessForLightLevel(this.sensorProxy.dbus.lightLevel, true);
        },
        action: {
          label: 'Settings',
          callback: () => {
            GLib.spawn_command_line_async(
              `gnome-extensions prefs ${this.metadata.uuid}`
            );
          },
        },
      }
    );

  }

  adjustBrightnessForLightLevel(luxValue, immediate = false) {
    if (!this.displayBrightness.displayIsActive || luxValue === null || this._isPaused) {
      this.keyboardBacklight.handleDisplayInactive().catch((e) => console.error(e));
      return;
    }

    const targetBucket = this.bucketMapper.mapLuxToBrightness(luxValue);

    if (targetBucket) {
      const targetBrightness = targetBucket.brightness;

      if (immediate) {
        this.displayBrightness.backend.brightness = targetBrightness;
      } else {
        this.displayBrightness.animateBrightness(targetBrightness).catch((e) => console.error(e));
      }

      // Update keyboard backlight based on enabled buckets
      const currentBucketIndex = this.bucketMapper.currentBucketIndex;
      const enabledBuckets = this._getEnabledKeyboardBuckets();
      const shouldEnableKeyboard = enabledBuckets.has(currentBucketIndex);

      this.keyboardBacklight.updateForLightLevel(shouldEnableKeyboard).catch((e) => console.error(e));
    }
  }

  _getEnabledKeyboardBuckets() {
    const bucketsVariant = this.settings.get_value('keyboard-backlight-buckets');
    const enabledBuckets = new Set();

    for (let i = 0; i < bucketsVariant.n_children(); i++) {
      enabledBuckets.add(bucketsVariant.get_child_value(i).get_uint32());
    }

    return enabledBuckets;
  }

  disable() {
    // "unlock-dialog" session mode is used to be able to listen for 'prepare-for-sleep' signal from LoginManager
    // in order to check light level immediately after resuming from suspend (with lock screen being shown).
    // This handles scenarios where resuming in dark environment does not trigger ALS event
    // and user might be exposed to very high brighness level causing discomfort for eyes.
    if (this.sleepResumeSignalId) {
      this.loginManager?.disconnect(this.sleepResumeSignalId);
      this.sleepResumeSignalId = null;
    }
    this.loginManager = null;

    if (this.bucketSettingsChangedId) {
      this.settings?.disconnect(this.bucketSettingsChangedId);
      this.bucketSettingsChangedId = null;
    }

    if (this.keyboardBucketSettingsChangedId) {
      this.settings?.disconnect(this.keyboardBucketSettingsChangedId);
      this.keyboardBucketSettingsChangedId = null;
    }

    if (this.sensorProxy) {
      this.sensorProxy.destroy();
      this.sensorProxy = null;
    }

    if (this.displayBrightness) {
      this.displayBrightness.destroy();
      this.displayBrightness = null;
    }

    if (this.keyboardBacklight) {
      this.keyboardBacklight.destroy();
      this.keyboardBacklight = null;
    }

    if (this.notifications) {
      this.notifications.destroy();
      this.notifications = null;
    }

    this.bucketMapper = null;
    this.settings = null;

    this._osdBaselineBucketIdx = -1;
    this._isPaused = false;
  }
}
