// SPDX-FileCopyrightText: [Your Copyright Information]
// SPDX-License-Identifier: GPL-3.0-or-later

import Gio from 'gi://Gio';
import GLib from 'gi://GLib';

/**
 * Extension entry point following GNOME Shell extension patterns
 * 
 * This class should be minimal and only handle extension lifecycle
 */
export default class AppearanceKeeperExtension {
    constructor() {
        this._manager = null;
        this._settings = this.getSettings();
    }

    enable() {
        this._manager = new AppearanceKeeperManager(this._settings);
        this._manager.start();
    }

    disable() {
        if (this._manager) {
            this._manager.stop();
            this._manager = null;
        }
    }

    getSettings() {
        return new Gio.Settings({ schema_id: 'org.gnome.shell.extensions.appearance-keeper' });
    }
}

/**
 * Core logic manager for appearance tracking
 */
class AppearanceKeeperManager {
    constructor(settings) {
        this._settings = settings;
        this._interfaceSettings = null;
        this._settingsExt = null;
        this._userThemeSettings = null;
        this._storedWallpapers = { light: null, dark: null };
        this._handlers = [];
        this._DEBUG = false;
        this._suspendSave = false;
        this._wallpaperDebounce = null;
        this._SCHEMA_BACKGROUND = 'org.gnome.desktop.background';
        this._SCHEMA_INTERFACE = 'org.gnome.desktop.interface';
/*
 * This method is used by other extensions, available on GNOME Extensions,
 * that manage the shell theme (e.g., AccentColorUserThemeExtension), thus
 * automatically applying a shell theme.
 *
 * We therefore believe we can do the same, despite the recommendation
 * "You shouldn't manage other extensions":
 *
 * 1. Without User Theme, we would have to reimplement its code in our extension
 *    and add a UI for selecting themes. This already exists in some extensions
 *    (e.g., User Theme X) and would go against the core idea of our extension.
 *
 * 2. Our approach ensures a **transparent user experience**:
 *    - The user continues to use their usual settings (GTK, icons, cursor, accent-color)
 *      without needing to open an additional interface.
 *    - When the user switches from light mode to dark mode (or vice versa),
 *      our extension automatically applies the stored themes without intervention.
 *    - If User Theme is not installed, only the standard settings are applied,
 *      ensuring compatibility and safety.
 */
        this._SCHEMA_USER_THEME = 'org.gnome.shell.extensions.user-theme';
    }

    // --- Lifecycle Management ---

    start() {
        this._log('Starting extension');

        if (!this._initializeServices()) {
            this._log('Service initialization failed', 'error');
            this.stop();
            return;
        }

        this._initializeMonitoring();
        this._log('Extension started successfully');
    }

    stop() {
        this._log('Stopping extension');

        if (this._wallpaperDebounce) {
            GLib.source_remove(this._wallpaperDebounce);
            this._wallpaperDebounce = null;
        }

        this._cleanupAllHandlers();
        this._resetAllSettings();

        this._log('Extension stopped');
    }

    // --- Logging ---

    _log(message, level = 'debug') {
        const prefix = '[AppearanceKeeper]';
        if (level === 'error') console.error(`${prefix} ERROR: ${message}`);
        else if (level === 'warning') console.warn(`${prefix} WARNING: ${message}`);
        else if (this._DEBUG) console.debug(`${prefix} ${message}`);
    }

    // --- Service Initialization ---

    _initializeServices() {
        try {
            return this._initMainSettings() &&
                   this._initUserThemeSettings();
        } catch (error) {
            this._log(`Service initialization error: ${error}`, 'error');
            return false;
        }
    }

    _initMainSettings() {
        try {
            this._interfaceSettings = new Gio.Settings({ schema_id: this._SCHEMA_INTERFACE });
            this._log('Main settings initialized');
            return true;
        } catch (error) {
            this._log(`Failed to initialize main settings: ${error}`, 'error');
            return false;
        }
    }

    _initUserThemeSettings() {
        try {
            const schemaSource = Gio.SettingsSchemaSource.get_default();
            const schema = schemaSource.lookup(this._SCHEMA_USER_THEME, true);

            if (schema) {
                this._userThemeSettings = new Gio.Settings({ schema_id: this._SCHEMA_USER_THEME });
                this._log('User theme settings initialized');
            } else {
                this._log('User theme extension not available', 'debug');
            }
            return true;
        } catch (error) {
            this._log(`Failed to initialize user theme settings: ${error}`, 'warning');
            return true;
        }
    }

    // --- Monitoring Setup ---

    _initializeMonitoring() {
        this._loadInitialWallpapers();
        this._setupWallpaperMonitoring();
        this._setupThemeMonitoring();
        this._setupStyleMonitoring();
    }

    _loadInitialWallpapers() {
        const bgSettings = new Gio.Settings({ schema_id: this._SCHEMA_BACKGROUND });
        this._storedWallpapers.light = bgSettings.get_string('picture-uri');
        this._storedWallpapers.dark = bgSettings.get_string('picture-uri-dark');
        this._log('Initial wallpapers loaded');
    }

    _setupWallpaperMonitoring() {
        const bgSettings = new Gio.Settings({ schema_id: this._SCHEMA_BACKGROUND });
        const handler = bgSettings.connect('changed', (settings, key) => {
            if (key === 'picture-uri' || key === 'picture-uri-dark') {
                this._handleWallpaperChange(key);
            }
        });
        this._storeHandler(bgSettings, handler);
    }

    _setupThemeMonitoring() {
        const themeKeys = ['gtk-theme', 'icon-theme', 'cursor-theme', 'accent-color'];
        themeKeys.forEach(key => {
            const handler = this._interfaceSettings.connect(`changed::${key}`, () => {
                this._saveThemeParameter(key);
            });
            this._storeHandler(this._interfaceSettings, handler);
        });

        if (this._userThemeSettings) {
            const shellHandler = this._userThemeSettings.connect('changed::name', () => {
                this._saveThemeParameter('shell-theme');
            });
            this._storeHandler(this._userThemeSettings, shellHandler);
        }
    }

    _setupStyleMonitoring() {
        const handler = this._interfaceSettings.connect('changed::color-scheme', () => {
            this._handleColorSchemeChange();
        });
        this._storeHandler(this._interfaceSettings, handler);
    }

    // --- Handler Management ---

    _storeHandler(settings, handlerId) {
        this._handlers.push({ settings, id: handlerId });
    }

    _cleanupAllHandlers() {
        this._handlers.forEach(handler => {
            try {
                if (handler?.settings && handler.id) {
                    handler.settings.disconnect(handler.id);
                }
            } catch (error) {
                this._log(`Error cleaning up handler: ${error}`, 'warning');
            }
        });
        this._handlers = [];
    }

    _resetAllSettings() {
        this._interfaceSettings = null;
        this._userThemeSettings = null;
        this._storedWallpapers = { light: null, dark: null };
    }

    // --- Settings Utilities ---

    _getSetting(settings, key, defaultValue = '') {
        if (!settings) return defaultValue;
        try { return settings.get_string(key) || defaultValue; }
        catch (error) { this._log(`Error reading setting ${key}: ${error}`, 'warning'); return defaultValue; }
    }

    _setSetting(settings, key, value) {
        if (!settings || value === null) return;
        try {
            const current = settings.get_string(key);
            if (current !== value) settings.set_string(key, value);
        } catch (error) {
            this._log(`Error setting ${key}: ${error}`, 'error');
        }
    }

    _getValidatedSetting(settings, key, defaultValue = '') {
        const value = this._getSetting(settings, key, defaultValue);
        if (typeof value !== 'string') return defaultValue;
        if (value.length > 100) return defaultValue;
        return value;
    }

    // --- Theme Management ---

    _saveThemeParameter(parameter) {
        if (this._suspendSave || !this._settings) return;

        const colorScheme = this._getSetting(this._interfaceSettings, 'color-scheme', 'default');
        const isDark = colorScheme.includes('dark');
        const prefix = isDark ? 'dark' : 'light';
        let value = '';

        switch (parameter) {
            case 'gtk-theme':
            case 'icon-theme':
            case 'cursor-theme':
            case 'accent-color':
                value = this._getValidatedSetting(this._interfaceSettings, parameter);
                break;
            case 'shell-theme':
                value = this._getSetting(this._userThemeSettings, 'name', '');
                break;
            default:
                this._log(`Unknown theme parameter: ${parameter}`, 'warning');
                return;
        }

        const key = parameter === 'shell-theme' ? `${prefix}-shell-theme` : `${prefix}-${parameter}`;
        this._setSetting(this._settings, key, value);
        this._log(`Saved ${parameter} for ${prefix} mode: ${value || '(default)'}`);
    }

    async _setShellTheme(themeName) {
        if (!this._userThemeSettings) return;
        try {
            if (!themeName) {
                this._userThemeSettings.reset('name');
                this._log('Shell theme reset to default');
            } else {
                this._userThemeSettings.set_string('name', themeName);
                this._log(`Shell theme applied: ${themeName}`);
            }
        } catch (error) {
            this._log(`Error setting shell theme: ${error}`, 'error');
        }
    }

    // --- Wallpaper Logic ---

    _handleWallpaperChange(changedKey) {
        if (this._wallpaperDebounce) GLib.source_remove(this._wallpaperDebounce);

        this._wallpaperDebounce = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 100, () => {
            this._wallpaperDebounce = null;
            this._processWallpaperChange(changedKey);
            return GLib.SOURCE_REMOVE;
        });
    }

    _processWallpaperChange(changedKey) {
        const bgSettings = new Gio.Settings({ schema_id: this._SCHEMA_BACKGROUND });
        const newValue = bgSettings.get_string(changedKey) || '';
        const colorScheme = this._getSetting(this._interfaceSettings, 'color-scheme', 'default');
        const isDark = colorScheme.includes('dark');

        this._log(`Wallpaper changed: ${changedKey} = ${newValue.substring(0, 50)}...`);
        this._storedWallpapers[changedKey === 'picture-uri' ? 'light' : 'dark'] = newValue;
    }

    _handleColorSchemeChange() {
        const colorScheme = this._getSetting(this._interfaceSettings, 'color-scheme', 'default');
        const isDark = colorScheme.includes('dark');
        this._log(`Color scheme changed to: ${colorScheme}`);
        this._applyThemeForScheme(isDark);
    }

    async _applyThemeForScheme(isDark) {
        if (!this._settings) return;
        try {
            this._suspendSave = true;
            const prefix = isDark ? 'dark' : 'light';

            const themes = {
                gtk: this._getValidatedSetting(this._settings, `${prefix}-gtk-theme`),
                shell: this._getValidatedSetting(this._settings, `${prefix}-shell-theme`),
                icon: this._getValidatedSetting(this._settings, `${prefix}-icon-theme`),
                cursor: this._getValidatedSetting(this._settings, `${prefix}-cursor-theme`),
                accent: this._getValidatedSetting(this._settings, `${prefix}-accent-color`)
            };

            this._setSetting(this._interfaceSettings, 'gtk-theme', themes.gtk);
            this._setSetting(this._interfaceSettings, 'icon-theme', themes.icon);
            this._setSetting(this._interfaceSettings, 'cursor-theme', themes.cursor);
            this._setSetting(this._interfaceSettings, 'accent-color', themes.accent);

            const currentShell = this._getSetting(this._userThemeSettings, 'name', '');
            if (themes.shell !== null && themes.shell !== currentShell) {
                await this._setShellTheme(themes.shell);
            }

            this._log(`Applied ${prefix} theme settings`);
            this._log(`Shell theme: ${themes.shell || '(default)'}`);

        } catch (error) {
            this._log(`Error applying theme: ${error}`, 'error');
        } finally {
            this._suspendSave = false;
        }
    }
}

