// SPDX-FileCopyrightText:
// SPDX-License-Identifier: GPL-3.0-or-later

import Gtk from 'gi://Gtk';
import Gio from 'gi://Gio';
import Adw from 'gi://Adw';
import Gdk from 'gi://Gdk';
import GdkPixbuf from 'gi://GdkPixbuf';
import GLib from 'gi://GLib';

const _ = (str) => str; // Placeholder for gettext

export default class AppearanceKeeperPreferences {
    constructor() {
        this._manager = new PreferencesManager();
    }

    fillPreferencesWindow(window) {
        this._manager.fillPreferencesWindow(window);
    }
}

class PreferencesManager {
    constructor() {
        this._valueLabels = new Map();
        this._imageWidgets = new Map();
        this._accentCircles = new Map();
        
        this._ACCENT_COLORS = {
            'default': '#1c71d8',
            'blue': '#1c71d8',
            'green': '#2ec27e',
            'yellow': '#e5a50a',
            'orange': '#ff7800',
            'red': '#c01c28',
            'purple': '#813d9c',
            'pink': '#e66197',
            'slate': '#52606d'
        };
        
        this._LOG_PREFIX = '[AppearanceKeeper]';
        this._displayedUris = new Map();
        
        this._initSettings();
    }

    _initSettings() {
        try {
            this._settings = new Gio.Settings({ schema_id: 'org.gnome.shell.extensions.appearance-keeper' });
            this._backgroundSettings = new Gio.Settings({ schema_id: 'org.gnome.desktop.background' });
        } catch (error) {
            console.error(`${this._LOG_PREFIX} Failed to initialize settings:`, error);
            throw error;
        }
    }

    fillPreferencesWindow(window) {
        this._window = window;
        
        this._window.connect('close-request', () => {
            this._cleanup();
            return false;
        });

        try {
            this._loadCSS();
            this._buildUI();
        } catch (error) {
            console.error(`${this._LOG_PREFIX} Error building UI:`, error);
            this._showErrorPage(error);
        }
    }

    _loadCSS() {
        const css = `
.wallpaper-vertical-preview {
    border-radius: 6px;
    border: 1px solid @borders;
    background-color: @card_bg_color;
    min-width: 96px;
    min-height: 72px;
}
.dim-label { opacity: 0.7 }
.error-icon { color: @error_color; opacity: 0.7 }
.param-container, .param-row { width: 280px; min-width: 280px }
.param-title { width: 130px; min-width: 130px; opacity: 0.7; font-weight: 600; text-align: left }
.param-value-container { width: 150px; min-width: 150px; justify-content: flex-end }
.param-value { text-align: right; opacity: 1 }
.accent-circle { font-size: 12px; margin-left: 4px; flex-shrink: 0 }
.refreshing { opacity: 0.5; transition: opacity 0.2s ease-in-out; }
        `;

        const provider = new Gtk.CssProvider();
        provider.load_from_data(css, -1);
        Gtk.StyleContext.add_provider_for_display(
            Gdk.Display.get_default(),
            provider,
            Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION
        );
    }

    _buildUI() {
        const page = new Adw.PreferencesPage({
            title: _('Appearance Keeper'),
            icon_name: 'preferences-desktop-theme-symbolic'
        });

        const mainGroup = new Adw.PreferencesGroup({
            title: _('Appearance Keeper'),
            description: _('Keeps your appearance settings in sync and applies them automatically when switching between light and dark modes.\nThis is a preview of your configuration. Change it using the GNOME tools.')
        });

        // Refresh button
        const refreshRow = new Adw.ActionRow({
            title: _('Refresh Preview'),
            subtitle: _('Update all values and wallpaper previews')
        });
        
        this._refreshButton = new Gtk.Button({
            label: _('Refresh'),
            halign: Gtk.Align.END,
            valign: Gtk.Align.CENTER
        });
        this._refreshButton.add_css_class('suggested-action');
        this._refreshButton.connect('clicked', () => this._refreshAll());
        
        refreshRow.add_suffix(this._refreshButton);
        refreshRow.set_activatable_widget(this._refreshButton);
        mainGroup.add(refreshRow);

        const verticalContainer = new Gtk.Box({
            orientation: Gtk.Orientation.VERTICAL,
            spacing: 12,
            margin_top: 4,
            margin_bottom: 8
        });

        verticalContainer.append(this._createModeSection(_('Light Mode'), 'light', 'picture-uri'));
        verticalContainer.append(new Gtk.Separator({ 
            orientation: Gtk.Orientation.HORIZONTAL, 
            margin_top: 12, 
            margin_bottom: 12 
        }));
        verticalContainer.append(this._createModeSection(_('Dark Mode'), 'dark', 'picture-uri-dark'));

        const customRow = new Adw.ActionRow();
        customRow.set_activatable(false);
        customRow.add_suffix(verticalContainer);
        mainGroup.add(customRow);
        
        page.add(mainGroup);
        this._window.add(page);
    }

    _createModeSection(modeTitle, mode, wallpaperKey) {
        const modeBox = new Gtk.Box({ 
            orientation: Gtk.Orientation.VERTICAL, 
            spacing: 8 
        });
        
        const horizontalContainer = new Gtk.Box({ 
            orientation: Gtk.Orientation.HORIZONTAL, 
            spacing: 16 
        });

        const leftSection = new Gtk.Box({ 
            orientation: Gtk.Orientation.VERTICAL, 
            spacing: 8, 
            halign: Gtk.Align.START 
        });
        
        leftSection.append(this._createModeTitleRow(modeTitle));
        leftSection.append(this._createParamsList(mode));

        horizontalContainer.append(leftSection);
        horizontalContainer.append(this._createWallpaperImage(wallpaperKey, mode));
        modeBox.append(horizontalContainer);
        
        return modeBox;
    }

    _createModeTitleRow(title) {
        const titleBox = new Gtk.Box({ 
            orientation: Gtk.Orientation.HORIZONTAL, 
            spacing: 6, 
            halign: Gtk.Align.START, 
            margin_bottom: 6 
        });
        
        const titleLabel = new Gtk.Label({ 
            label: title,
            halign: Gtk.Align.START,
            xalign: 0
        });
        titleLabel.add_css_class('title-4');
        titleBox.append(titleLabel);
        
        return titleBox;
    }

    _createParamsList(mode) {
        const paramsBox = new Gtk.Box({ 
            orientation: Gtk.Orientation.VERTICAL, 
            spacing: 6, 
            halign: Gtk.Align.START 
        });
        paramsBox.set_size_request(280, -1);
        paramsBox.add_css_class('param-container');

        const SETTINGS_KEYS = ['gtk-theme', 'shell-theme', 'icon-theme', 'cursor-theme', 'accent-color'];
        
        SETTINGS_KEYS.forEach(key => {
            const title = this._formatSettingTitle(key);
            paramsBox.append(this._createParamRow(title, `${mode}-${key}`, key));
        });
        
        return paramsBox;
    }

    _formatSettingTitle(key) {
        const titles = {
            'gtk-theme': _('GTK Theme'),
            'shell-theme': _('Shell Theme'), 
            'icon-theme': _('Icon Theme'),
            'cursor-theme': _('Cursor Theme'),
            'accent-color': _('Accent Color')
        };
        return titles[key] || key.replace('-', ' ').replace(/\b\w/g, l => l.toUpperCase());
    }

    _createParamRow(title, settingsKey, paramType) {
        const rowBox = new Gtk.Box({ 
            orientation: Gtk.Orientation.HORIZONTAL, 
            spacing: 8, 
            halign: Gtk.Align.START 
        });
        rowBox.add_css_class('param-row');
        rowBox.set_size_request(280, -1);

        const titleLabel = new Gtk.Label({ 
            label: title,
            halign: Gtk.Align.START,
            xalign: 0
        });
        titleLabel.add_css_class('param-title');
        titleLabel.add_css_class('dim-label');
        titleLabel.set_size_request(130, -1);
        rowBox.append(titleLabel);

        rowBox.append(new Gtk.Box({ orientation: Gtk.Orientation.HORIZONTAL, hexpand: true }));

        const valueContainer = new Gtk.Box({ 
            orientation: Gtk.Orientation.HORIZONTAL, 
            spacing: 4, 
            halign: Gtk.Align.END 
        });
        valueContainer.add_css_class('param-value-container');
        valueContainer.set_size_request(150, -1);

        const currentValue = this._settings.get_string(settingsKey) || _('Default');
        const valueLabel = new Gtk.Label({ 
            label: currentValue,
            halign: Gtk.Align.END,
            xalign: 1
        });
        valueLabel.add_css_class('param-value');

        if (paramType === 'accent-color' && currentValue !== _('Default')) {
            valueLabel.set_size_request(120, -1);
            valueContainer.append(valueLabel);
            const circleLabel = this._createAccentCircle(currentValue);
            valueContainer.append(circleLabel);
            this._accentCircles.set(settingsKey, circleLabel);
        } else {
            valueLabel.set_size_request(150, -1);
            valueContainer.append(valueLabel);
        }

        this._valueLabels.set(settingsKey, { 
            label: valueLabel, 
            type: paramType, 
            container: valueContainer,
            row: rowBox 
        });
        
        rowBox.append(valueContainer);
        return rowBox;
    }

    _createAccentCircle(colorValue) {
        const circleLabel = new Gtk.Label({ 
            label: '⬤',
            halign: Gtk.Align.CENTER,
            valign: Gtk.Align.CENTER
        });
        
        const colorHex = this._ACCENT_COLORS[colorValue] || this._ACCENT_COLORS['default'];
        const cssProvider = new Gtk.CssProvider();
        cssProvider.load_from_data(`.accent-circle-${colorValue} { color: ${colorHex}; }`, -1);

        const styleContext = circleLabel.get_style_context();
        styleContext.add_provider(cssProvider, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION);
        styleContext.add_class(`accent-circle-${colorValue}`);
        circleLabel.add_css_class('accent-circle');
        circleLabel.set_size_request(16, -1);

        return circleLabel;
    }

    _createWallpaperImage(wallpaperKey, mode) {
        const imageBox = new Gtk.Box({ 
            orientation: Gtk.Orientation.VERTICAL, 
            spacing: 4,
            halign: Gtk.Align.CENTER, 
            valign: Gtk.Align.CENTER, 
            hexpand: false 
        });
        
        const imageContainer = new Gtk.Box({ 
            orientation: Gtk.Orientation.VERTICAL, 
            halign: Gtk.Align.CENTER, 
            valign: Gtk.Align.CENTER 
        });
        
        this._imageWidgets.set(wallpaperKey, imageContainer);
        this._updateWallpaperImage(wallpaperKey, imageContainer);
        imageBox.append(imageContainer);
        return imageBox;
    }

    _updateWallpaperImage(wallpaperKey, imageContainer) {
        try {
            const wallpaperValue = this._backgroundSettings.get_string(wallpaperKey);

            // Check if URI has changed
            const currentDisplayedUri = this._displayedUris.get(wallpaperKey);
            if (currentDisplayedUri === wallpaperValue) {
                return;
            }

            // Store new displayed URI
            this._displayedUris.set(wallpaperKey, wallpaperValue);

            // Clear container only if URI changed
            let child = imageContainer.get_first_child();
            while (child) {
                const next = child.get_next_sibling();
                imageContainer.remove(child);
                child = next;
            }

            if (wallpaperValue && wallpaperValue !== 'Not set' && wallpaperValue.startsWith('file://')) {
                const filePath = decodeURIComponent(wallpaperValue.substring(7));

                // XML file special case
                if (filePath.endsWith('.xml')) {
                    this._handleXmlWallpaper(imageContainer, filePath, wallpaperKey);
                    return;
                }

                const file = Gio.File.new_for_path(filePath);
                if (file.query_exists(null)) {
                    // Direct loading without cache
                    const texture = Gdk.Texture.new_from_filename(filePath);
                    
                    if (texture) {
                        // Use Gtk.Picture with content-fit
                        const wallpaperImage = Gtk.Picture.new_for_paintable(texture);
                        wallpaperImage.set_content_fit(Gtk.ContentFit.CONTAIN);
                        wallpaperImage.add_css_class('wallpaper-vertical-preview');
                        wallpaperImage.set_size_request(160, 120);

                        imageContainer.append(wallpaperImage);
                        return;
                    }
                } else {
                    console.error(`${this._LOG_PREFIX} File does not exist:`, filePath);
                }
            } else {
                console.warn(`${this._LOG_PREFIX} Wallpaper key is not set or not a file URL:`, wallpaperValue);
            }
        } catch (e) {
            console.error(`${this._LOG_PREFIX} Error loading wallpaper image:`, e);
        }

        // If we get here, there was an error or invalid URI
        // Clear container and show placeholder
        let child = imageContainer.get_first_child();
        while (child) {
            const next = child.get_next_sibling();
            imageContainer.remove(child);
            child = next;
        }
        this._addPlaceholderImage(imageContainer, 'image-missing', 'dim-label');
    }

    _handleXmlWallpaper(imageContainer, xmlFilePath, wallpaperKey) {
        try {
            const file = Gio.File.new_for_path(xmlFilePath);
            const [success, contents] = file.load_contents(null);
            
            if (success) {
                const content = new TextDecoder().decode(contents);
                
                // Determine whether to look for day or night image based on wallpaperKey
                const isDarkMode = wallpaperKey === 'picture-uri-dark';
                const targetSuffix = isDarkMode ? 'night' : 'day';
                
                // Find all images in XML - expanded to handle nested size elements
                const fileMatches = content.matchAll(/<file>([^<]+)<\/file>/g);
                const fromMatches = content.matchAll(/<from>([^<]+)<\/from>/g);
                const toMatches = content.matchAll(/<to>([^<]+)<\/to>/g);
                
                // NEW: Extract images from <size> elements within <file>
                const sizeMatches = content.matchAll(/<size[^>]*>([^<]+)<\/size>/g);
                
                const allImages = [
                    ...Array.from(fileMatches, m => m[1]),
                    ...Array.from(fromMatches, m => m[1]),
                    ...Array.from(toMatches, m => m[1]),
                    ...Array.from(sizeMatches, m => m[1]) // Add size elements
                ];
                
                // Filter out empty or invalid entries
                const validImages = allImages.filter(img => img && img.trim() !== '');
                
                // First look for image with target suffix
                let targetImage = validImages.find(img => {
                    const cleanImg = img.trim();
                    return cleanImg.includes(`-${targetSuffix}.`) || 
                           cleanImg.includes(`_${targetSuffix}.`);
                });
                
                // If no target suffix found, try to find any image with common extensions
                if (!targetImage && validImages.length > 0) {
                    const imageExtensions = ['.png', '.jpg', '.jpeg', '.svg', '.webp'];
                    targetImage = validImages.find(img => {
                        const cleanImg = img.trim().toLowerCase();
                        return imageExtensions.some(ext => cleanImg.includes(ext));
                    });
                }
                
                // Otherwise take first available image
                if (!targetImage && validImages.length > 0) {
                    targetImage = validImages[0];
                }
                
                if (targetImage) {
                    const cleanTargetImage = targetImage.trim();
                    
                    // Resolve relative paths
                    let fullImagePath;
                    if (cleanTargetImage.startsWith('/')) {
                        fullImagePath = cleanTargetImage;
                    } else {
                        fullImagePath = `${GLib.path_get_dirname(xmlFilePath)}/${cleanTargetImage}`;
                    }
                    
                    // Direct loading without cache for XML images too
                    const texture = Gdk.Texture.new_from_filename(fullImagePath);
                    
                    if (texture) {
                        // Create overlay container for image + clock icon
                        const overlayBox = new Gtk.Overlay();
                        
                        const wallpaperImage = Gtk.Picture.new_for_paintable(texture);
                        wallpaperImage.set_content_fit(Gtk.ContentFit.CONTAIN);
                        wallpaperImage.add_css_class('wallpaper-vertical-preview');
                        wallpaperImage.set_size_request(160, 120);
                        
                        overlayBox.set_child(wallpaperImage);
                        
                        // Add clock icon overlay
                        const clockIcon = this._createClockIcon();
                        overlayBox.add_overlay(clockIcon);
                        
                        imageContainer.append(overlayBox);
                        return;
                    } else {
                        console.error(`${this._LOG_PREFIX} Failed to load texture from:`, fullImagePath);
                    }
                } else {
                    console.warn(`${this._LOG_PREFIX} No suitable image found in XML file`);
                }
            }
            
            // If we can't extract image, show special XML icon
            this._addXmlPlaceholder(imageContainer);
            
        } catch (e) {
            console.error(`${this._LOG_PREFIX} Error processing XML wallpaper:`, e);
            this._addXmlPlaceholder(imageContainer);
        }
    }

    _createClockIcon() {
        const clockBox = new Gtk.Box({
            orientation: Gtk.Orientation.HORIZONTAL,
            halign: Gtk.Align.START,
            valign: Gtk.Align.END,
            margin_start: 4,
            margin_bottom: 4
        });
        
        const clockImage = new Gtk.Image();
        clockImage.set_from_icon_name('preferences-system-time-symbolic');
        clockImage.set_pixel_size(16);
        
        // Apply style to make icon visible
        const cssProvider = new Gtk.CssProvider();
        cssProvider.load_from_data(`
            .clock-icon {
                color: white;
                border-radius: 50%;
                padding: 2px;
            }
        `, -1);
        
        clockImage.get_style_context().add_provider(cssProvider, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION);
        clockImage.get_style_context().add_class('clock-icon');
        
        clockBox.append(clockImage);
        return clockBox;
    }

    _addXmlPlaceholder(imageContainer) {
        const box = new Gtk.Box({ 
            orientation: Gtk.Orientation.VERTICAL, 
            spacing: 8, 
            halign: Gtk.Align.CENTER, 
            valign: Gtk.Align.CENTER 
        });
        
        const icon = new Gtk.Image();
        icon.set_from_icon_name('text-x-script');
        icon.set_pixel_size(36);
        icon.add_css_class('dim-label');
        box.append(icon);
        
        imageContainer.append(box);
    }

    _addPlaceholderImage(imageContainer, iconName, cssClass) {
        const box = new Gtk.Box({ orientation: Gtk.Orientation.VERTICAL, spacing: 4, halign: Gtk.Align.CENTER, valign: Gtk.Align.CENTER });
        const icon = new Gtk.Image();
        icon.set_from_icon_name(iconName);
        icon.set_pixel_size(36);
        icon.add_css_class(cssClass);
        box.append(icon);
        imageContainer.append(box);
    }

    _refreshAll() {
        // Disable button during refresh
        this._refreshButton.set_sensitive(false);
        this._refreshButton.set_label(_('Refreshing...'));
        
        this._applyRefreshingEffect(true);
        
        // STEP 1: Theme parameters (immediate)
        this._refreshThemeParameters();
        
        // STEP 2: Images with visual delay
        GLib.timeout_add(GLib.PRIORITY_DEFAULT, 300, () => {
            this._refreshWallpaperImages();
            
            // STEP 3: Restore interface
            GLib.timeout_add(GLib.PRIORITY_DEFAULT, 200, () => {
                this._applyRefreshingEffect(false);
                this._refreshButton.set_sensitive(true);
                this._refreshButton.set_label(_('Refresh'));
                return GLib.SOURCE_REMOVE;
            });
            
            return GLib.SOURCE_REMOVE;
        });
    }

    _applyRefreshingEffect(enable) {
        // Apply/remove refreshing class to all elements
        this._valueLabels.forEach((data, key) => {
            if (data.row) {
                if (enable) {
                    data.row.add_css_class('refreshing');
                } else {
                    data.row.remove_css_class('refreshing');
                }
            }
        });

        this._imageWidgets.forEach((container, key) => {
            if (enable) {
                container.add_css_class('refreshing');
            } else {
                container.remove_css_class('refreshing');
            }
        });
    }

    _refreshThemeParameters() {
        const extensionKeys = [
            'light-gtk-theme', 'light-shell-theme', 'light-icon-theme', 'light-cursor-theme', 'light-accent-color',
            'dark-gtk-theme', 'dark-shell-theme', 'dark-icon-theme', 'dark-cursor-theme', 'dark-accent-color'
        ];

        extensionKeys.forEach(key => this._updateValueLabel(key));
    }

    _refreshWallpaperImages() {
        ['picture-uri', 'picture-uri-dark'].forEach(key => {
            const imageContainer = this._imageWidgets.get(key);
            if (imageContainer) {
                const currentUri = this._backgroundSettings.get_string(key);
                const displayedUri = this._displayedUris.get(key);
                
                if (currentUri !== displayedUri) {
                    this._updateWallpaperImage(key, imageContainer);
                }
            }
        });
    }

    _updateValueLabel(settingsKey) {
        const valueData = this._valueLabels.get(settingsKey);
        if (!valueData) return;

        const { label, type, container } = valueData;
        const currentValue = this._settings.get_string(settingsKey) || _('Default');
        
        // Update label
        label.set_label(currentValue);

        if (type === 'accent-color') {
            const existingCircle = this._accentCircles.get(settingsKey);
            if (currentValue !== _('Default')) {
                if (existingCircle) {
                    // Update existing circle
                    const ctx = existingCircle.get_style_context();
                    Object.keys(this._ACCENT_COLORS).forEach(c => ctx.remove_class(`accent-circle-${c}`));
                    const colorHex = this._ACCENT_COLORS[currentValue] || this._ACCENT_COLORS['default'];
                    const cssProvider = new Gtk.CssProvider();
                    cssProvider.load_from_data(`.accent-circle-${currentValue} { color: ${colorHex} }`, -1);
                    ctx.add_provider(cssProvider, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION);
                    ctx.add_class(`accent-circle-${currentValue}`);
                } else {
                    // Create new circle
                    const newCircle = this._createAccentCircle(currentValue);
                    container.append(newCircle);
                    this._accentCircles.set(settingsKey, newCircle);
                }
            } else if (existingCircle) {
                // Remove circle if default value
                container.remove(existingCircle);
                this._accentCircles.delete(settingsKey);
            }
        }
    }

    _cleanup() {
        // Clean up all resources
        this._valueLabels?.clear();
        this._imageWidgets?.clear();
        this._accentCircles?.clear();
        this._displayedUris?.clear();

        this._valueLabels = null;
        this._imageWidgets = null;
        this._accentCircles = null;
        this._displayedUris = null;
        this._settings = null;
        this._backgroundSettings = null;
        this._window = null;
        this._refreshButton = null;
    }

    _showErrorPage(error) {
        const page = new Adw.PreferencesPage();
        const group = new Adw.PreferencesGroup();
        const row = new Adw.ActionRow({
            title: _('Error loading preferences'),
            subtitle: String(error)
        });
        
        group.add(row);
        page.add(group);
        this._window.add(page);
    }
}
