/**
 * An object describing a token with its associated weight (=multiplier)
 *
 * For instance, a tokenizer might want to return tokens similar to the ones
 * that appear in the passed object with a lower weight.
 *
 * @typedef {Object} WeightedToken
 * @property {string} token - The token
 * @property {number} weight - A multiplier
 */

/**
 * Split a value (ideally of specific type) into all tokens that may be relevant to a search.
 *
 * If this function is passed to a {@linkcode getScorer} in a {@linkcode KeyWeight},
 * this is passed values that the objects passed to the scorer contain under the specified key.
 *
 * If this function is passed to an {@linkcode Index} object,
 * it will be called on a list of terms, split at whitespace, of a query.
 *
 * @callback tokenizer
 * @param {Object} value - the value to tokenize, as described above
 * @return {WeightedToken[]} An array of all relevant tokens that value contains
 */

const NONWORD_RE = /\W/g;

const FIRST_WORD_BOOST = 1;
const FIRST_TOKEN_BOOST = 2;

const ACRONYM_PENALTY = -1;
const CHARACTER_SWAP_PENALTY = -1;

/**
 * Swap the character at index `position` in the string with the next character.
 *
 * @param {string} string - The string whose characters to swap
 * @param {number} position - At what index to swap the character with the next
 *                            character.
 * @return {string} A string with two characters swapped
 */
const swapCharacters = (string, position) =>
    string.substring(0, position) +
    string.charAt(position + 1) +
    string.charAt(position) +
    string.substring(position + 2);

/**
 * Save weights for an n-gram and (optionally) character-swapped versions of
 * that n-gram to an object.
 *
 * Character-swapped versions are assigned weights of CHARACTER_SWAP_PENALTY
 * lower, which may combine with other weight boosts/penalties.
 *
 * @param {object} weight - The object to save the weights to, with the n-gram
 *                          (and char-swap versions) as key and an integer
 *                          weight as value
 * @param {string} nGram - The n-gram for which to save weights
 * @param {number} baseWeight - The "neutral" weight of an n-gram that doesn't
 *                              match any special cases
 * @param {number} wordIndex - The index of the word this n-gram originated
 *                             from. If 0, score is increased by
 *                             FIRST_WORD_BOOST
 * @param {number} nGramIndex - The index of the n-gram in its word. If 0,
 *                              score is increased by FIRST_TOKEN_BOOST
 * @param {boolean} isAcronym - Whether the n-gram was generated by using first
 *                              letters of words or capital letters and should
 *                              therefore have a weight decreased by
 *                              ACRONYM_PENALTY
 */
const saveNGramWeights = (
    weights,
    nGram,
    baseWeight,
    wordIndex,
    nGramIndex,
    isAcronym,
) => {
    let weight = baseWeight;

    // Prefer first word, reduce weight of last two words (guessed acronyms)
    if (wordIndex == 0) {
        weight += FIRST_WORD_BOOST;
    }

    if (isAcronym) {
        weight += ACRONYM_PENALTY;
    }

    // Prefer start of words
    if (nGramIndex == 0) {
        weight += FIRST_TOKEN_BOOST;
    }

    if (!(nGram in weights)) {
        weights[nGram] = weight;
    } else {
        weights[nGram] = Math.max(weights[nGram], baseWeight);
    }

    if (baseWeight > 1)
        for (var i = 0; i < nGram.length - 1; i++) {
            const swappedNGram = swapCharacters(nGram, i);

            if (!(swappedNGram in weights)) {
                weights[swappedNGram] = weight + CHARACTER_SWAP_PENALTY;
            } else {
                weights[swappedNGram] = Math.max(
                    weights[swappedNGram],
                    weight + CHARACTER_SWAP_PENALTY,
                );
            }
        }
};

/**
 * Get a tokenizer function that takes a string and splits it into n-grams,
 * optionally also including n-grams with two adjacent characters swapped
 *
 * @param  {number} [maxNGramLength=5] - The maximum length of the n-grams,
 *                                       e.g. 2 for ["hi", "hit"] -> ["h", "i", "t", "hi", "it"]
 * @param  {number} [unswappedWeight=1] - What weight to give n-grams in their original form with
 *                                        no char swaps.
 *                                        A character swap always deducts 1 from this weight.
 *                                        If this value is 1, no character swaps are performed.
 * @return {tokenizer} A tokenizer that splits keywords (string array) into
 *                       n-grams of up to a certain length
 */
export const getStringNgramTokenizer = (
    maxNGramLength = 5,
    unswappedWeight = 1,
    generateAcronyms = false,
) => {
    return (str) => {
        const tokenWeights = {};

        if (typeof str === "string") {
            const words = str
                .toLowerCase()
                .split(NONWORD_RE)
                .filter((word) => !NONWORD_RE.test(word));

            if (generateAcronyms) {
                // Add acronym generated from first letter of each word to list
                words.push(
                    words.reduce((previous, current) => previous + current[0]),
                );

                // Add acronym generated from all capital letters to list
                words.push(
                    Array.from(str)
                        .filter(
                            (character) => character.toLowerCase() != character,
                        )
                        .join(""),
                );
            }

            for (
                var nGramLength = 1;
                nGramLength <= maxNGramLength;
                nGramLength++
            ) {
                words.forEach((word, wordIndex) =>
                    getNGrams(word.toLowerCase(), nGramLength).forEach(
                        (nGram, index) => {
                            saveNGramWeights(
                                tokenWeights,
                                nGram,
                                unswappedWeight,
                                wordIndex,
                                index,
                                wordIndex >= words.length - 2,
                            );
                        },
                    ),
                );
            }
        }

        return Object.keys(tokenWeights).map((token) => ({
            token: token,
            weight: tokenWeights[token],
        }));
    };
};

/**
 * Get a tokenizer function that takes an array of strings and splits them into
 * n-grams, optionally also including n-grams with two adjacent characters swapped
 *
 * @param  {number} [maxNGramLength=5] - The maximum length of the n-grams,
 *                                       e.g. 2 for ["hi", "hit"] -> ["h", "i", "t", "hi", "it"]
 * @param  {number} [unswappedWeight=1] - What weight to give n-grams in their original form with
 *                                        no char swaps.
 *                                        A character swap always deducts 1 from this weight.
 *                                        If this value is 1, no character swaps are performed.
 * @return {tokenizer} A tokenizer that splits keywords (string array) into
 *                       n-grams of up to a certain length
 */
export const getKeywordArrayNgramTokenizer = (
    maxNGramLength = 5,
    unswappedWeight = 1,
) => {
    return (keywords) => {
        const tokenWeights = {};

        if (
            keywords &&
            typeof keywords === "object" &&
            "map" in keywords &&
            "reduce" in keywords
        )
            for (
                var nGramLength = 1;
                nGramLength <= maxNGramLength;
                nGramLength++
            ) {
                keywords
                    .map((kw) => kw.replace(NONWORD_RE, "").toLowerCase())
                    .forEach((word) =>
                        getNGrams(word, nGramLength).forEach((nGram, index) => {
                            saveNGramWeights(
                                tokenWeights,
                                nGram,
                                unswappedWeight,
                                1, // Don't use wordIndex boost for keywords
                                index,
                                false, // No acronyms are generated for keywords
                            );
                        }),
                    );
            }

        return Object.keys(tokenWeights).map((token) => ({
            token: token,
            weight: tokenWeights[token],
        }));
    };
};

/**
 * Get an array of all substrings of str that are n long, i.e. n-grams
 *
 * @param  {String} str - The string to split into n-grams
 * @param  {Number} n - The length of the substrings, e.g. 2 for ["he", "el", "lo"]
 * @return {Array} An array of all substrings of str of length n
 */
const getNGrams = (str, n) => {
    if (str.length < n) {
        return [];
    }

    const nGrams = [];

    for (let index = str.length - n; index >= 0; index--) {
        nGrams.unshift(str.slice(index, index + n));
    }

    return nGrams;
};
