/*
 * Compatibility / Internal API usage
 *
 * This extension customizes the GNOME Shell app grid and folder popups by
 * interacting with internal overview and appDisplay structures
 * (e.g. Main.overview._overview._controls._appDisplay, _folderIcons,
 * icon.view._grid, icon._dialog, layout_manager internals, etc.).
 *
 * These are private Shell APIs and can change between GNOME Shell versions.
 * The code therefore performs defensive checks (typeof/function guards,
 * version detection via Config.PACKAGE_VERSION, optional chaining) and may
 * fall back gracefully when a given field or method is not available.
 *
 * As a consequence, behaviour may differ slightly across Shell releases,
 * and updates to GNOME Shell can require adjustments to this extension.
 */

import Clutter from 'gi://Clutter';
import St from 'gi://St';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as IconGrid from 'resource:///org/gnome/shell/ui/iconGrid.js';
import * as Config from 'resource:///org/gnome/shell/misc/config.js';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';

const SHELL_VERSION = Config.PACKAGE_VERSION;
const SHELL_MAJOR = parseInt(SHELL_VERSION.split('.')[0], 10);

let appGridTuner = null;

class AppGridTuner {

  constructor(settings) {
    this.settings = settings;
    this.originalStates = new Map();
    this.isCompatible = this._checkCompatibility();
  }

  _checkCompatibility() {
    try {
      // Check if required components exist
      if (!Main.overview) {
        console.warn('[AppGridTuner] Main.overview not available');
        return false;
      }

      if (!Main.overview._overview || !Main.overview._overview._controls) {
        console.warn('[AppGridTuner] Overview controls not available');
        return false;
      }

      const appDisplay = Main.overview._overview._controls._appDisplay;
      if (!appDisplay) {
        console.warn('[AppGridTuner] App display not available');
        return false;
      }

      return true;
    } catch (e) {
      console.error(`[AppGridTuner] Compatibility check failed: ${e.message}`);
      return false;
    }
  }

  _getAppDisplay() {
    try {
      if (!this.isCompatible) return null;
      return Main.overview._overview._controls._appDisplay;
    } catch (e) {
      console.error(`[AppGridTuner] Failed to get app display: ${e.message}`);
      return null;
    }
  }

  _safeExecute(callback, fallback = null) {
    try {
      return callback();
    } catch (e) {
      console.error(`[AppGridTuner] Safe execution failed: ${e.message}`);
      return fallback;
    }
  }

  applyFolderViewChanges() {
    return this._safeExecute(() => {
      const MainAppDisplay = this._getAppDisplay();
      if (!MainAppDisplay) return;

      if (!MainAppDisplay._folderIcons || !Array.isArray(MainAppDisplay._folderIcons)) {
        console.warn('[AppGridTuner] No folder icons available');
        return;
      }

      MainAppDisplay._folderIcons.forEach(icon => {
        this._safeExecute(() => {
          if (!icon.view || !icon.view._grid || !icon.view._grid.layout_manager) {
            console.warn('[AppGridTuner] Invalid folder icon structure');
            return;
          }

          // Store original state before modifying
          if (!this.originalStates.has(icon)) {
            this.originalStates.set(icon, {
              fixedIconSize: icon.view._grid.layout_manager.fixedIconSize,
              rowsPerPage: icon.view._grid.layout_manager.rowsPerPage,
              columnsPerPage: icon.view._grid.layout_manager.columnsPerPage,
              dialogStyle: undefined,
              scrollContentPadding: undefined,
              scrollPolicy: undefined,
              scrollClip: undefined,
            });
          }

          const storedState = this.originalStates.get(icon);

          icon.view._grid.layout_manager.fixedIconSize = this.iconSize;
          icon.view._grid.layout_manager.rowsPerPage = (Math.ceil(icon.getAppIds().length / this.fColumns) < this.fRows) ? Math.ceil(icon.getAppIds().length / this.fColumns) : this.fRows;
          icon.view._grid.layout_manager.columnsPerPage = (icon.getAppIds().length > this.fColumns) ? this.fColumns : icon.getAppIds().length;
          
          if (typeof icon._ensureFolderDialog === 'function') {
            icon._ensureFolderDialog();
          }
          
          if (icon._dialog && icon._dialog.child) {
            if (storedState && storedState.dialogStyle === undefined) {
              storedState.dialogStyle = icon._dialog.child.style || '';
            }

            const dialogWidth = Math.min(
              Math.max(
                icon.view._grid.layout_manager.columnsPerPage * (this.iconSize + this.folderItemSpacing) + this.folderPopupWidth,
                this.folderPopupWidth
              ),
              800
            );
            const dialogHeight = Math.min(
              icon.view._grid.layout_manager.rowsPerPage * (this.iconSize + this.folderItemHeightOffset) + this.folderPopupHeightOffset,
              600
            );
            const horizontalPadding = this.folderSidePadding || 0;

            icon._dialog._popdownCallbacks = [];
            icon._dialog.child.style = `box-sizing: border-box; height: ${dialogHeight}px; width: ${dialogWidth}px; padding-left: ${horizontalPadding}px; padding-right: ${horizontalPadding}px;`;

            if (icon._dialog._scrollView) {
              const scrollView = icon._dialog._scrollView;

              if (storedState) {
                if (storedState.scrollClip === undefined) {
                  storedState.scrollClip = scrollView.clip_to_allocation ?? false;
                }
                if (storedState.scrollPolicy === undefined && typeof scrollView.get_policy === 'function') {
                  storedState.scrollPolicy = scrollView.get_policy();
                }
                if (storedState.scrollContentPadding === undefined && scrollView.content_padding) {
                  const padding = scrollView.content_padding;
                  storedState.scrollContentPadding = {
                    left: padding.left ?? 0,
                    right: padding.right ?? 0,
                    top: padding.top ?? 0,
                    bottom: padding.bottom ?? 0,
                  };
                }
              }

              scrollView.clip_to_allocation = true;
              scrollView.content_padding = new Clutter.Margin({
                left: horizontalPadding,
                right: horizontalPadding,
                top: 0,
                bottom: 0,
              });

              if (typeof scrollView.set_policy === 'function') {
                scrollView.set_policy(St.PolicyType.NEVER, St.PolicyType.AUTOMATIC);
              }
            }
          }
          if (typeof icon.view._redisplay === 'function') {
            icon.view._redisplay();
          }

          const grid = icon.view._grid;
          const layoutManager = grid && grid.layout_manager;

          if (layoutManager) {
            if (SHELL_MAJOR < 47) {
              if (typeof layoutManager._adjustIconSize === 'function') {
                layoutManager._adjustIconSize();
              }
            } else {
              if (typeof grid.queue_relayout === 'function') {
                grid.queue_relayout();
              } else if (typeof layoutManager._adjustIconSize === 'function') {
                layoutManager._adjustIconSize();
              }
            }
          }
        });
      });
    });
  }

  applyAnimationSettings() {
    return this._safeExecute(() => {
      const MainAppDisplay = this._getAppDisplay();
      if (!MainAppDisplay) return;

      // Apply page switch animation time
      if (MainAppDisplay._pageIndicators) {
        MainAppDisplay._pageIndicators.style = `transition-duration: ${this.pageSwitchAnimationTime}ms;`;
      }

      // Apply open animation time for folder dialogs
      if (MainAppDisplay._folderIcons && Array.isArray(MainAppDisplay._folderIcons)) {
        MainAppDisplay._folderIcons.forEach(icon => {
          this._safeExecute(() => {
            if (icon._dialog && icon._dialog.child) {
              const currentStyle = icon._dialog.child.style || '';
              icon._dialog.child.style = currentStyle + `transition-duration: ${this.openAnimationTime}ms;`;
            }
          });
        });
      }
    });
  }

  startAppGridTuning() {
    if (!this.isCompatible) {
      console.warn('[AppGridTuner] Extension is not compatible with current GNOME Shell version');
      return;
    }

    return this._safeExecute(() => {
      const MainAppDisplay = this._getAppDisplay();
      if (!MainAppDisplay) return;

      let appGrid = MainAppDisplay._grid;
      if (!appGrid) {
        console.warn('[AppGridTuner] App grid not available');
        return;
      }

      if (this.reloadSignal) {
        this.settings.disconnect(this.reloadSignal);
        this.reloadSignal = 0;
      }

      this.reloadSignal = this.settings.connect('changed::reload-signal', () => {
        this.undoChanges();
        this.startAppGridTuning();
      });

      const gridRows = this.settings.get_int("appgrid-max-rows");
      const gridCols = this.settings.get_int("appgrid-max-columns");
      appGrid.style = "font-size: " + this.settings.get_double("app-icon-font-size") + "px;" + this.settings.get_string("label-style");
      this.iconSize = this.settings.get_int("appgrid-icon-size");
      this.fRows = this.settings.get_int("folder-max-rows");
      this.fColumns = this.settings.get_int("folder-max-columns");
      this.sidePadding = this.settings.get_int("side-padding");
      this.folderSidePadding = this.settings.get_int("folder-side-padding");
      this.folderPopupWidth = this.settings.get_int("folder-popup-width");
      this.folderPopupHeightOffset = this.settings.get_int("folder-popup-height-offset");
      this.folderItemSpacing = this.settings.get_int("folder-item-spacing");
      this.folderItemHeightOffset = this.settings.get_int("folder-item-height-offset");
      this.openAnimationTime = this.settings.get_int("open-animation-time");
      this.pageSwitchAnimationTime = this.settings.get_int("page-switch-animation-time");
      
      // Store original grid state
      if (!this.originalGridState) {
        this.originalGridState = {
          fixedIconSize: appGrid.layout_manager.fixedIconSize,
          style: appGrid.style,
          adaptToSize: MainAppDisplay.adaptToSize
        };
      }
      
      appGrid.layout_manager.fixedIconSize = this.iconSize;
      
      // Store and override adaptToSize function safely
      if (typeof MainAppDisplay.adaptToSize === 'function') {
        this.defaultAdaptToSizeFunction = MainAppDisplay.adaptToSize;
        MainAppDisplay.adaptToSize = this.overriddenAdaptToSizeFunction.bind(this);
      }

      if(MainAppDisplay._folderIcons && MainAppDisplay._folderIcons.length > 0) {
        this.applyFolderViewChanges();
      }
       
      if (this.reloadingSig) {
        MainAppDisplay.disconnect(this.reloadingSig);
      }

      this.reloadingSig = MainAppDisplay.connect("view-loaded", ()=> {
        this.applyFolderViewChanges();  
        this.applyAnimationSettings();
        this.updatePages();
      });

      // Apply initial animation settings
      this.applyAnimationSettings();   

      // Apply grid modes via API and enforce mode 0
      if (typeof appGrid.setGridModes === 'function') {
        appGrid.setGridModes([{ rows: gridRows, columns: gridCols }]);
      } else if (appGrid._gridModes && appGrid._gridModes[0]) {
        appGrid._gridModes[0] = { rows: gridRows, columns: gridCols };
      }

      appGrid._currentMode = 0; 
      appGrid.layout_manager.rowsPerPage      = gridRows;
      appGrid.layout_manager.columnsPerPage   = gridCols;

      // Force re-layout and icon size recomputation
      appGrid.layout_manager.fixedIconSize = this.iconSize;
      if (appGrid.layout_manager._pageWidth !== undefined) {
        appGrid.layout_manager._pageWidth--;
        appGrid.layout_manager.adaptToSize(appGrid.layout_manager._pageWidth, appGrid.layout_manager._pageHeight);
      }

      this.updatePages();
      if (typeof MainAppDisplay._redisplay === 'function') {
        MainAppDisplay._redisplay();
      }
    });
  }
  
  updatePages() {  
    return this._safeExecute(() => {
      const MainAppDisplay = this._getAppDisplay();
      if (!MainAppDisplay || !MainAppDisplay._grid || !MainAppDisplay._grid.layout_manager) return;

      const layoutManager = MainAppDisplay._grid.layout_manager;
      if (layoutManager._pages && Array.isArray(layoutManager._pages)) {
        for(let i = 0; i < layoutManager._pages.length; i++) {
          if (typeof layoutManager._fillItemVacancies === 'function') {
            layoutManager._fillItemVacancies(i);
          }
        }   
        if (typeof layoutManager._updatePages === 'function') {
          layoutManager._updatePages();
        }
        for(let i = 0; i < layoutManager._pages.length; i++) {
          if (typeof layoutManager._fillItemVacancies === 'function') {
            layoutManager._fillItemVacancies(i);
          }
        }
      }
    });
  }
  
  undoChanges() {
    return this._safeExecute(() => {
      const MainAppDisplay = this._getAppDisplay();
      if (!MainAppDisplay) return;

      let appGrid = MainAppDisplay._grid;
      if (!appGrid || !appGrid.layout_manager) return;

      // Restore original grid state
      if (this.originalGridState) {
        if (appGrid.layout_manager._pageWidth !== undefined) {
          appGrid.layout_manager._pageWidth--;
        }
        
        if (this.defaultAdaptToSizeFunction && typeof MainAppDisplay.adaptToSize === 'function') {
          MainAppDisplay.adaptToSize = this.defaultAdaptToSizeFunction;
        }
        
        appGrid.layout_manager.fixedIconSize = this.originalGridState.fixedIconSize;
        appGrid.style = this.originalGridState.style;
        
        if (appGrid._gridModes && appGrid._gridModes[0]) {
          appGrid._gridModes[0].rows = 8;
          appGrid._gridModes[0].columns = 3;
        }
        
        if (appGrid.layout_manager._pageWidth !== undefined && appGrid.layout_manager._pageHeight !== undefined) {
          appGrid.layout_manager.adaptToSize(appGrid.layout_manager._pageWidth, appGrid.layout_manager._pageHeight);
        }
      }
      
      // Restore folder states
      this.originalStates.forEach((originalState, icon) => {
        this._safeExecute(() => {
          if (icon.view && icon.view._grid && icon.view._grid.layout_manager) {
            icon.view._grid.layout_manager.fixedIconSize = originalState.fixedIconSize;
            icon.view._grid.layout_manager.rowsPerPage = originalState.rowsPerPage;
            icon.view._grid.layout_manager.columnsPerPage = originalState.columnsPerPage;
          }

          if (icon._dialog && icon._dialog.child) {
            if (originalState.dialogStyle !== undefined) {
              icon._dialog.child.style = originalState.dialogStyle;
            }

            if (icon._dialog._scrollView) {
              const scrollView = icon._dialog._scrollView;

              if (originalState.scrollClip !== undefined) {
                scrollView.clip_to_allocation = originalState.scrollClip;
              }

              if (originalState.scrollContentPadding) {
                scrollView.content_padding = new Clutter.Margin({
                  left: originalState.scrollContentPadding.left,
                  right: originalState.scrollContentPadding.right,
                  top: originalState.scrollContentPadding.top,
                  bottom: originalState.scrollContentPadding.bottom,
                });
              }

              if (originalState.scrollPolicy && typeof scrollView.set_policy === 'function') {
                const [hPolicy, vPolicy] = originalState.scrollPolicy;
                scrollView.set_policy(hPolicy, vPolicy);
              }
            }
          }
        });
      });

      if (this.reloadingSig) {
        MainAppDisplay.disconnect(this.reloadingSig);
        this.reloadingSig = 0;
      }
      
      this.updatePages();
      if (typeof MainAppDisplay._redisplay === 'function') {
        MainAppDisplay._redisplay();
      }
    });
  }

  destroy() {
    this.undoChanges();
    
    if (this.reloadSignal) {
      this.settings.disconnect(this.reloadSignal);
      this.reloadSignal = 0;
    }
    
    // Clear stored states and internal references
    this.originalStates.clear();
    this.originalGridState = null;
    this.defaultAdaptToSizeFunction = null;
    this.reloadingSig = 0;
    this.settings = null;
  }
  
  overriddenAdaptToSizeFunction(width, height) {
    return this._safeExecute(() => {
      const MainAppDisplay = this._getAppDisplay();
      if (!MainAppDisplay) return;

      // Safely get indicator height
      let indicatorHeight = 0;
      if (MainAppDisplay._pageIndicators && typeof MainAppDisplay._pageIndicators.get_preferred_height === 'function') {
        try {
          [, indicatorHeight] = MainAppDisplay._pageIndicators.get_preferred_height(-1);
        } catch (e) {
          console.warn(`[AppGridTuner] Failed to get indicator height: ${e.message}`);
        }
      }
      height -= indicatorHeight;

      let box = new Clutter.ActorBox({
        x2: width,
        y2: height,
      });
      
      // Safely apply theme nodes
      try {
        if (typeof MainAppDisplay.get_theme_node === 'function') {
          box = MainAppDisplay.get_theme_node().get_content_box(box);
        }
        if (MainAppDisplay._scrollView && typeof MainAppDisplay._scrollView.get_theme_node === 'function') {
          box = MainAppDisplay._scrollView.get_theme_node().get_content_box(box);
        }
        if (MainAppDisplay._grid && typeof MainAppDisplay._grid.get_theme_node === 'function') {
          box = MainAppDisplay._grid.get_theme_node().get_content_box(box);
        }
      } catch (e) {
        console.warn(`[AppGridTuner] Theme node application failed: ${e.message}`);
      }

      const availWidth = box.get_width();
      const availHeight = box.get_height();

      let pageHeight = availHeight;
      let hPadding = this.sidePadding || 0;
      // Allow negative padding to override system padding, but ensure reasonable minimum
      let pageWidth = availWidth - hPadding * 2;
      
      // Ensure pageWidth doesn't exceed available width by too much (safety check)
      if (hPadding < 0) {
        pageWidth = Math.min(pageWidth, availWidth * 1.2); // Max 20% over available width
      }

      if (MainAppDisplay._grid && MainAppDisplay._grid.layout_manager && typeof MainAppDisplay._grid.layout_manager.adaptToSize === 'function') {
        MainAppDisplay._grid.layout_manager.adaptToSize(pageWidth, pageHeight);
      }

      const vPadding = Math.floor((availHeight - (MainAppDisplay._grid?.layout_manager?.pageHeight || 0)) / 2);

      if (MainAppDisplay._scrollView) {
        MainAppDisplay._scrollView.content_padding = new Clutter.Margin({
          left: hPadding,
          right: hPadding,
          top: vPadding,
          bottom: vPadding,
        });
      }

      MainAppDisplay._availWidth = availWidth;
      MainAppDisplay._availHeight = availHeight;

      MainAppDisplay._pageIndicatorOffset = hPadding;
      MainAppDisplay._pageArrowOffset = Math.max(hPadding - 80, 0);
    });
  }  
    
}

export default class AppGridTunerExtension extends Extension {
  enable() {
    const settings = this.getSettings();
    appGridTuner = new AppGridTuner(settings);
    appGridTuner.startAppGridTuning();
  }

  disable() {
    appGridTuner?.destroy();
    appGridTuner = null;
  }
}
