'use strict';

import St from 'gi://St';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Gio from 'gi://Gio';
import Clutter from 'gi://Clutter';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import {Extension} from 'resource:///org/gnome/shell/extensions/extension.js';

export default class BoatmanExtension extends Extension {
    enable() {
        this._settings = this.getSettings();
        this._indicator = new BoatmanIndicator(this, this._settings);
        Main.panel.addToStatusArea(this.uuid, this._indicator);
    }

    disable() {
        this._indicator?.destroy();
        this._indicator = null;
        this._settings = null;
    }
}

class BoatmanIndicator extends PanelMenu.Button {
    static {
        GObject.registerClass(this);
    }

    constructor(extension, settings) {
        super(0.0, 'WinBoat');
        this._extension = extension;
        this._settings = settings;
        this._containerRuntime = null;
        
        // Store last stats to persist across menu opens/closes
        this._lastStats = {
            cpu: 'Click to check',
            memory: 'Click to check',
            ports: 'Click to check',
            timestamp: null
        };
        
        // Set up the panel icon with proper GNOME Shell styling
        this._icon = new St.Icon({
            icon_name: 'system-run-symbolic',
            style_class: 'system-status-icon',
            y_expand: false,
            y_align: Clutter.ActorAlign.CENTER
        });
        this.add_child(this._icon);
        
        // Set up the menu with proper GNOME Shell structure
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add status item
        this._statusItem = new PopupMenu.PopupMenuItem('Checking status...');
        this._statusItem.reactive = false;
        this.menu.addMenuItem(this._statusItem);
        
        // Add action buttons
        this._startButton = new PopupMenu.PopupMenuItem('Start WinBoat');
        this._stopButton = new PopupMenu.PopupMenuItem('Stop WinBoat');
        this._restartButton = new PopupMenu.PopupMenuItem('Restart WinBoat');
        
        this._startButton.connect('activate', () => this._startWinBoat());
        this._stopButton.connect('activate', () => this._stopWinBoat());
        this._restartButton.connect('activate', () => this._restartWinBoat());
        
        this.menu.addMenuItem(this._startButton);
        this.menu.addMenuItem(this._stopButton);
        this.menu.addMenuItem(this._restartButton);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add uptime info item
        this._uptimeItem = new PopupMenu.PopupMenuItem('Checking uptime...');
        this._uptimeItem.reactive = false;
        this.menu.addMenuItem(this._uptimeItem);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add stats items (hidden by default)
        this._cpuItem = new PopupMenu.PopupMenuItem('CPU: Click to check');
        this._cpuItem.reactive = true;
        this._cpuItem.connect('activate', () => this._updateStats());
        this.menu.addMenuItem(this._cpuItem);
        
        this._memoryItem = new PopupMenu.PopupMenuItem('Memory: Click to check');
        this._memoryItem.reactive = true;
        this._memoryItem.connect('activate', () => this._updateStats());
        this.menu.addMenuItem(this._memoryItem);
        
        this._portsItem = new PopupMenu.PopupMenuItem('Ports: Click to check');
        this._portsItem.reactive = true;
        this._portsItem.connect('activate', () => this._updateStats());
        this.menu.addMenuItem(this._portsItem);
        
        // Add refresh item
        const refreshItem = new PopupMenu.PopupMenuItem('Refresh Status');
        refreshItem.connect('activate', () => {
            this._updateStatus();
            // Also refresh stats if they are enabled
            if (this._settings.get_boolean('show-stats')) {
                this._updateStats();
            }
        });
        this.menu.addMenuItem(refreshItem);
        
        // Initial status update
        this._updateStatus();
        
        // Set up periodic updates using configured refresh interval
        this._scheduleStatusUpdates();

        // Connect to settings changes
        this._showUptimeSignalId = this._settings.connect('changed::show-uptime', () => this._updateStatus());
        this._showStatsSignalId = this._settings.connect('changed::show-stats', () => this._updateMenuVisibility());
        this._compactModeSignalId = this._settings.connect('changed::compact-mode', () => this._updateStatus());
    }

    _setStatusIcon(status) {
        let icon_name = null;

        switch (status) {
            case 'running':
                icon_name = 'emblem-ok-symbolic';
                break;
            case 'stopped':
                icon_name = 'media-playback-stop-symbolic';
                break;
            case 'booting':
                icon_name = 'system-run-symbolic';
                break;
            case 'docker_down':
            case 'error':
            default:
                icon_name = 'action-unavailable-symbolic';
                break;
        }

        // Try custom icons first, then fallback to symbolic
        try {
            const customIcons = {
                'running': 'boat.svg',
                'stopped': 'anchor.svg',
                'booting': 'life-guard.svg',
                'error': 'life-guard.svg',
                'docker_down': 'life-guard.svg'
            };
            
            const fileName = customIcons[status] || 'life-guard.svg';
            const file = this._extension.dir.get_child('icons').get_child(fileName);
            
            if (file && file.query_exists(null)) {
                const gicon = Gio.icon_new_for_string(file.get_path());
                this._icon.gicon = gicon;
                log(`WINBOAT: Loaded custom icon: ${fileName}`);
            } else {
                // Fallback to symbolic
                this._icon.icon_name = icon_name;
                log(`WINBOAT: Using symbolic icon: ${icon_name}`);
            }
        } catch (e) {
            logError(e, 'WINBOAT: Failed to load custom icon, using symbolic');
            // Fallback to symbolic
            this._icon.icon_name = icon_name;
        }
    }

    _scheduleStatusUpdates() {
        const interval = Math.max(1, this._settings.get_int('refresh-interval'));

        if (this._timeout) {
            GLib.Source.remove(this._timeout);
            this._timeout = null;
        }

        this._timeout = GLib.timeout_add_seconds(
            GLib.PRIORITY_DEFAULT,
            interval,
            () => {
                this._updateStatus();
                return GLib.SOURCE_CONTINUE;
            }
        );

        // React to changes in refresh-interval
        if (!this._refreshSignalId) {
            this._refreshSignalId = this._settings.connect('changed::refresh-interval', () => {
                this._scheduleStatusUpdates();
            });
        }
    }

    _getContainerRuntime() {
        if (this._containerRuntime)
            return this._containerRuntime;

        // Respect user override first
        const mode = this._settings.get_string('runtime-mode');
        if (mode === 'docker' || mode === 'podman') {
            try {
                const [ok] = GLib.spawn_command_line_sync(`${mode} ps`);
                if (ok) {
                    this._containerRuntime = mode;
                    return mode;
                }
            } catch (e) {
                // Fall through to auto-detect
            }
        }

        const runtimes = ['docker', 'podman'];

        for (const runtime of runtimes) {
            try {
                const [ok] = GLib.spawn_command_line_sync(`${runtime} ps`);
                if (ok) {
                    this._containerRuntime = runtime;
                    return runtime;
                }
            } catch (e) {
                // Ignore and try next runtime
            }
        }

        return null;
    }

    _updateStatus() {
        const previousStatus = this._lastStatus;
        let status = this._isBoatmanRunning();

        // Ensure we always have a well-formed status object
        if (!status || typeof status !== 'object' || !status.status) {
            status = { status: 'error' };
        }

        this._lastStatus = status.status;

        switch (status.status) {
            case 'running':
                this._setStatusIcon('running');
                const compactMode = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactMode ? 'Running' : 'WinBoat is running';
                this._startButton.visible = false;
                this._stopButton.visible = true;
                this._restartButton.visible = true;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    const uptime = this._getContainerUptime();
                    this._uptimeItem.label.text = compactMode ? uptime : `Uptime: ${uptime}`;
                }
                break;
                
            case 'booting':
                this._setStatusIcon('booting');
                const compactModeBoot = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactModeBoot ? 'Starting...' : 'WinBoat is starting...';
                this._startButton.visible = false;
                this._stopButton.visible = true;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    this._uptimeItem.label.text = compactModeBoot ? 'Starting...' : 'Starting...';
                }
                
                // Check status again in 5 seconds
                if (this._bootCheckTimeout) {
                    GLib.Source.remove(this._bootCheckTimeout);
                }
                this._bootCheckTimeout = GLib.timeout_add_seconds(
                    GLib.PRIORITY_DEFAULT,
                    5,
                    () => {
                        this._bootCheckTimeout = null;
                        this._updateStatus();
                        return GLib.SOURCE_REMOVE;
                    }
                );
                break;
                
            case 'docker_down':
                this._setStatusIcon('docker_down');
                const compactModeDocker = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactModeDocker ? 'Docker down' : 'Docker not running';
                this._startButton.visible = false;
                this._stopButton.visible = false;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    this._uptimeItem.label.text = compactModeDocker ? 'Docker down' : 'Docker unavailable';
                }
                break;
                
            case 'error':
                this._setStatusIcon('error');
                const compactModeError = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactModeError ? 'Error' : 'Error checking WinBoat status';
                this._startButton.visible = true;
                this._stopButton.visible = false;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    this._uptimeItem.label.text = compactModeError ? 'Unknown' : 'Status unknown';
                }
                break;

            case 'stopped':
            default:
                this._setStatusIcon('stopped');
                const compactModeStopped = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactModeStopped ? 'Stopped' : 'WinBoat is stopped';
                this._startButton.visible = true;
                this._stopButton.visible = false;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    this._uptimeItem.label.text = compactModeStopped ? 'Stopped' : 'Container stopped';
                }
        }
        
        // Update menu visibility based on settings
        this._updateMenuVisibility();
        
        this._maybeNotify(previousStatus, status.status);
        return status.status === 'running';
    }

    _updateMenuVisibility() {
        const showUptime = this._settings.get_boolean('show-uptime');
        const showStats = this._settings.get_boolean('show-stats');
        const compactMode = this._settings.get_boolean('compact-mode');

        // Show/hide uptime item
        if (this._uptimeItem) {
            this._uptimeItem.visible = showUptime;
        }

        // Show/hide stats items and separator
        if (this._cpuItem && this._memoryItem && this._portsItem) {
            this._cpuItem.visible = showStats;
            this._memoryItem.visible = showStats;
            this._portsItem.visible = showStats;
            
            // Restore last stats when showing stats
            if (showStats) {
                const timeAgo = this._formatTimestamp(this._lastStats.timestamp);
                
                if (compactMode) {
                    this._cpuItem.label.text = `CPU: ${this._lastStats.cpu}${timeAgo}`;
                    this._memoryItem.label.text = `Memory: ${this._lastStats.memory}${timeAgo}`;
                    this._portsItem.label.text = `Ports: ${this._lastStats.ports}${timeAgo}`;
                } else {
                    this._cpuItem.label.text = `CPU: ${this._lastStats.cpu}${timeAgo}`;
                    this._memoryItem.label.text = `Memory: ${this._lastStats.memory}${timeAgo}`;
                    this._portsItem.label.text = `Ports: ${this._lastStats.ports}${timeAgo}`;
                }
            }
        }

        // Update button labels in compact mode
        if (compactMode) {
            this._startButton.label.text = 'Start';
            this._stopButton.label.text = 'Stop';
            this._restartButton.label.text = 'Restart';
        } else {
            this._startButton.label.text = 'Start WinBoat';
            this._stopButton.label.text = 'Stop WinBoat';
            this._restartButton.label.text = 'Restart WinBoat';
        }
    }

    _maybeNotify(previousStatus, currentStatus) {
        if (!this._settings.get_boolean('enable-notifications'))
            return;

        if (!previousStatus || previousStatus === currentStatus)
            return;

        let message = null;

        if (currentStatus === 'running') {
            message = 'WinBoat is now running.';
        } else if (currentStatus === 'stopped') {
            message = 'WinBoat has stopped.';
        } else if (currentStatus === 'docker_down' || currentStatus === 'error') {
            message = 'WinBoat extension: container runtime or WinBoat status error.';
        }

        if (message)
            Main.notify('WinBoat', message);
    }

    _isBoatmanRunning() {
        try {
            const runtime = this._getContainerRuntime();
            if (!runtime) {
                if (this._settings.get_boolean('debug-mode')) {
                    log('WINBOAT: Container runtime not accessible');
                }
                return { status: 'docker_down' };
            }

            // Use docker inspect for reliable status check
            const cmd = `${runtime} inspect WinBoat --format "{{.State.Status}}"`;
            if (this._settings.get_boolean('debug-mode')) {
                log(`WINBOAT: Running command: ${cmd}`);
            }
            const [success, stdout, stderr] = GLib.spawn_command_line_sync(cmd);

            if (this._settings.get_boolean('debug-mode')) {
                log(`WINBOAT: Command result: success=${success}, stdout=${stdout ? stdout.toString().trim() : 'null'}, stderr=${stderr ? stderr.toString().trim() : 'null'}`);
            }

            if (!success || !stdout) {
                if (this._settings.get_boolean('debug-mode')) {
                    log('WINBOAT: Container not found or inspect failed');
                }
                return { status: 'stopped' };
            }

            const status = stdout.toString().trim();
            if (this._settings.get_boolean('debug-mode')) {
                log(`WINBOAT: Container status: ${status}`);
            }

            if (status === 'running') {
                return { status: 'running', containerRunning: true };
            } else {
                return { status: 'stopped' };
            }
                
        } catch (e) {
            logError(e, 'WINBOAT: Error checking status');
            return { status: 'error' };
        }
    }

    _startWinBoat() {
        try {
            log('WINBOAT: Starting container...');
            this._statusItem.label.text = 'Starting...';
            
            // Disable action buttons during operation
            this._startButton.reactive = false;
            this._stopButton.reactive = false;
            this._restartButton.reactive = false;

            const runtime = this._getContainerRuntime();
            const [success, stdout, stderr] = GLib.spawn_command_line_sync(
                `${runtime} start WinBoat`
            );

            if (!success) {
                log('WINBOAT: Start failed');
                this._statusItem.label.text = 'Start failed';
                this._maybeNotify('Failed to start WinBoat container');
                
                // Re-enable buttons on failure
                this._startButton.reactive = true;
                this._stopButton.reactive = false;
                this._restartButton.reactive = false;
                return;
            }

            log('WINBOAT: Start command executed');
            this._maybeNotify('WinBoat container started');
            
            // Force immediate status update using GLib timeout
            GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                this._updateStatus();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            logError(e, 'WINBOAT: Error starting container');
            this._statusItem.label.text = 'Error starting';
            
            // Re-enable buttons on error
            this._startButton.reactive = true;
            this._stopButton.reactive = false;
            this._restartButton.reactive = false;
        }
    }

    _stopWinBoat() {
        try {
            if (this._settings.get_boolean('debug-mode')) {
                log('WINBOAT: Stopping container...');
            }
            this._statusItem.label.text = 'Stopping...';
            
            // Disable action buttons during operation
            this._startButton.reactive = false;
            this._stopButton.reactive = false;
            this._restartButton.reactive = false;

            const runtime = this._getContainerRuntime();
            const [success, stdout, stderr] = GLib.spawn_command_line_sync(
                `${runtime} stop WinBoat`
            );

            if (!success) {
                if (this._settings.get_boolean('debug-mode')) {
                    log('WINBOAT: Stop failed');
                }
                this._statusItem.label.text = 'Stop failed';
                this._maybeNotify('Failed to stop WinBoat container');
                
                // Re-enable buttons on failure
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                return;
            }

            if (this._settings.get_boolean('debug-mode')) {
                log('WINBOAT: Stop command executed');
            }
            
            // Force immediate status update using GLib timeout
            GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                this._updateStatus();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            logError(e, 'WINBOAT: Error stopping container');
            this._startButton.reactive = true;
            this._stopButton.reactive = true;
            this._restartButton.reactive = true;
            return;
        }
    }

    _restartWinBoat() {
        try {
            if (this._settings.get_boolean('debug-mode')) {
                log('WINBOAT: Restarting container...');
            }
            this._statusItem.label.text = 'Restarting...';
            
            // Disable action buttons during operation
            this._startButton.reactive = false;
            this._stopButton.reactive = false;
            this._restartButton.reactive = false;

            const runtime = this._getContainerRuntime();
            const [success, stdout, stderr] = GLib.spawn_command_line_sync(
                `${runtime} restart WinBoat`
            );

            if (!success) {
                if (this._settings.get_boolean('debug-mode')) {
                    log('WINBOAT: Restart failed');
                }
                this._statusItem.label.text = 'Restart failed';
                this._maybeNotify('Failed to restart WinBoat container');
                
                // Re-enable buttons on failure
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                return;
            }

            if (this._settings.get_boolean('debug-mode')) {
                log('WINBOAT: Restart command executed');
            }
            this._maybeNotify('WinBoat container restarted');
            
            // Force immediate status update using GLib timeout
            GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                this._updateStatus();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            logError(e, 'WINBOAT: Error restarting container');
            this._startButton.reactive = true;
            this._stopButton.reactive = true;
            this._restartButton.reactive = true;
            return;
        }
    }

    _getContainerUptime() {
        try {
            const runtime = this._getContainerRuntime();
            if (!runtime)
                return 'Unknown';

            const [success, stdout] = GLib.spawn_command_line_sync(
                `${runtime} inspect WinBoat --format "{{.State.StartedAt}}"`
            );

            if (!success || !stdout) {
                return 'Unknown';
            }

            const startedAt = stdout.toString().trim();
            const startTime = new Date(startedAt).getTime();
            const now = Date.now();
            const uptimeMs = now - startTime;

            if (uptimeMs < 0)
                return 'Unknown';

            // Format uptime nicely
            const seconds = Math.floor(uptimeMs / 1000);
            const minutes = Math.floor(seconds / 60);
            const hours = Math.floor(minutes / 60);
            const days = Math.floor(hours / 24);

            if (days > 0) {
                return `${days}d ${hours % 24}h ${minutes % 60}m`;
            } else if (hours > 0) {
                return `${hours}h ${minutes % 60}m`;
            } else if (minutes > 0) {
                return `${minutes}m ${seconds % 60}s`;
            } else {
                return `${seconds}s`;
            }

        } catch (e) {
            logError(e, 'WINBOAT: Error getting uptime');
            return 'Error';
        }
    }

    _getContainerStats() {
        try {
            const runtime = this._getContainerRuntime();
            if (!runtime) {
                return { cpu: 'Runtime unavailable', memory: 'Runtime unavailable', ports: 'Runtime unavailable' };
            }

            // Get container stats using docker stats --no-stream
            const cmd = `${runtime} stats WinBoat --no-stream --format "{{.CPUPerc}}\t{{.MemUsage}}"`;
            const [success, stdout, stderr] = GLib.spawn_command_line_sync(cmd);

            if (!success || !stdout) {
                log(`WINBOAT: Stats command failed: ${stderr ? stderr.toString().trim() : 'No output'}`);
                return { cpu: 'Stats unavailable', memory: 'Stats unavailable', ports: 'Stats unavailable' };
            }

            const output = stdout.toString().trim();
            log(`WINBOAT: Raw stats output: "${output}"`);
            
            if (!output) {
                return { cpu: 'No data', memory: 'No data', ports: 'No data' };
            }

            // Split by whitespace and filter out empty parts
            const parts = output.split(/\s+/).filter(part => part.length > 0);
            
            if (parts.length < 2) {
                log(`WINBOAT: Stats parse error - not enough parts: ${parts.length}`);
                return { cpu: 'Parse error', memory: 'Parse error', ports: 'Parse error' };
            }

            // First part should be CPU percentage, second should be memory usage
            const cpuPercent = parts[0];
            const memUsage = parts[1];

            // Validate CPU format (should end with %)
            if (!cpuPercent.includes('%')) {
                log(`WINBOAT: Invalid CPU format: ${cpuPercent}`);
                return { cpu: 'Invalid CPU', memory: memUsage, ports: 'Ports unavailable' };
            }

            // Extract only the used memory from "used / total" format
            let usedMemory = memUsage;
            if (memUsage.includes('/')) {
                usedMemory = memUsage.split('/')[0].trim();
            }

            // Get ports
            const ports = this._getContainerPorts();

            log(`WINBOAT: Parsed stats - CPU: ${cpuPercent}, Memory: ${usedMemory}, Ports: ${ports}`);

            return { 
                cpu: cpuPercent, 
                memory: usedMemory,
                ports: ports
            };

        } catch (e) {
            logError(e, 'WINBOAT: Error getting stats');
            return { cpu: 'Error', memory: 'Error', ports: 'Error' };
        }
    }

    _getContainerPorts() {
        try {
            const runtime = this._getContainerRuntime();
            if (!runtime) {
                return 'Runtime unavailable';
            }

            // Get container ports (filter by container name)
            const cmd = `${runtime} ps --filter "name=WinBoat" --format "{{.Ports}}"`;
            const [success, stdout, stderr] = GLib.spawn_command_line_sync(cmd);

            if (!success || !stdout) {
                log(`WINBOAT: Ports command failed: ${stderr ? stderr.toString().trim() : 'No output'}`);
                return 'Ports unavailable';
            }

            const portsOutput = stdout.toString().trim();
            log(`WINBOAT: Raw ports output: "${portsOutput}"`);
            
            if (!portsOutput || portsOutput === '<nil>' || portsOutput === '') {
                return 'No ports';
            }

            // Parse ports and format nicely
            const portMappings = portsOutput.split(',').map(port => port.trim()).filter(port => port.length > 0);
            
            if (portMappings.length === 0) {
                return 'No ports';
            }

            // Extract just the external ports for cleaner display
            const externalPorts = portMappings.map(mapping => {
                const match = mapping.match(/:(\d+)->/);
                return match ? match[1] : mapping.split('->')[0];
            }).filter(port => port && port !== '');

            // Remove duplicates
            const uniquePorts = [...new Set(externalPorts)];

            if (uniquePorts.length === 0) {
                return 'Internal ports';
            }

            // Format ports - split into two lines if there are many
            const portString = uniquePorts.join(', ');
            
            // If port string is too long, split into two lines
            if (portString.length > 30) {
                const midPoint = Math.ceil(uniquePorts.length / 2);
                const firstLine = uniquePorts.slice(0, midPoint).join(', ');
                const secondLine = uniquePorts.slice(midPoint).join(', ');
                return `${firstLine}\n${secondLine}`;
            }

            return portString;

        } catch (e) {
            logError(e, 'WINBOAT: Error getting ports');
            return 'Error';
        }
    }

    _formatTimestamp(timestamp) {
        if (!timestamp) return '';
        
        const now = Date.now();
        const diff = now - timestamp;
        
        if (diff < 60000) { // Less than 1 minute
            return ' (just now)';
        } else if (diff < 3600000) { // Less than 1 hour
            const minutes = Math.floor(diff / 60000);
            return ` (${minutes}m ago)`;
        } else {
            const hours = Math.floor(diff / 3600000);
            return ` (${hours}h ago)`;
        }
    }

    _updateStats() {
        try {
            if (this._settings.get_boolean('debug-mode')) {
                log('WINBOAT: Updating stats on user request...');
            }
            
            // Show loading state immediately
            this._cpuItem.label.text = 'CPU: Checking...';
            this._memoryItem.label.text = 'Memory: Checking...';
            this._portsItem.label.text = 'Ports: Checking...';
            
            // Update stats with a small delay to prevent menu from closing
            GLib.timeout_add(GLib.PRIORITY_DEFAULT, 50, () => {
                try {
                    // Get stats
                    const stats = this._getContainerStats();
                    
                    // Store last stats for persistence
                    this._lastStats = {
                        cpu: stats.cpu,
                        memory: stats.memory,
                        ports: stats.ports,
                        timestamp: Date.now()
                    };
                    
                    // Update display
                    this._cpuItem.label.text = `CPU: ${stats.cpu}`;
                    this._memoryItem.label.text = `Memory: ${stats.memory}`;
                    this._portsItem.label.text = `Ports: ${stats.ports}`;
                    
                    if (this._settings.get_boolean('debug-mode')) {
                        log(`WINBOAT: Stats updated - CPU: ${stats.cpu}, Memory: ${stats.memory}, Ports: ${stats.ports}`);
                    }
                    
                } catch (e) {
                    logError(e, 'WINBOAT: Error updating stats');
                    this._cpuItem.label.text = 'CPU: Error';
                    this._memoryItem.label.text = 'Memory: Error';
                    this._portsItem.label.text = 'Ports: Error';
                }
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            logError(e, 'WINBOAT: Error updating stats');
            this._cpuItem.label.text = 'CPU: Error';
            this._memoryItem.label.text = 'Memory: Error';
            this._portsItem.label.text = 'Ports: Error';
        }
    }

    destroy() {
        // Clean up timeouts
        if (this._timeout) {
            GLib.Source.remove(this._timeout);
            this._timeout = null;
        }
        if (this._bootCheckTimeout) {
            GLib.Source.remove(this._bootCheckTimeout);
            this._bootCheckTimeout = null;
        }
        
        // Disconnect all signals
        if (this._refreshSignalId) {
            this._settings.disconnect(this._refreshSignalId);
            this._refreshSignalId = null;
        }
        if (this._showUptimeSignalId) {
            this._settings.disconnect(this._showUptimeSignalId);
            this._showUptimeSignalId = null;
        }
        if (this._showStatsSignalId) {
            this._settings.disconnect(this._showStatsSignalId);
            this._showStatsSignalId = null;
        }
        if (this._compactModeSignalId) {
            this._settings.disconnect(this._compactModeSignalId);
            this._compactModeSignalId = null;
        }
        
        // Call parent destroy
        super.destroy();
    }
}