// SPDX-License-Identifier: GPL-3.0-or-later

import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import Meta from 'gi://Meta';
import Shell from 'gi://Shell';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';

export default class AppearanceKeeperExtension extends Extension {
    constructor(metadata) {
        super(metadata);
        this._manager = null;
    }

    enable() {
        this._manager = new AppearanceKeeperManager(this.getSettings());
        this._manager.start();
    }

    disable() {
        if (this._manager) {
            this._manager.stop();
            this._manager = null;
        }
    }
}

class AppearanceKeeperManager {
    constructor(settings) {
        // Public settings object for storing extension preferences
        this._settings = settings;

        // GSettings wrappers
        this._interfaceSettings = null;    // org.gnome.desktop.interface
        this._userThemeSettings = null;    // org.gnome.shell.extensions.user-theme (optional)
        this._backgroundSettings = null;   // org.gnome.desktop.background

        // Stored wallpaper URIs for light/dark pairing logic
        this._storedWallpapers = { light: null, dark: null };

        // GSettings handlers so we can disconnect on stop
        this._handlers = [];

        // Internals
        this._DEBUG = false;
        this._suspendSave = false;
        this._debounceTimers = {};

        // Schema ids
        this._SCHEMA_BACKGROUND = 'org.gnome.desktop.background';
        this._SCHEMA_INTERFACE = 'org.gnome.desktop.interface';
        this._SCHEMA_USER_THEME = 'org.gnome.shell.extensions.user-theme';

        // Keybinding id (from extension A)
        this._keybindingId = 'dark-light-toggle';
    }

    // --- Lifecycle Management ---

    start() {
        this._log('Starting extension');

        if (!this._initializeServices()) {
            this._log('Service initialization failed', 'error');
            this.stop();
            return;
        }

        this._initializeMonitoring();
        this._registerKeybinding();

        this._log('Extension started successfully');
    }

    stop() {
        this._log('Stopping extension');

        // Disconnect handlers and clear timers
        this._cleanupAllHandlers();
        this._cleanupDebounceTimers();
        this._cleanupKeybinding();

        // Reset references
        this._resetAllSettings();

        this._log('Extension stopped');
    }

    // --- Logging ---

    _log(message, level = 'debug') {
        const prefix = '[AppearanceKeeper]';
        if (level === 'error') console.error(`${prefix} ERROR: ${message}`);
        else if (level === 'warning') console.warn(`${prefix} WARNING: ${message}`);
        else if (this._DEBUG) console.debug(`${prefix} ${message}`);
    }

    // --- Service Initialization ---

    _initializeServices() {
        return this._initMainSettings() &&
               this._initBackgroundSettings() &&
               this._initUserThemeSettings();
    }

    _initMainSettings() {
        this._interfaceSettings = new Gio.Settings({ schema_id: this._SCHEMA_INTERFACE });
        this._log('Main settings initialized');
        return true;
    }

    _initBackgroundSettings() {
        this._backgroundSettings = new Gio.Settings({ schema_id: this._SCHEMA_BACKGROUND });
        this._log('Background settings initialized');
        return true;
    }

    _initUserThemeSettings() {
        const schemaSource = Gio.SettingsSchemaSource.get_default();
        const schema = schemaSource.lookup(this._SCHEMA_USER_THEME, true);

        if (schema) {
            this._userThemeSettings = new Gio.Settings({ schema_id: this._SCHEMA_USER_THEME });
            this._log('User theme settings initialized');
        } else {
            this._log('User theme extension not available', 'debug');
        }
        return true;
    }

    // --- Monitoring Setup ---

    _initializeMonitoring() {
        this._loadInitialWallpapers();
        this._setupWallpaperMonitoring();
        this._setupThemeMonitoring();
        this._setupStyleMonitoring();
    }

    _loadInitialWallpapers() {
        this._storedWallpapers.light = this._backgroundSettings.get_string('picture-uri');
        this._storedWallpapers.dark = this._backgroundSettings.get_string('picture-uri-dark');
        this._log('Initial wallpapers loaded');
    }

    _setupWallpaperMonitoring() {
        const handler = this._backgroundSettings.connect('changed', (settings, key) => {
            if (key === 'picture-uri' || key === 'picture-uri-dark') {
                this._handleWallpaperChange(key);
            }
        });
        this._storeHandler(this._backgroundSettings, handler);
        this._log('Wallpaper monitoring set up');
    }

    _setupThemeMonitoring() {
        const themeKeys = ['gtk-theme', 'icon-theme', 'cursor-theme', 'accent-color'];
        themeKeys.forEach(key => {
            const handler = this._interfaceSettings.connect(`changed::${key}`, () => {
                this._saveThemeParameter(key);
            });
            this._storeHandler(this._interfaceSettings, handler);
        });

        if (this._userThemeSettings) {
            const shellHandler = this._userThemeSettings.connect('changed::name', () => {
                this._saveThemeParameter('shell-theme');
            });
            this._storeHandler(this._userThemeSettings, shellHandler);
        }

        this._log('Theme monitoring set up');
    }

    _setupStyleMonitoring() {
        const handler = this._interfaceSettings.connect('changed::color-scheme', () => {
            this._handleColorSchemeChange();
        });
        this._storeHandler(this._interfaceSettings, handler);
        this._log('Style (color-scheme) monitoring set up');
    }

    // --- Handler Management ---

    _storeHandler(settings, handlerId) {
        this._handlers.push({ settings, id: handlerId });
    }

    _cleanupAllHandlers() {
        this._handlers.forEach(handler => {
            if (handler?.settings && handler.id) {
                handler.settings.disconnect(handler.id);
            }
        });
        this._handlers = [];
    }

    // --- Debounce Timer Management ---

    _cleanupDebounceTimers() {
        for (const key in this._debounceTimers) {
            if (this._debounceTimers[key]) {
                GLib.source_remove(this._debounceTimers[key]);
            }
        }
        this._debounceTimers = {};
    }

    // --- Reset Settings ---
    _resetAllSettings() {
        this._interfaceSettings = null;
        this._userThemeSettings = null;
        this._backgroundSettings = null;
        this._storedWallpapers = { light: null, dark: null };
    }

    // --- Settings Utilities ---

    _getSetting(settings, key, defaultValue = '') {
        if (!settings) return defaultValue;
        return settings.get_string(key) || defaultValue;
    }

    _setSetting(settings, key, value) {
        if (!settings || value === null) return;
        const current = settings.get_string(key);
        if (current !== value) settings.set_string(key, value);
    }

    _getValidatedSetting(settings, key, defaultValue = '') {
        const value = this._getSetting(settings, key, defaultValue);
        if (typeof value !== 'string') return defaultValue;
        if (value.length > 100) return defaultValue;
        return value;
    }

    // --- Theme Management ---

    _saveThemeParameter(parameter) {
        if (this._suspendSave || !this._settings) return;

        const colorScheme = this._getSetting(this._interfaceSettings, 'color-scheme', 'default');
        const isDark = colorScheme.includes('dark');
        const prefix = isDark ? 'dark' : 'light';
        let value = '';

        switch (parameter) {
            case 'gtk-theme':
            case 'icon-theme':
            case 'cursor-theme':
            case 'accent-color':
                value = this._getValidatedSetting(this._interfaceSettings, parameter);
                break;
            case 'shell-theme':
                value = this._getSetting(this._userThemeSettings, 'name', '');
                break;
            default:
                this._log(`Unknown theme parameter: ${parameter}`, 'warning');
                return;
        }

        const key = parameter === 'shell-theme' ? `${prefix}-shell-theme` : `${prefix}-${parameter}`;
        this._setSetting(this._settings, key, value);
        this._log(`Saved ${parameter} for ${prefix} mode: ${value || '(default)'}`);
    }

    async _setShellTheme(themeName) {
        if (!this._userThemeSettings) return;
        if (!themeName) {
            this._userThemeSettings.reset('name');
            this._log('Shell theme reset to default');
        } else {
            this._userThemeSettings.set_string('name', themeName);
            this._log(`Shell theme applied: ${themeName}`);
        }
    }

    // --- Wallpaper Logic with Debounce ---

    _handleWallpaperChange(changedKey) {
        // Debounce: cancel existing timers for this key
        if (this._debounceTimers[changedKey]) {
            GLib.source_remove(this._debounceTimers[changedKey]);
        }

        // Schedule new debounce timer
        this._debounceTimers[changedKey] = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 50, () => {
            this._processWallpaperChange(changedKey);
            delete this._debounceTimers[changedKey];
            return GLib.SOURCE_REMOVE;
        });
    }

    _processWallpaperChange(changedKey) {
        const newValue = this._backgroundSettings.get_string(changedKey) || '';
        const colorScheme = this._getSetting(this._interfaceSettings, 'color-scheme', 'default');
        const isDark = colorScheme.includes('dark');

        this._log(`Wallpaper changed: ${changedKey} = ${newValue.substring(0, 100)}...`);

        if (newValue.endsWith('.xml')) {
            this._log('XML wallpaper detected, storing directly');
            this._storedWallpapers[changedKey === 'picture-uri' ? 'light' : 'dark'] = newValue;
            return;
        }

        if (this._isSpecialBackgroundFile(newValue)) {
            this._handleSpecialBackgroundFile(changedKey, newValue, isDark);
            return;
        }

        this._handleRegularWallpaper(changedKey, newValue, isDark);
    }

    _isSpecialBackgroundFile(uri) {
        if (!uri) return false;
        return uri.includes('/.config/background');
    }

    _isPairedWallpaper(uri) {
        if (!uri) return false;
        const filename = uri.replace('file://', '').split('/').pop() || '';
        const hasThemeSuffix = /[-_](l|d|light|dark|day|night)\.[a-zA-Z0-9]+$/.test(filename);
        const isXmlFile = filename.endsWith('.xml');
        return hasThemeSuffix || isXmlFile;
    }

    _handleSpecialBackgroundFile(changedKey, newValue, isDark) {
        const userHome = GLib.get_home_dir();
        const backgroundPath = `${userHome}/.config/background`;
        const extension = this._getFileExtension(backgroundPath);
        const isLightChange = changedKey === 'picture-uri';

        this._log(`Processing special background file for ${isDark ? 'dark' : 'light'} mode`);

        if ((isLightChange && !isDark) || (!isLightChange && isDark)) {
            const suffix = isDark ? '-dark' : '-light';
            const newPath = `${backgroundPath}${suffix}${extension}`;
            const fileOrig = Gio.File.new_for_path(backgroundPath);
            const fileNew = Gio.File.new_for_path(newPath);

            if (fileOrig.query_exists(null)) {
                fileOrig.copy(fileNew, Gio.FileCopyFlags.OVERWRITE, null, null);
                const newUri = `file://${newPath}`;
                this._backgroundSettings.set_string(changedKey, newUri);
                this._storedWallpapers[isDark ? 'dark' : 'light'] = newUri;
            }
        } else {
            const restoreKey = changedKey === 'picture-uri' ? 'light' : 'dark';
            this._backgroundSettings.set_string(changedKey, this._storedWallpapers[restoreKey]);
        }
    }

    _handleRegularWallpaper(changedKey, newValue, isDark) {
        const isPaired = this._isPairedWallpaper(newValue);

        if (isPaired) {
            this._storedWallpapers[changedKey === 'picture-uri' ? 'light' : 'dark'] = newValue;
        } else {
            if (changedKey === 'picture-uri' && isDark) {
                // A light wallpaper was changed while in dark mode: revert light wallpaper to stored
                this._backgroundSettings.set_string('picture-uri', this._storedWallpapers.light);
            } else if (changedKey === 'picture-uri-dark' && !isDark) {
                // A dark wallpaper was changed while in light mode: revert dark wallpaper to stored
                this._backgroundSettings.set_string('picture-uri-dark', this._storedWallpapers.dark);
            } else {
                this._storedWallpapers[changedKey === 'picture-uri' ? 'light' : 'dark'] = newValue;
            }
        }
    }

    _getFileExtension(filePath) {
        const basename = filePath.split('/').pop() || '';
        const lastDotIndex = basename.lastIndexOf('.');
        return lastDotIndex !== -1 ? basename.slice(lastDotIndex) : '.png';
    }

    // --- Color Scheme Handling ---

    _handleColorSchemeChange() {
        const colorScheme = this._getSetting(this._interfaceSettings, 'color-scheme', 'default');
        const isDark = colorScheme.includes('dark');
        this._log(`Color scheme changed to: ${colorScheme}`);
        this._applyThemeForScheme(isDark);
    }

    async _applyThemeForScheme(isDark) {
        if (!this._settings) return;
        this._suspendSave = true;
        const prefix = isDark ? 'dark' : 'light';

        const themes = {
            gtk: this._getValidatedSetting(this._settings, `${prefix}-gtk-theme`),
            shell: this._getValidatedSetting(this._settings, `${prefix}-shell-theme`),
            icon: this._getValidatedSetting(this._settings, `${prefix}-icon-theme`),
            cursor: this._getValidatedSetting(this._settings, `${prefix}-cursor-theme`),
            accent: this._getValidatedSetting(this._settings, `${prefix}-accent-color`)
        };

        this._setSetting(this._interfaceSettings, 'gtk-theme', themes.gtk);
        this._setSetting(this._interfaceSettings, 'icon-theme', themes.icon);
        this._setSetting(this._interfaceSettings, 'cursor-theme', themes.cursor);
        this._setSetting(this._interfaceSettings, 'accent-color', themes.accent);

        const currentShell = this._getSetting(this._userThemeSettings, 'name', '');
        if (themes.shell !== null && themes.shell !== currentShell) {
            await this._setShellTheme(themes.shell);
        }

        this._log(`Applied ${prefix} theme settings`);
        this._log(`Shell theme: ${themes.shell || '(default)'}'`);

        this._suspendSave = false;
    }

    // --- Keybinding Management ---

    _registerKeybinding() {
        Main.wm.addKeybinding(
            this._keybindingId,
            this._settings,
            Meta.KeyBindingFlags.IGNORE_AUTOREPEAT,
            Shell.ActionMode.NORMAL | Shell.ActionMode.OVERVIEW,
            this._toggleColorScheme.bind(this)
        );
        this._log('Keybinding registered');
    }

    _cleanupKeybinding() {
        Main.wm.removeKeybinding(this._keybindingId);
        this._log('Keybinding removed');
    }

    _toggleColorScheme() {
        const current = this._interfaceSettings.get_string('color-scheme');
        const newScheme = current === 'prefer-dark' ? 'default' : 'prefer-dark';
        this._interfaceSettings.set_string('color-scheme', newScheme);
        this._log(`Toggled color scheme: ${current} → ${newScheme}`);
    }
}
