import Clutter from 'gi://Clutter';
import GObject from 'gi://GObject';
import St from 'gi://St';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';

import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import {Extension} from 'resource:///org/gnome/shell/extensions/extension.js';

const SILENT_FRAMES_THRESHOLD = 10;
const MIN_ACTIVE_HEIGHT = 4;
const MIN_HEIGHT = 2;
const BAR_STYLE_BASE = 'width: 4px; margin: 0 1px;';

let VisualizerButton = GObject.registerClass(
class VisualizerButton extends PanelMenu.Button {
    _init(settings, extension) {
        super._init(0.0, 'SoundBar');

        this._settings = settings;
        this._extension = extension;

        this._bottomPadding = this._settings.get_int('bottom-padding');

        this._box = new St.BoxLayout({
            style_class: 'audio-visual-container',
            style: `padding-bottom: ${this._bottomPadding}px;`,
            reactive: false,
            y_align: Clutter.ActorAlign.END,
            y_expand: false
        });
        this.add_child(this._box);

        // Add preferences menu item
        this._prefsItem = new PopupMenu.PopupMenuItem('Preferences');
        this._prefsItem.connect('activate', () => {
            this._extension.openPreferences();
        });
        this.menu.addMenuItem(this._prefsItem);

        this._numBars = this._settings.get_int('bar-count');
        this._bars = [];
        this._maxHeight = this._settings.get_int('max-height');
        this._barColor = this._settings.get_string('bar-color');
        this._useGradient = this._settings.get_boolean('use-gradient');
        this._gradientColor = this._settings.get_string('gradient-color');
        this._sensitivity = this._settings.get_int('sensitivity');
        this._framerate = this._settings.get_int('framerate');
        this._alphaRise = this._settings.get_double('alpha-rise');
        this._alphaFall = this._settings.get_double('alpha-fall');
        this._noiseFloor = this._settings.get_int('noise-floor');
        this._silenceZeroFrames = this._settings.get_int('silence-zero-frames');

        this._settingsChangedIds = [
            this._settings.connect('changed::bar-count', () => {
                this._rebuildBars();
                this._restartCava();
            }),
            this._settings.connect('changed::bar-color', () => this._updateBarColors()),
            this._settings.connect('changed::use-gradient', () => this._updateBarColors()),
            this._settings.connect('changed::gradient-color', () => this._updateBarColors()),
            this._settings.connect('changed::max-height', () => {
                this._maxHeight = this._settings.get_int('max-height');
            }),
            this._settings.connect('changed::hide-when-silent', () => {
                this._updateVisibility();
            }),
            this._settings.connect('changed::sensitivity', () => {
                this._sensitivity = this._settings.get_int('sensitivity');
                this._restartCava();
            }),
            this._settings.connect('changed::framerate', () => {
                this._framerate = this._settings.get_int('framerate');
                this._restartCava();
            }),
            this._settings.connect('changed::alpha-rise', () => {
                this._alphaRise = this._settings.get_double('alpha-rise');
            }),
            this._settings.connect('changed::alpha-fall', () => {
                this._alphaFall = this._settings.get_double('alpha-fall');
            }),
            this._settings.connect('changed::noise-floor', () => {
                this._noiseFloor = this._settings.get_int('noise-floor');
            }),
            this._settings.connect('changed::silence-zero-frames', () => {
                this._silenceZeroFrames = this._settings.get_int('silence-zero-frames');
            }),
            this._settings.connect('changed::bottom-padding', () => {
                this._bottomPadding = this._settings.get_int('bottom-padding');
                this._box.set_style(`padding-bottom: ${this._bottomPadding}px;`);
            })
        ];

        this._hideWhenSilent = this._settings.get_boolean('hide-when-silent');
        this._settingsChangedIds.push(
            this._settings.connect('changed::hide-when-silent', () => {
                this._hideWhenSilent = this._settings.get_boolean('hide-when-silent');
                this._updateVisibility();
            })
        );

        this._buildBars();
        this._prevHeights = new Array(this._numBars).fill(MIN_HEIGHT);
        this._silentFrames = 0;
        this._bins = new Array(this._numBars);

        this._procPid = null;
        this._stdout = null;
        this._stderr = null;
        this._stderrStream = null;
        this._stdoutCancellable = null;
        this._rawBuffer = new Uint8Array(8192);
        this._bufferUsed = 0;
        this._tmpConfigPath = null;
        this._startCava();
    }

    _getBarColor(index) {
        return this._useGradient && (index % 2 === 1) ? this._gradientColor : this._barColor;
    }

    _buildBars() {
        for (let i = 0; i < this._numBars; i++) {
            let bar = new St.Bin({
                style: `background-color: ${this._getBarColor(i)}; ${BAR_STYLE_BASE}`,
                y_align: Clutter.ActorAlign.END
            });
            bar.height = MIN_HEIGHT;
            this._box.add_child(bar);
            this._bars.push(bar);
        }
    }

    _rebuildBars() {
        this._bars.forEach(bar => bar.destroy());
        this._bars = [];
        this._numBars = this._settings.get_int('bar-count');
        this._buildBars();
        this._prevHeights = new Array(this._numBars).fill(MIN_HEIGHT);
        this._bins = new Array(this._numBars);
    }

    _updateBarColors() {
        this._barColor = this._settings.get_string('bar-color');
        this._useGradient = this._settings.get_boolean('use-gradient');
        this._gradientColor = this._settings.get_string('gradient-color');
        for (let i = 0; i < this._bars.length; i++) {
            this._bars[i].set_style(`background-color: ${this._getBarColor(i)}; ${BAR_STYLE_BASE}`);
        }
    }

    _updateVisibility() {
        this._box.visible = !this._hideWhenSilent || this._silentFrames < SILENT_FRAMES_THRESHOLD;
    }

    _restartCava() {
        this._stopCava();
        this._startCava();
    }

    _startCava() {
        if (this._procPid) {
            return;
        }

        try {
            if (!GLib.find_program_in_path('cava')) {
                this._box.visible = false;
                return;
            }
            let tmpDir = GLib.get_tmp_dir();
            let tmpConfig = tmpDir + `/soundbar-cava-config-${GLib.get_monotonic_time()}`;
            let cfg = `[general]\n` +
                      `bars = ${this._numBars}\n` +
                      `framerate = ${this._framerate}\n` +
                      `sensitivity = ${this._sensitivity}\n` +
                      `\n[input]\n` +
                      `method = pulse\n` +
                      `source = auto\n` +
                      `\n[output]\n` +
                      `method = raw\n` +
                      `bit_format = 16bit\n` +
                      `channels = mono\n` +
                      `raw_target = /dev/stdout\n`;
            GLib.file_set_contents(tmpConfig, cfg);
            this._tmpConfigPath = tmpConfig;

            let argv = ['cava', '-p', tmpConfig];
            let flags = GLib.SpawnFlags.SEARCH_PATH | GLib.SpawnFlags.DO_NOT_REAP_CHILD;
            let [ok, pid, stdinFd, stdoutFd, stderrFd] = GLib.spawn_async_with_pipes(
                null, argv, null, flags, null
            );
            if (!ok)
                return;

            this._procPid = pid;
            this._stdout = new Gio.UnixInputStream({ fd: stdoutFd, close_fd: true });
            this._stderr = new Gio.UnixInputStream({ fd: stderrFd, close_fd: true });

            this._stdoutCancellable = new Gio.Cancellable();
            this._bufferUsed = 0;
            this._readStdoutBytes();
            this._readStderrLine();

            GLib.child_watch_add(GLib.PRIORITY_DEFAULT, pid, (pid, status) => {
                GLib.spawn_close_pid(pid);
                if (this._procPid === pid) {
                    this._procPid = null;
                }
            });
        } catch (e) {}
    }

    _readStdoutBytes() {
        if (!this._stdout)
            return;

        const readSize = Math.max(4096, this._numBars * 2 * 4);
        this._stdout.read_bytes_async(readSize, GLib.PRIORITY_DEFAULT, this._stdoutCancellable, (stream, res) => {
            try {
                let gbytes = stream.read_bytes_finish(res);
                if (!gbytes) {
                    return;
                }
                let chunk = gbytes.get_data ? gbytes.get_data() : null;
                if (!chunk || chunk.length === 0) {
                    this._readStdoutBytes();
                    return;
                }

                const needed = this._bufferUsed + chunk.length;
                if (needed > this._rawBuffer.length) {
                    const newBuffer = new Uint8Array(Math.max(needed, this._rawBuffer.length * 2));
                    newBuffer.set(this._rawBuffer.subarray(0, this._bufferUsed));
                    this._rawBuffer = newBuffer;
                }
                this._rawBuffer.set(chunk, this._bufferUsed);
                this._bufferUsed += chunk.length;

                const frameSize = this._numBars * 2;
                let offset = 0;
                while (this._bufferUsed - offset >= frameSize) {
                    let dv = new DataView(this._rawBuffer.buffer, this._rawBuffer.byteOffset + offset, frameSize);
                    let maxVal = 1;
                    for (let i = 0; i < this._numBars; i++) {
                        let v = dv.getInt16(i * 2, true);
                        v = v < 0 ? -v : v;
                        this._bins[i] = v;
                        if (v > maxVal) maxVal = v;
                    }

                    if (maxVal < this._noiseFloor) this._silentFrames++; else this._silentFrames = 0;
                    this._updateVisibility();

                    if (this._silentFrames >= this._silenceZeroFrames) {
                        for (let i = 0; i < this._numBars; i++) {
                            let prev = this._prevHeights[i];
                            if (prev !== MIN_HEIGHT) {
                                this._prevHeights[i] = MIN_HEIGHT;
                                this._bars[i].height = MIN_HEIGHT;
                            }
                        }
                    } else {
                        const invMaxVal = maxVal > 0 ? 1 / maxVal : 0;
                        for (let i = 0; i < this._numBars; i++) {
                            let v = this._bins[i];
                            let norm = v * invMaxVal;
                            let target = Math.max(MIN_HEIGHT, Math.round(Math.sqrt(norm) * this._maxHeight));
                            if (this._silentFrames === 0 && v > 0 && target < MIN_ACTIVE_HEIGHT) target = MIN_ACTIVE_HEIGHT;
                            let prev = this._prevHeights[i];
                            let alpha = target < prev ? this._alphaFall : this._alphaRise;
                            let height = Math.round(prev * (1 - alpha) + target * alpha);
                            if (height !== prev) {
                                this._prevHeights[i] = height;
                                this._bars[i].height = height;
                            }
                        }
                    }

                    offset += frameSize;
                }

                if (offset > 0) {
                    this._rawBuffer.copyWithin(0, offset, this._bufferUsed);
                    this._bufferUsed -= offset;
                }

                this._readStdoutBytes();
            } catch (e) {
            }
        });
    }

    _readStderrLine() {
        if (!this._stderr)
            return;

        if (!this._stderrStream)
            this._stderrStream = new Gio.DataInputStream({ base_stream: this._stderr });

        this._stderrStream.read_line_async(null, (stream, res) => {
            try {
                let [line] = stream.read_line_finish(res);
                if (line !== null)
                    this._readStderrLine();
            } catch (e) {}
        });
    }

    _stopCava() {
        if (this._procPid) {
            try {
                // Send SIGTERM directly to avoid spawning an extra shell
                GLib.kill(this._procPid, 15);
                GLib.spawn_close_pid(this._procPid);
            } catch (e) {}
            this._procPid = null;
        }
        if (this._stdout) {
            try {
                if (this._stdoutCancellable && !this._stdoutCancellable.is_cancelled())
                    this._stdoutCancellable.cancel();
                this._stdout.close(null);
            } catch (e) {}
            this._stdout = null;
        }
        this._stdoutCancellable = null;
        this._bufferUsed = 0;
        if (this._stderr) {
            try { this._stderr.close(null); } catch (e) {}
            this._stderr = null;
        }
        if (this._stderrStream) {
            try { this._stderrStream.close(null); } catch (e) {}
            this._stderrStream = null;
        }
        if (this._tmpConfigPath) {
            try {
                const file = Gio.File.new_for_path(this._tmpConfigPath);
                if (file.query_exists(null))
                    file.delete(null);
            } catch (e) {}
            this._tmpConfigPath = null;
        }
        this._cleanupOldTempFiles();
    }

    _cleanupOldTempFiles() {
        try {
            const tmpDir = Gio.File.new_for_path(GLib.get_tmp_dir());
            const enumerator = tmpDir.enumerate_children('standard::name', Gio.FileQueryInfoFlags.NONE, null);
            let info;
            while ((info = enumerator.next_file(null)) !== null) {
                const name = info.get_name();
                if (name.startsWith('soundbar-cava-config-')) {
                    try {
                        tmpDir.get_child(name).delete(null);
                    } catch (e) {}
                }
            }
        } catch (e) {}
    }

    destroy() {
        if (this._settingsChangedIds) {
            this._settingsChangedIds.forEach(id => this._settings.disconnect(id));
            this._settingsChangedIds = null;
        }
        this._stopCava();
        if (this._box) {
            this._box.destroy();
            this._box = null;
        }
        super.destroy();
    }
});

export default class AudioVisualExtension extends Extension {
    constructor(metadata) {
        super(metadata);
        this._button = null;
        this._settings = null;
    }

    enable() {
        this._settings = this.getSettings();
        this._button = new VisualizerButton(this._settings, this);
        
        const panelPosition = this._settings.get_string('panel-position');
        const positionIndex = this._settings.get_int('position-index');
        
        // Map panel-position to the actual panel box
        let box;
        if (panelPosition === 'left') {
            box = Main.panel._leftBox;
        } else if (panelPosition === 'center') {
            box = Main.panel._centerBox;
        } else { // right
            box = Main.panel._rightBox;
        }
        
        // Add to the appropriate box at the specified index
        Main.panel._addToPanelBox('audio-visual', this._button, positionIndex, box);
        
        // Listen for position changes and restart
        this._positionChangedId = this._settings.connect('changed::panel-position', () => {
            this.disable();
            this.enable();
        });
        this._indexChangedId = this._settings.connect('changed::position-index', () => {
            this.disable();
            this.enable();
        });
    }

    disable() {
        if (this._positionChangedId) {
            this._settings.disconnect(this._positionChangedId);
            this._positionChangedId = null;
        }
        if (this._indexChangedId) {
            this._settings.disconnect(this._indexChangedId);
            this._indexChangedId = null;
        }
        if (this._button) {
            this._button.destroy();
            this._button = null;
        }
        this._settings = null;
    }
}
