import Adw from 'gi://Adw';
import Gtk from 'gi://Gtk';
import Gdk from 'gi://Gdk';
import Gio from 'gi://Gio';
import {ExtensionPreferences} from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';

export default class AudioVisualPreferences extends ExtensionPreferences {
    fillPreferencesWindow(window) {
        const settings = this.getSettings();

        const page = new Adw.PreferencesPage();
        const group = new Adw.PreferencesGroup({
            title: 'SoundBar Settings',
            description: 'Configure the audio visualizer appearance'
        });
        page.add(group);

        const barCountRow = new Adw.SpinRow({
            title: 'Bar Count',
            subtitle: 'Number of visualizer bars to display',
            adjustment: new Gtk.Adjustment({
                lower: 8,
                upper: 64,
                step_increment: 1
            })
        });
        settings.bind('bar-count', barCountRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        group.add(barCountRow);

        const maxHeightRow = new Adw.SpinRow({
            title: 'Maximum Height',
            subtitle: 'Maximum height of bars in pixels',
            adjustment: new Gtk.Adjustment({
                lower: 10,
                upper: 50,
                step_increment: 1
            })
        });
        settings.bind('max-height', maxHeightRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        group.add(maxHeightRow);

        const colorButton = new Gtk.ColorButton();
        const rgba = new Gdk.RGBA();
        rgba.parse(settings.get_string('bar-color'));
        colorButton.set_rgba(rgba);
        
        colorButton.connect('color-set', () => {
            const color = colorButton.get_rgba();
            const colorString = `rgba(${Math.round(color.red * 255)}, ${Math.round(color.green * 255)}, ${Math.round(color.blue * 255)}, ${color.alpha.toFixed(2)})`;
            settings.set_string('bar-color', colorString);
        });

        const colorRow = new Adw.ActionRow({
            title: 'Bar Color',
            subtitle: 'Choose the color of visualizer bars'
        });
        colorRow.add_suffix(colorButton);
        colorRow.set_activatable_widget(colorButton);
        group.add(colorRow);

        const gradientRow = new Adw.SwitchRow({
            title: 'Use Gradient',
            subtitle: 'Alternate between two colors for gradient effect'
        });
        settings.bind('use-gradient', gradientRow, 'active', Gio.SettingsBindFlags.DEFAULT);
        group.add(gradientRow);

        const gradientColorButton = new Gtk.ColorButton();
        const gradientRgba = new Gdk.RGBA();
        gradientRgba.parse(settings.get_string('gradient-color'));
        gradientColorButton.set_rgba(gradientRgba);
        
        gradientColorButton.connect('color-set', () => {
            const color = gradientColorButton.get_rgba();
            const colorString = `rgba(${Math.round(color.red * 255)}, ${Math.round(color.green * 255)}, ${Math.round(color.blue * 255)}, ${color.alpha.toFixed(2)})`;
            settings.set_string('gradient-color', colorString);
        });

        const gradientColorRow = new Adw.ActionRow({
            title: 'Gradient Color',
            subtitle: 'Second color for gradient effect'
        });
        gradientColorRow.add_suffix(gradientColorButton);
        gradientColorRow.set_activatable_widget(gradientColorButton);
        group.add(gradientColorRow);

        const hideSilentRow = new Adw.SwitchRow({
            title: 'Hide When Silent',
            subtitle: 'Hide the visualizer when there is no audio playback'
        });
        settings.bind('hide-when-silent', hideSilentRow, 'active', Gio.SettingsBindFlags.DEFAULT);
        group.add(hideSilentRow);

        const sensitivityRow = new Adw.SpinRow({
            title: 'Sensitivity',
            subtitle: 'Audio sensitivity/gain level (50-200%)',
            adjustment: new Gtk.Adjustment({
                lower: 50,
                upper: 200,
                step_increment: 5
            })
        });
        settings.bind('sensitivity', sensitivityRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        group.add(sensitivityRow);

        const framerateRow = new Adw.SpinRow({
            title: 'Framerate (FPS)',
            subtitle: 'How often the visualizer updates (10–60)',
            adjustment: new Gtk.Adjustment({
                lower: 10,
                upper: 60,
                step_increment: 1
            })
        });
        settings.bind('framerate', framerateRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        group.add(framerateRow);

        const riseRow = new Adw.SpinRow({
            title: 'Rise Smoothing',
            subtitle: '0.05–1.0 (lower = slower rise)',
            digits: 2,
            adjustment: new Gtk.Adjustment({
                lower: 0.05,
                upper: 1.0,
                step_increment: 0.05
            })
        });
        settings.bind('alpha-rise', riseRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        group.add(riseRow);

        const fallRow = new Adw.SpinRow({
            title: 'Fall Smoothing',
            subtitle: '0.5–1.0 (higher = faster fall)',
            digits: 2,
            adjustment: new Gtk.Adjustment({
                lower: 0.5,
                upper: 1.0,
                step_increment: 0.05
            })
        });
        settings.bind('alpha-fall', fallRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        group.add(fallRow);

        const noiseRow = new Adw.SpinRow({
            title: 'Noise Floor',
            subtitle: 'Treat levels below this as silence (20–1000)',
            adjustment: new Gtk.Adjustment({
                lower: 20,
                upper: 1000,
                step_increment: 10
            })
        });
        settings.bind('noise-floor', noiseRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        group.add(noiseRow);

        const silenceFramesRow = new Adw.SpinRow({
            title: 'Silence Snap Frames',
            subtitle: 'Consecutive silent frames before snapping to baseline (1–12)',
            adjustment: new Gtk.Adjustment({
                lower: 1,
                upper: 12,
                step_increment: 1
            })
        });
        settings.bind('silence-zero-frames', silenceFramesRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        group.add(silenceFramesRow);

        // Panel Position Settings
        const positionGroup = new Adw.PreferencesGroup({
            title: 'Panel Position',
            description: 'Configure where the visualizer appears in the top panel'
        });
        page.add(positionGroup);

        const panelPositionRow = new Adw.ComboRow({
            title: 'Position in Top Panel',
            subtitle: 'Left, Center, or Right section of the panel'
        });
        const positionModel = new Gtk.StringList();
        positionModel.append('Left');
        positionModel.append('Center');
        positionModel.append('Right');
        panelPositionRow.set_model(positionModel);

        const currentPosition = settings.get_string('panel-position');
        const positionMap = {'left': 0, 'center': 1, 'right': 2};
        panelPositionRow.set_selected(positionMap[currentPosition] || 0);

        panelPositionRow.connect('notify::selected', () => {
            const selected = panelPositionRow.get_selected();
            const positions = ['left', 'center', 'right'];
            settings.set_string('panel-position', positions[selected]);
        });
        positionGroup.add(panelPositionRow);

        const positionIndexRow = new Adw.SpinRow({
            title: 'Position Index',
            subtitle: 'Order relative to other elements',
            adjustment: new Gtk.Adjustment({
                lower: -10,
                upper: 10,
                step_increment: 1
            })
        });
        settings.bind('position-index', positionIndexRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        positionGroup.add(positionIndexRow);

        const bottomPaddingRow = new Adw.SpinRow({
            title: 'Bottom Padding',
            subtitle: 'Padding from the bottom of the panel in pixels',
            adjustment: new Gtk.Adjustment({
                lower: 0,
                upper: 15,
                step_increment: 1
            })
        });
        settings.bind('bottom-padding', bottomPaddingRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        positionGroup.add(bottomPaddingRow);

        window.add(page);
    }
}
