'use strict';

import St from 'gi://St';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Gio from 'gi://Gio';
import Clutter from 'gi://Clutter';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import {Extension} from 'resource:///org/gnome/shell/extensions/extension.js';

// Debug helper function
function debugLog(message, settings = null) {
    if (settings && settings.get_boolean('debug-mode')) {
        console.debug(`WINBOAT: ${message}`);
    }
}

export default class BoatmanExtension extends Extension {
    enable() {
        this._settings = this.getSettings();
        this._indicator = new BoatmanIndicator(this, this._settings);
        Main.panel.addToStatusArea(this.uuid, this._indicator);
    }

    disable() {
        this._indicator?.destroy();
        this._indicator = null;
        this._settings = null;
    }
}

class BoatmanIndicator extends PanelMenu.Button {
    static {
        GObject.registerClass(this);
    }

    constructor(extension, settings) {
        super(0.0, 'WinBoat');
        this._extension = extension;
        this._settings = settings;
        this._containerRuntime = null;
        
        // Store last stats to persist across menu opens/closes
        this._lastStats = {
            cpu: 'Click to check',
            memory: 'Click to check',
            ports: 'Click to check',
            timestamp: null
        };
        
        // Set up the panel icon with proper GNOME Shell styling
        this._icon = new St.Icon({
            icon_name: 'system-run-symbolic',
            style_class: 'system-status-icon',
            y_expand: false,
            y_align: Clutter.ActorAlign.CENTER
        });
        this.add_child(this._icon);
        
        // Set up the menu with proper GNOME Shell structure
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add status item
        this._statusItem = new PopupMenu.PopupMenuItem('Checking status...');
        this._statusItem.reactive = false;
        this.menu.addMenuItem(this._statusItem);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add Open WinBoat App item (before action buttons)
        this._webUiItem = new PopupMenu.PopupMenuItem('Open Interface');
        this._webUiItem.connect('activate', () => this._openWinBoatApplication());
        this.menu.addMenuItem(this._webUiItem);
        
        // Add action buttons
        this._startButton = new PopupMenu.PopupMenuItem('Start WinBoat');
        this._stopButton = new PopupMenu.PopupMenuItem('Stop WinBoat');
        this._restartButton = new PopupMenu.PopupMenuItem('Restart WinBoat');
        
        this._startButton.connect('activate', () => this._startWinBoat());
        this._stopButton.connect('activate', () => this._stopWinBoat());
        this._restartButton.connect('activate', () => this._restartWinBoat());
        
        this.menu.addMenuItem(this._startButton);
        this.menu.addMenuItem(this._stopButton);
        this.menu.addMenuItem(this._restartButton);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add uptime info item
        this._uptimeItem = new PopupMenu.PopupMenuItem('Checking uptime...');
        this._uptimeItem.reactive = false;
        this.menu.addMenuItem(this._uptimeItem);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add stats items (hidden by default)
        this._cpuItem = new PopupMenu.PopupMenuItem('CPU: Click to check');
        this._cpuItem.reactive = true;
        this._cpuItem.connect('activate', () => this._updateStats());
        this.menu.addMenuItem(this._cpuItem);
        
        this._memoryItem = new PopupMenu.PopupMenuItem('Memory: Click to check');
        this._memoryItem.reactive = true;
        this._memoryItem.connect('activate', () => this._updateStats());
        this.menu.addMenuItem(this._memoryItem);
        
        this._portsItem = new PopupMenu.PopupMenuItem('Ports: Click to check');
        this._portsItem.reactive = true;
        this._portsItem.connect('activate', () => this._updateStats());
        this.menu.addMenuItem(this._portsItem);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
     
        // Add refresh item
        const refreshItem = new PopupMenu.PopupMenuItem('Refresh Status');
        refreshItem.connect('activate', () => {
            this._updateStatus();
            // Also refresh stats if they are enabled AND container is running
            if (this._settings.get_boolean('show-stats') && this._lastStatus === 'running') {
                this._updateStats();
            }
        });
        this.menu.addMenuItem(refreshItem);
        
        // Connect to settings changes
        this._showUptimeSignalId = this._settings.connect('changed::show-uptime', () => this._updateStatus());
        this._showStatsSignalId = this._settings.connect('changed::show-stats', () => this._updateMenuVisibility());
        this._compactModeSignalId = this._settings.connect('changed::compact-mode', () => this._updateStatus());
        
        // Set up periodic updates using configured refresh interval
        this._scheduleStatusUpdates();

        // Initial status update
        this._updateStatus();
    }

    _setStatusIcon(status) {
        let icon_name = null;

        switch (status) {
            case 'running':
                icon_name = 'emblem-ok-symbolic';
                break;
            case 'stopped':
                icon_name = 'media-playback-stop-symbolic';
                break;
            case 'booting':
                icon_name = 'system-run-symbolic';
                break;
            case 'docker_down':
            case 'error':
            default:
                icon_name = 'action-unavailable-symbolic';
                break;
        }

        // Try custom icons first, then fallback to symbolic
        try {
            const customIcons = {
                'running': 'boat.svg',
                'stopped': 'anchor.svg',
                'booting': 'life-guard.svg',
                'error': 'life-guard.svg',
                'docker_down': 'life-guard.svg'
            };
            
            const fileName = customIcons[status] || 'life-guard.svg';
            const file = this._extension.dir.get_child('icons').get_child(fileName);
            
            if (file && file.query_exists(null)) {
                const gicon = Gio.icon_new_for_string(file.get_path());
                this._icon.gicon = gicon;
                console.info(`WINBOAT: Loaded custom icon: ${fileName}`);
            } else {
                // Fallback to symbolic
                this._icon.icon_name = icon_name;
                console.info(`WINBOAT: Using symbolic icon: ${icon_name}`);
            }
        } catch (e) {
            console.error('WINBOAT: Failed to load custom icon, using symbolic', e);
            // Fallback to symbolic
            this._icon.icon_name = icon_name;
        }
    }

    _scheduleStatusUpdates() {
        const interval = Math.max(1, this._settings.get_int('refresh-interval'));

        if (this._timeout) {
            GLib.Source.remove(this._timeout);
            this._timeout = null;
        }

        this._timeout = GLib.timeout_add_seconds(
            GLib.PRIORITY_DEFAULT,
            interval,
            () => {
                this._updateStatus();
                return GLib.SOURCE_CONTINUE;
            }
        );

        // React to changes in refresh-interval
        if (!this._refreshSignalId) {
            this._refreshSignalId = this._settings.connect('changed::refresh-interval', () => {
                this._scheduleStatusUpdates();
            });
        }
    }

    async _runCommandAsync(command) {
        return new Promise((resolve, reject) => {
            try {
                const proc = Gio.Subprocess.new(
                    ['/bin/bash', '-c', command],
                    Gio.SubprocessFlags.STDOUT_PIPE | Gio.SubprocessFlags.STDERR_SILENCE
                );
                
                proc.wait_check_async(null, (source, result) => {
                    try {
                        const success = source.wait_check_finish(result);
                        resolve(success);
                    } catch (e) {
                        reject(e);
                    }
                });
            } catch (e) {
                reject(e);
            }
        });
    }

    async _runCommandAsyncWithOutput(command) {
        return new Promise((resolve, reject) => {
            try {
                const proc = Gio.Subprocess.new(
                    ['/bin/bash', '-c', command],
                    Gio.SubprocessFlags.STDOUT_PIPE | Gio.SubprocessFlags.STDERR_PIPE
                );
                
                proc.communicate_utf8_async(null, null, (source, result) => {
                    try {
                        const [, stdout, stderr] = source.communicate_utf8_finish(result);
                        const success = source.get_successful();
                        resolve([success, stdout, stderr]);
                    } catch (e) {
                        reject(e);
                    }
                });
            } catch (e) {
                reject(e);
            }
        });
    }

    async _getContainerRuntime() {
        if (this._containerRuntime)
            return this._containerRuntime;

        // Respect user override first
        const mode = this._settings.get_string('runtime-mode');
        if (mode === 'docker' || mode === 'podman') {
            try {
                const success = await this._runCommandAsync(`${mode} ps`);
                if (success) {
                    this._containerRuntime = mode;
                    return mode;
                }
            } catch (e) {
                // Fall through to auto-detect
            }
        }

        const runtimes = ['docker', 'podman'];

        for (const runtime of runtimes) {
            try {
                const success = await this._runCommandAsync(`${runtime} ps`);
                if (success) {
                    this._containerRuntime = runtime;
                    return runtime;
                }
            } catch (e) {
                // Ignore and try next runtime
            }
        }

        return null;
    }

    _clearBootCheckTimeout() {
        if (this._bootCheckTimeout) {
            GLib.Source.remove(this._bootCheckTimeout);
            this._bootCheckTimeout = null;
        }
    }

    _clearStatusUpdateTimeout() {
        if (this._statusUpdateTimeout) {
            GLib.Source.remove(this._statusUpdateTimeout);
            this._statusUpdateTimeout = null;
        }
    }

    _clearStatsUpdateTimeout() {
        if (this._statsUpdateTimeout) {
            GLib.Source.remove(this._statsUpdateTimeout);
            this._statsUpdateTimeout = null;
        }
    }

    async _updateStatus() {
        const previousStatus = this._lastStatus;
        let status = await this._isBoatmanRunning();

        // Ensure we always have a well-formed status object
        if (!status || typeof status !== 'object' || !status.status) {
            status = { status: 'error' };
        }

        this._lastStatus = status.status;

        switch (status.status) {
            case 'running':
                this._setStatusIcon('running');
                const compactMode = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactMode ? 'Running' : 'WinBoat is running';
                this._startButton.visible = false;
                this._stopButton.visible = true;
                this._restartButton.visible = true;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    const uptime = await this._getContainerUptime();
                    this._uptimeItem.label.text = compactMode ? uptime : `Uptime: ${uptime}`;
                }
                break;
                
            case 'booting':
                this._setStatusIcon('booting');
                const compactModeBoot = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactModeBoot ? 'Starting...' : 'WinBoat is starting...';
                this._startButton.visible = false;
                this._stopButton.visible = true;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    this._uptimeItem.label.text = compactModeBoot ? 'Starting...' : 'Starting...';
                }
                
                // Check status again in 5 seconds
                this._clearBootCheckTimeout();
                this._bootCheckTimeout = GLib.timeout_add_seconds(
                    GLib.PRIORITY_DEFAULT,
                    5,
                    () => {
                        this._bootCheckTimeout = null;
                        this._updateStatus();
                        return GLib.SOURCE_REMOVE;
                    }
                );
                break;
                
            case 'docker_down':
                this._setStatusIcon('docker_down');
                const compactModeDocker = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactModeDocker ? 'Docker down' : 'Docker not running';
                this._startButton.visible = false;
                this._stopButton.visible = false;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    this._uptimeItem.label.text = compactModeDocker ? 'Docker down' : 'Docker unavailable';
                }
                break;
                
            case 'error':
                this._setStatusIcon('error');
                const compactModeError = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactModeError ? 'Error' : 'Error checking WinBoat status';
                this._startButton.visible = true;
                this._stopButton.visible = false;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    this._uptimeItem.label.text = compactModeError ? 'Unknown' : 'Status unknown';
                }
                break;

            case 'stopped':
            default:
                this._setStatusIcon('stopped');
                const compactModeStopped = this._settings.get_boolean('compact-mode');
                this._statusItem.label.text = compactModeStopped ? 'Stopped' : 'WinBoat is stopped';
                this._startButton.visible = true;
                this._stopButton.visible = false;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._settings.get_boolean('show-uptime')) {
                    this._uptimeItem.label.text = compactModeStopped ? 'Stopped' : 'Container stopped';
                }
        }
        
        // Update menu visibility based on settings
        this._updateMenuVisibility();
        
        this._maybeNotify(previousStatus, status.status);
        return status.status === 'running';
    }

    _updateMenuVisibility() {
        const showUptime = this._settings.get_boolean('show-uptime');
        const showStats = this._settings.get_boolean('show-stats');
        const compactMode = this._settings.get_boolean('compact-mode');
        
        // Only show stats if container is running and stats setting is enabled
        const containerRunning = this._lastStatus === 'running';
        const shouldShowStats = showStats && containerRunning;

        // Show/hide uptime item
        if (this._uptimeItem) {
            this._uptimeItem.visible = showUptime;
        }

        // Show/hide stats items and separator
        if (this._cpuItem && this._memoryItem && this._portsItem) {
            this._cpuItem.visible = shouldShowStats;
            this._memoryItem.visible = shouldShowStats;
            this._portsItem.visible = shouldShowStats;
            
            // Only restore/update stats when container is running and stats are enabled
            if (shouldShowStats) {
                const timeAgo = this._formatTimestamp(this._lastStats.timestamp);
                
                if (compactMode) {
                    this._cpuItem.label.text = `CPU: ${this._lastStats.cpu}${timeAgo}`;
                    this._memoryItem.label.text = `Memory: ${this._lastStats.memory}${timeAgo}`;
                    this._portsItem.label.text = `Ports: ${this._lastStats.ports}${timeAgo}`;
                } else {
                    this._cpuItem.label.text = `CPU: ${this._lastStats.cpu}${timeAgo}`;
                    this._memoryItem.label.text = `Memory: ${this._lastStats.memory}${timeAgo}`;
                    this._portsItem.label.text = `Ports: ${this._lastStats.ports}${timeAgo}`;
                }
            }
        }

        // Update button labels in compact mode
        if (compactMode) {
            this._startButton.label.text = 'Start';
            this._stopButton.label.text = 'Stop';
            this._restartButton.label.text = 'Restart';
            this._webUiItem.label.text = 'Interface';
        } else {
            this._startButton.label.text = 'Start WinBoat';
            this._stopButton.label.text = 'Stop WinBoat';
            this._restartButton.label.text = 'Restart WinBoat';
            this._webUiItem.label.text = 'Open Interface';
        }
    }

    _maybeNotify(previousStatus, currentStatus) {
        if (!this._settings.get_boolean('enable-notifications'))
            return;

        if (!previousStatus || previousStatus === currentStatus)
            return;

        let message = null;

        if (currentStatus === 'running') {
            message = 'WinBoat is now running.';
        } else if (currentStatus === 'stopped') {
            message = 'WinBoat has stopped.';
        } else if (currentStatus === 'docker_down' || currentStatus === 'error') {
            message = 'WinBoat extension: container runtime or WinBoat status error.';
        }

        if (message)
            Main.notify('WinBoat', message);
    }

    async _isBoatmanRunning() {
        try {
            const runtime = await this._getContainerRuntime();
            if (!runtime) {
                debugLog('Container runtime not accessible', this._settings);
                return { status: 'docker_down' };
            }

            // Use docker inspect for reliable status check
            const cmd = `${runtime} inspect WinBoat --format "{{.State.Status}}"`;
            debugLog(`Running command: ${cmd}`, this._settings);
            const [success, stdout, stderr] = await this._runCommandAsyncWithOutput(cmd);

            debugLog(`Command result: success=${success}, stdout=${stdout ? stdout.toString().trim() : 'null'}, stderr=${stderr ? stderr.toString().trim() : 'null'}`, this._settings);

            if (!success || !stdout) {
                debugLog('Container not found or inspect failed', this._settings);
                return { status: 'stopped' };
            }

            const status = stdout.toString().trim();
            debugLog(`Container status: ${status}`, this._settings);

            if (status === 'running') {
                return { status: 'running', containerRunning: true };
            } else {
                return { status: 'stopped' };
            }
                
        } catch (e) {
            console.error('WINBOAT: Error checking status', e);
            return { status: 'error' };
        }
    }

    async _startWinBoat() {
        try {
            console.info('WINBOAT: Starting container...');
            this._statusItem.label.text = 'Starting...';
            
            // Disable action buttons during operation
            this._startButton.reactive = false;
            this._stopButton.reactive = false;
            this._restartButton.reactive = false;

            const runtime = await this._getContainerRuntime();
            const success = await this._runCommandAsync(`${runtime} start WinBoat`);

            if (!success) {
                console.warn('WINBOAT: Start failed');
                this._statusItem.label.text = 'Start failed';
                this._maybeNotify('Failed to start WinBoat container');
                
                // Re-enable buttons on failure
                this._startButton.reactive = true;
                this._stopButton.reactive = false;
                this._restartButton.reactive = false;
                return;
            }

            console.info('WINBOAT: Start command executed');
            this._maybeNotify('WinBoat container started');
            
            // Force immediate status update using GLib timeout
            this._clearStatusUpdateTimeout();
            this._statusUpdateTimeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                this._statusUpdateTimeout = null;
                this._updateStatus();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            console.error('WINBOAT: Error starting container', e);
            this._statusItem.label.text = 'Error starting';
            
            // Re-enable buttons on error
            this._startButton.reactive = true;
            this._stopButton.reactive = false;
            this._restartButton.reactive = false;
        }
    }

    async _stopWinBoat() {
        try {
            debugLog('Stopping container...', this._settings);
            this._statusItem.label.text = 'Stopping...';
            
            // Disable action buttons during operation
            this._startButton.reactive = false;
            this._stopButton.reactive = false;
            this._restartButton.reactive = false;

            const runtime = await this._getContainerRuntime();
            const success = await this._runCommandAsync(`${runtime} stop WinBoat`);

            if (!success) {
                debugLog('Stop failed', this._settings);
                this._statusItem.label.text = 'Stop failed';
                this._maybeNotify('Failed to stop WinBoat container');
                
                // Re-enable buttons on failure
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                return;
            }

            debugLog('Stop command executed', this._settings);
            
            // Force immediate status update using GLib timeout
            this._clearStatusUpdateTimeout();
            this._statusUpdateTimeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                this._statusUpdateTimeout = null;
                this._updateStatus();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            console.error('WINBOAT: Error stopping container', e);
            this._startButton.reactive = true;
            this._stopButton.reactive = true;
            this._restartButton.reactive = true;
            return;
        }
    }

    async _restartWinBoat() {
        try {
            debugLog('Restarting container...', this._settings);
            this._statusItem.label.text = 'Restarting...';
            
            // Disable action buttons during operation
            this._startButton.reactive = false;
            this._stopButton.reactive = false;
            this._restartButton.reactive = false;

            const runtime = await this._getContainerRuntime();
            const success = await this._runCommandAsync(`${runtime} restart WinBoat`);

            if (!success) {
                debugLog('Restart failed', this._settings);
                this._statusItem.label.text = 'Restart failed';
                this._maybeNotify('Failed to restart WinBoat container');
                
                // Re-enable buttons on failure
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                return;
            }

            debugLog('Restart command executed', this._settings);
            this._maybeNotify('WinBoat container restarted');
            
            // Force immediate status update using GLib timeout
            this._clearStatusUpdateTimeout();
            this._statusUpdateTimeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                this._statusUpdateTimeout = null;
                this._updateStatus();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            console.error('WINBOAT: Error restarting container', e);
            this._startButton.reactive = true;
            this._stopButton.reactive = true;
            this._restartButton.reactive = true;
            return;
        }
    }

    async _getContainerUptime() {
        try {
            const runtime = await this._getContainerRuntime();
            if (!runtime)
                return 'Unknown';

            const [success, stdout] = await this._runCommandAsyncWithOutput(
                `${runtime} inspect WinBoat --format "{{.State.StartedAt}}"`
            );

            if (!success || !stdout) {
                return 'Unknown';
            }

            const startedAt = stdout.toString().trim();
            const startTime = new Date(startedAt).getTime();
            const now = Date.now();
            const uptimeMs = now - startTime;

            if (uptimeMs < 0)
                return 'Unknown';

            // Format uptime nicely
            const seconds = Math.floor(uptimeMs / 1000);
            const minutes = Math.floor(seconds / 60);
            const hours = Math.floor(minutes / 60);
            const days = Math.floor(hours / 24);

            if (days > 0) {
                return `${days}d ${hours % 24}h ${minutes % 60}m`;
            } else if (hours > 0) {
                return `${hours}h ${minutes % 60}m`;
            } else if (minutes > 0) {
                return `${minutes}m ${seconds % 60}s`;
            } else {
                return `${seconds}s`;
            }

        } catch (e) {
            console.error('WINBOAT: Error getting uptime', e);
            return 'Error';
        }
    }

    async _getContainerStats() {
        try {
            const runtime = await this._getContainerRuntime();
            if (!runtime) {
                return { cpu: 'Runtime unavailable', memory: 'Runtime unavailable', ports: 'Runtime unavailable' };
            }

            // Get container stats using docker stats --no-stream
            const cmd = `${runtime} stats WinBoat --no-stream --format "{{.CPUPerc}}\t{{.MemUsage}}"`;
            const [success, stdout, stderr] = await this._runCommandAsyncWithOutput(cmd);

            if (!success || !stdout) {
                console.warn(`WINBOAT: Stats command failed: ${stderr ? stderr.toString().trim() : 'No output'}`);
                return { cpu: 'Stats unavailable', memory: 'Stats unavailable', ports: 'Stats unavailable' };
            }

            const output = stdout.toString().trim();
            debugLog(`Raw stats output: "${output}"`, this._settings);
            
            if (!output) {
                return { cpu: 'No data', memory: 'No data', ports: 'No data' };
            }

            // Split by whitespace and filter out empty parts
            const parts = output.split(/\s+/).filter(part => part.length > 0);
            
            if (parts.length < 2) {
                console.warn(`WINBOAT: Stats parse error - not enough parts: ${parts.length}`);
                return { cpu: 'Parse error', memory: 'Parse error', ports: 'Parse error' };
            }

            // First part should be CPU percentage, second should be memory usage
            let cpuPercent = parts[0];
            const memUsage = parts[1];

            // Validate CPU format (should end with %)
            if (!cpuPercent.includes('%')) {
                console.warn(`WINBOAT: Invalid CPU format: ${cpuPercent}`);
                return { cpu: 'Invalid CPU', memory: memUsage, ports: 'Ports unavailable' };
            }

            // Extract numeric value and format properly
            const cpuValue = parseFloat(cpuPercent.replace('%', ''));
            
            // Handle Docker's multi-core CPU reporting
            if (isNaN(cpuValue)) {
                cpuPercent = '0%';
            } else if (cpuValue > 100) {
                // Docker reports CPU usage as percentage of all cores
                // Calculate single-core equivalent and display both
                const normalizedCpu = Math.min(cpuValue, 999); // Cap at 999% for display
                const singleCoreCpu = (cpuValue / 100 * 100).toFixed(1); // Convert to single core percentage
                cpuPercent = `${normalizedCpu.toFixed(1)}% (${singleCoreCpu}% 1-core)`;
            } else {
                cpuPercent = `${cpuValue.toFixed(1)}%`;
            }

            // Extract only the used memory from "used / total" format
            let usedMemory = memUsage;
            if (memUsage.includes('/')) {
                usedMemory = memUsage.split('/')[0].trim();
            }

            // Get ports
            const ports = await this._getContainerPorts();

            debugLog(`Parsed stats - CPU: ${cpuPercent}, Memory: ${usedMemory}, Ports: ${ports}`, this._settings);

            return { 
                cpu: cpuPercent, 
                memory: usedMemory,
                ports: ports
            };

        } catch (e) {
            console.error('WINBOAT: Error getting stats', e);
            return { cpu: 'Error', memory: 'Error', ports: 'Error' };
        }
    }

    async _getContainerPorts() {
        try {
            const runtime = await this._getContainerRuntime();
            if (!runtime) {
                return 'Runtime unavailable';
            }

            // Get container ports (filter by container name)
            const cmd = `${runtime} ps --filter "name=WinBoat" --format "{{.Ports}}"`;
            const [success, stdout, stderr] = await this._runCommandAsyncWithOutput(cmd);

            if (!success || !stdout) {
                console.warn(`WINBOAT: Ports command failed: ${stderr ? stderr.toString().trim() : 'No output'}`);
                return 'Ports unavailable';
            }

            const portsOutput = stdout.toString().trim();
            debugLog(`Raw ports output: "${portsOutput}"`, this._settings);
            
            if (!portsOutput || portsOutput === '<nil>' || portsOutput === '') {
                return 'No ports exposed';
            }

            // Parse ports and format nicely
            const portMappings = portsOutput.split(',').map(port => port.trim()).filter(port => port.length > 0);
            
            if (portMappings.length === 0) {
                return 'No ports exposed';
            }

            // Extract just the external ports for cleaner display
            const externalPorts = portMappings.map(mapping => {
                const match = mapping.match(/:(\d+)->/);
                return match ? match[1] : mapping.split('->')[0];
            }).filter(port => port && port !== '');

            // Remove duplicates
            const uniquePorts = [...new Set(externalPorts)];

            if (uniquePorts.length === 0) {
                return 'Internal only';
            }

            // Format ports - clean and simple
            const portString = uniquePorts.join(', ');
            
            // If port string is too long, split into two lines
            if (portString.length > 30) {
                const midPoint = Math.ceil(uniquePorts.length / 2);
                const firstLine = uniquePorts.slice(0, midPoint).join(', ');
                const secondLine = uniquePorts.slice(midPoint).join(', ');
                return `${firstLine}\n${secondLine}`;
            }

            return portString;

        } catch (e) {
            console.error('WINBOAT: Error getting ports', e);
            return 'Error';
        }
    }

    _formatTimestamp(timestamp) {
        if (!timestamp) return '';
        
        const now = Date.now();
        const diff = now - timestamp;
        
        if (diff < 60000) { // Less than 1 minute
            return ' (just now)';
        } else if (diff < 3600000) { // Less than 1 hour
            const minutes = Math.floor(diff / 60000);
            return ` (${minutes}m ago)`;
        } else {
            const hours = Math.floor(diff / 3600000);
            return ` (${hours}h ago)`;
        }
    }

    async _updateStats() {
        try {
            // Only update stats if container is running
            if (this._lastStatus !== 'running') {
                debugLog('Skipping stats update - container is not running', this._settings);
                return;
            }
            
            debugLog('Updating stats on user request...', this._settings);
            
            // Show loading state immediately
            this._cpuItem.label.text = 'CPU: Checking...';
            this._memoryItem.label.text = 'Memory: Checking...';
            this._portsItem.label.text = 'Ports: Checking...';
            
            // Update stats with a small delay to prevent menu from closing
            this._clearStatsUpdateTimeout();
            this._statsUpdateTimeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 50, () => {
                this._statsUpdateTimeout = null;
                this._performStatsUpdate();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            console.error('WINBOAT: Error updating stats', e);
            this._cpuItem.label.text = 'CPU: Error';
            this._memoryItem.label.text = 'Memory: Error';
            this._portsItem.label.text = 'Ports: Error';
        }
    }

    async _performStatsUpdate() {
        try {
            // Get stats
            const stats = await this._getContainerStats();
            
            // Store last stats for persistence
            this._lastStats = {
                cpu: stats.cpu,
                memory: stats.memory,
                ports: stats.ports,
                timestamp: Date.now()
            };
            
            // Update display
            this._cpuItem.label.text = `CPU: ${stats.cpu}`;
            this._memoryItem.label.text = `Memory: ${stats.memory}`;
            this._portsItem.label.text = `Ports: ${stats.ports}`;
            
            debugLog(`Stats updated - CPU: ${stats.cpu}, Memory: ${stats.memory}, Ports: ${stats.ports}`, this._settings);
            
        } catch (e) {
            console.error('WINBOAT: Error updating stats', e);
            this._cpuItem.label.text = 'CPU: Error';
            this._memoryItem.label.text = 'Memory: Error';
            this._portsItem.label.text = 'Ports: Error';
        }
    }

    _openWinBoatApplication() {
        try {
            const windowActors = global.get_window_actors();
            let winBoatWindow = null;
            
            for (const actor of windowActors) {
                const window = actor.get_meta_window();
                const appId = window.get_gtk_application_id();
                const title = window.get_title();
                const wmClass = window.get_wm_class();
                
                if (appId && (appId === 'winboat' || appId === 'WinBoat')) {
                    winBoatWindow = window;
                    break;
                }
                
                if (wmClass && (wmClass === 'winboat' || wmClass === 'WinBoat')) {
                    winBoatWindow = window;
                    break;
                }
                
                if (title && (title === 'winboat' || title === 'WinBoat')) {
                    winBoatWindow = window;
                    break;
                }
            }
            
            if (winBoatWindow) {
                winBoatWindow.activate(global.get_current_time());
                console.info('WINBOAT: Focused WinBoat window');
            } else {
                this._launchWinBoatApp();
            }
            
        } catch (e) {
            console.error('WINBOAT: Error managing WinBoat application', e);
            Main.notify('WinBoat', 'Failed to open WinBoat application');
        }
    }

    _launchWinBoatApp() {
        try {
            const appInfo = Gio.AppInfo.get_default_for_type('x-scheme-handler/winboat', false) ||
                          Gio.AppInfo.get_default_for_type('application/x-winboat', false);
            
            if (appInfo) {
                appInfo.launch([], null);
                console.info('WINBOAT: Launched WinBoat application');
            } else {
                // Fallback: try to launch by common executable names
                const commonNames = ['winboat', 'WinBoat', 'winboat-desktop'];
                let launched = false;
                
                for (const appName of commonNames) {
                    try {
                        const proc = Gio.Subprocess.new(
                            [appName],
                            Gio.SubprocessFlags.NONE
                        );
                        launched = true;
                        console.info(`WINBOAT: Launched ${appName}`);
                        break;
                    } catch (e) {
                        // Try next name
                    }
                }
                
                if (!launched) {
                    Main.notify('WinBoat', 'WinBoat application not found. Please install WinBoat first.');
                }
            }
        } catch (e) {
            console.error('WINBOAT: Error launching WinBoat application', e);
            Main.notify('WinBoat', 'Failed to launch WinBoat application');
        }
    }

    destroy() {
        // Clean up all timeouts
        this._clearBootCheckTimeout();
        this._clearStatusUpdateTimeout();
        this._clearStatsUpdateTimeout();
        
        if (this._timeout) {
            GLib.Source.remove(this._timeout);
            this._timeout = null;
        }
        
        // Disconnect all signals
        if (this._refreshSignalId) {
            this._settings.disconnect(this._refreshSignalId);
            this._refreshSignalId = null;
        }
        if (this._showUptimeSignalId) {
            this._settings.disconnect(this._showUptimeSignalId);
            this._showUptimeSignalId = null;
        }
        if (this._showStatsSignalId) {
            this._settings.disconnect(this._showStatsSignalId);
            this._showStatsSignalId = null;
        }
        if (this._compactModeSignalId) {
            this._settings.disconnect(this._compactModeSignalId);
            this._compactModeSignalId = null;
        }
        
        // Call parent destroy
        super.destroy();
    }
}