import * as lib from './lib.js';
import * as log from './log.js';
import * as once_cell from './once_cell.js';
import * as Rect from './rectangle.js';
import * as Tags from './tags.js';
import * as utils from './utils.js';
import * as xprop from './xprop.js';
import * as focus from './focus.js';
import Meta from 'gi://Meta';
import Shell from 'gi://Shell';
import St from 'gi://St';
import GLib from 'gi://GLib';
import Gio from 'gi://Gio';
import Mtk from 'gi://Mtk';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
const { OnceCell } = once_cell;
const WM_TITLE_BLACKLIST = [
    'Firefox',
    'Nightly',
    'Tor Browser',
];
export class ShellWindow {
    entity;
    meta;
    known_workspace;
    grab = false;
    activate_after_move = false;
    ignore_detach = false;
    was_attached_to;
    destroying = false;
    active_hint_show_id = null;
    reassignment = false;
    smart_gapped = false;
    border = new St.Bin({
        style_class: 'gnome-mosaic-active-hint gnome-mosaic-border-normal',
    });
    window_app;
    was_hidden = false;
    extra = {
        normal_hints: new OnceCell(),
        wm_role_: new OnceCell(),
        xid_: new OnceCell(),
    };
    last_border_rect = null;
    border_width = 0;
    constructor(entity, window, window_app, ext) {
        this.window_app = window_app;
        this.entity = entity;
        this.meta = window;
        this.known_workspace = this.workspace_id();
        if (this.meta.is_fullscreen()) {
            ext.add_tag(entity, Tags.Floating);
        }
        this.decorate(ext);
        this.bind_window_events(ext);
        this.bind_hint_events(ext);
        if (this.border) {
            global.window_group.add_child(this.border);
            this.restack();
        }
        this.hide_border();
        this.update_border_layout(ext);
        if (this.meta.get_compositor_private()?.get_stage())
            this.on_style_changed(ext);
    }
    toJSON() {
        return {
            entity: this.entity,
            workspace: this.known_workspace,
            grab: this.grab,
            activate_after_move: this.activate_after_move,
            ignore_detach: this.ignore_detach,
            was_attached_to: this.was_attached_to,
            destroying: this.destroying,
            reassignment: this.reassignment,
            smart_gapped: this.smart_gapped,
            meta_id: this.meta.get_id(),
        };
    }
    static fromJSON(data, ext) {
        const win = new ShellWindow(data.entity, data.meta, Shell.WindowTracker.get_default().get_window_app(data.meta), ext);
        win.known_workspace = data.workspace;
        win.grab = data.grab;
        win.activate_after_move = data.activate_after_move;
        win.ignore_detach = data.ignore_detach;
        win.destroying = data.destroying;
        win.reassignment = data.reassignment;
        win.smart_gapped = data.smart_gapped;
        return win;
    }
    activate(ext, move_mouse = true) {
        activate(ext, move_mouse, this.meta);
    }
    actor_exists() {
        return !this.destroying && this.meta.get_compositor_private() !== null;
    }
    bind_window_events(ext) {
        ext.window_signals
            .get_or(this.entity, () => new Array())
            .push(this.meta.connect('size-changed', () => {
            this.window_changed(ext);
        }), this.meta.connect('position-changed', () => {
            this.window_changed(ext);
        }), this.meta.connect('workspace-changed', () => {
            this.workspace_changed();
        }), this.meta.connect('notify::wm-class', () => {
            this.wm_class_changed(ext);
        }), this.meta.connect('raised', () => {
            this.window_raised(ext);
        }), global.display.connect('restacked', () => {
            this.restack();
        }));
    }
    bind_hint_events(ext) {
        if (!this.border)
            return;
        let change_id = ext.settings.ext.connect('changed', () => {
            return false;
        });
        this.border.connect('destroy', () => {
            ext.settings.ext.disconnect(change_id);
        });
        this.border.connect('style-changed', () => {
            this.on_style_changed(ext);
        });
    }
    cmdline() {
        let pid = this.meta.get_pid(), out = null;
        if (-1 === pid)
            return out;
        const path = '/proc/' + pid + '/cmdline';
        if (!utils.exists(path))
            return out;
        const result = utils.read_to_string(path);
        if (result.kind == 1) {
            out = result.value.trim();
        }
        else {
            log.error(`failed to fetch cmdline: ${result.value.format()}`);
        }
        return out;
    }
    async decorate(ext) {
        if (await this.may_decorate()) {
            if (!this.is_client_decorated()) {
                if (ext.settings.show_title()) {
                    this.decoration_show(ext);
                }
                else {
                    this.decoration_hide(ext);
                }
            }
        }
    }
    async decoration(_ext, callback) {
        if (await this.may_decorate()) {
            const xid = this.xid();
            if (xid)
                callback(xid);
        }
    }
    decoration_hide(ext) {
        if (this.ignore_decoration())
            return;
        this.was_hidden = true;
        this.decoration(ext, xid => xprop.set_hint(xid, xprop.MOTIF_HINTS, xprop.HIDE_FLAGS));
    }
    decoration_show(ext) {
        if (!this.was_hidden)
            return;
        this.decoration(ext, xid => xprop.set_hint(xid, xprop.MOTIF_HINTS, xprop.SHOW_FLAGS));
    }
    icon(_ext, size) {
        let icon = this.window_app.create_icon_texture(size);
        if (!icon) {
            icon = new St.Icon({
                icon_name: 'applications-other',
                icon_type: St.IconType.FULLCOLOR,
                icon_size: size,
            });
        }
        return icon;
    }
    ignore_decoration() {
        const name = this.meta.get_wm_class();
        if (name === null)
            return true;
        return WM_TITLE_BLACKLIST.findIndex(n => name.startsWith(n)) !== -1;
    }
    is_client_decorated() {
        const xid = this.xid();
        const extents = xid ? xprop.get_frame_extents(xid) : false;
        if (!extents)
            return false;
        return true;
    }
    is_maximized() {
        return this.meta.is_maximized
            ? this.meta.is_maximized()
            : this.meta.get_maximized() !== 0;
    }
    is_max_screen(ext) {
        return (this.is_maximized() ||
            ext.settings.gap_inner() === 0 ||
            this.smart_gapped);
    }
    is_single_max_screen() {
        const display = this.meta.get_display();
        if (display) {
            let monitor_count = display.get_n_monitors();
            return ((this.is_maximized() || this.smart_gapped) && monitor_count == 1);
        }
        return false;
    }
    is_snap_edge() {
        return ((this.meta.get_maximize_flags
            ? this.meta.get_maximize_flags()
            : this.meta.get_maximized()) == Meta.MaximizeFlags.VERTICAL);
    }
    is_tilable(ext) {
        let tile_checks = () => {
            let wm_class = this.meta.get_wm_class();
            if (wm_class !== null && wm_class.trim().length === 0) {
                wm_class = this.name(ext);
            }
            const role = this.meta.get_role();
            if (role === 'quake')
                return false;
            if (this.meta.get_title() === 'Steam') {
                const rect = this.rect();
                const is_dialog = rect.width < 400 && rect.height < 200;
                const is_first_login = rect.width === 432 && rect.height === 438;
                if (is_dialog || is_first_login)
                    return false;
            }
            if (wm_class !== null &&
                ext.conf.window_shall_float(wm_class, this.title(ext))) {
                return ext.contains_tag(this.entity, Tags.ForceTile);
            }
            return (this.meta.window_type == Meta.WindowType.NORMAL &&
                !this.is_transient() &&
                wm_class !== null);
        };
        return !ext.contains_tag(this.entity, Tags.Floating) && tile_checks();
    }
    is_transient() {
        return this.meta.get_transient_for() !== null;
    }
    async may_decorate() {
        const xid = this.xid();
        return xid ? await xprop.may_decorate(xid) : false;
    }
    move(ext, rect, on_complete) {
        if (!this.same_workspace() && this.is_maximized()) {
            return;
        }
        this.hide_border();
        const max_width = ext.settings.max_window_width();
        if (max_width > 0 && rect.width > max_width) {
            rect.x += (rect.width - max_width) / 2;
            rect.width = max_width;
        }
        const clone = Rect.Rectangle.from_meta(rect);
        const meta = this.meta;
        const actor = meta.get_compositor_private();
        if (actor) {
            if (meta.set_unmaximize_flags)
                meta.set_unmaximize_flags(Meta.MaximizeFlags.BOTH);
            else
                meta.unmaximize(Meta.MaximizeFlags.BOTH);
            actor.remove_all_transitions();
            ext.movements.insert(this.entity, clone);
            ext.register({ tag: 2, window: this, kind: { tag: 1 } });
            if (on_complete)
                ext.register_fn(on_complete);
            if (meta.appears_focused) {
                this.update_border_layout(ext);
                ext.show_border_on_focused();
            }
        }
    }
    name(ext) {
        return ext.names.get_or(this.entity, () => 'unknown');
    }
    on_style_changed(ext) {
        if (!this.border)
            return;
        GLib.idle_add(GLib.PRIORITY_LOW, () => {
            if (!this.destroying && this.border) {
                this.update_border_style(ext);
            }
            return GLib.SOURCE_REMOVE;
        });
    }
    rect() {
        return Rect.Rectangle.from_meta(this.meta.get_frame_rect());
    }
    async size_hint() {
        const xid = this.xid();
        const hint = xid ? await xprop.get_size_hints(xid) : null;
        return this.extra.normal_hints.get_or_init(() => hint);
    }
    swap(ext, other) {
        let ar = this.rect().clone();
        let br = other.rect().clone();
        other.move(ext, ar);
        this.move(ext, br, () => place_pointer_on(ext, this.meta));
    }
    title(ext) {
        const title = this.meta.get_title();
        return title ? title : this.name(ext);
    }
    async wm_role() {
        const xid = this.xid();
        const role = xid ? await xprop.get_window_role(xid) : null;
        return this.extra.wm_role_.get_or_init(() => role);
    }
    workspace_id() {
        const workspace = this.meta.get_workspace();
        if (workspace) {
            return workspace.index();
        }
        else {
            this.meta.change_workspace_by_index(0, false);
            return 0;
        }
    }
    xid() {
        return this.extra.xid_.get_or_init(() => {
            if (utils.is_wayland())
                return null;
            return xprop.get_xid(this.meta);
        });
    }
    show_border(ext) {
        if (!this.border)
            return;
        this.update_border_style(ext);
        if (ext.settings.active_hint()) {
            const border = this.border;
            const permitted = () => {
                return (this.actor_exists() &&
                    ext.focus_window() == this &&
                    !this.meta.is_fullscreen() &&
                    (!this.is_single_max_screen() || this.is_snap_edge()) &&
                    !this.meta.minimized);
            };
            if (permitted()) {
                if (this.meta.appears_focused) {
                    border.show();
                    let applications = 0;
                    if (this.active_hint_show_id !== null) {
                        GLib.source_remove(this.active_hint_show_id);
                        this.active_hint_show_id = null;
                    }
                    this.active_hint_show_id = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 600, () => {
                        if ((applications > 4 && !this.same_workspace()) ||
                            !permitted()) {
                            this.active_hint_show_id = null;
                            return GLib.SOURCE_REMOVE;
                        }
                        applications += 1;
                        border.show();
                        return GLib.SOURCE_CONTINUE;
                    });
                }
            }
        }
    }
    same_workspace() {
        const workspace = this.meta.get_workspace();
        if (workspace) {
            let workspace_id = workspace.index();
            return (workspace_id ===
                global.workspace_manager.get_active_workspace_index());
        }
        return false;
    }
    same_monitor() {
        return this.meta.get_monitor() === global.display.get_current_monitor();
    }
    get always_top_windows() {
        let above_windows = new Array();
        for (const actor of global.get_window_actors()) {
            if (actor &&
                actor.get_meta_window() &&
                actor.get_meta_window().is_above())
                above_windows.push(actor);
        }
        return above_windows;
    }
    hide_border() {
        this.timeouts_remove();
        let b = this.border;
        if (b)
            b.hide();
    }
    update_border_layout(ext) {
        if (this.border) {
            const actor = this.meta.get_compositor_private();
            if (actor) {
                const borderWidth = ext.settings.active_hint_border_width();
                const rect = this.meta.get_frame_rect();
                const newX = rect.x - borderWidth;
                const newY = rect.y - borderWidth;
                const newW = rect.width + 2 * borderWidth;
                const newH = rect.height + 2 * borderWidth;
                if (this.last_border_rect &&
                    this.last_border_rect.x === newX &&
                    this.last_border_rect.y === newY &&
                    this.last_border_rect.w === newW &&
                    this.last_border_rect.h === newH) {
                    return;
                }
                this.border.set_position(newX, newY);
                this.border.set_size(newW, newH);
                this.last_border_rect = { x: newX, y: newY, w: newW, h: newH };
            }
        }
    }
    async update_border_style(ext) {
        const margin = 6;
        const radii = await getBorderRadii(this.meta.get_compositor_private());
        const radii_values = radii?.map(v => `${v + margin}px`).join(' ') || '0px 0px 0px 0px';
        const borderWidth = ext.settings.active_hint_border_width();
        const colors = utils.get_accent_colors(ext.settings);
        if (this.border) {
            this.border.set_style(`border-radius: ${radii_values};` +
                `border-width: ${borderWidth}px;` +
                `border-color: ${colors[0]}`);
            if (this.border_width !== borderWidth) {
                this.border_width = borderWidth;
                this.update_border_layout(ext);
            }
        }
    }
    wm_class_changed(ext) {
        if (this.is_tilable(ext)) {
            ext.connect_window(this);
            if (!this.meta.minimized) {
                ext.auto_tiler?.auto_tile(ext, this, ext.init);
            }
        }
    }
    window_changed(ext) {
        this.update_border_layout(ext);
        if (ext.focus_window() === this &&
            (!this.border || !this.border.visible)) {
            ext.show_border_on_focused();
        }
    }
    window_raised(ext) {
        ext.show_border_on_focused();
    }
    workspace_changed() { }
    restack() {
        if (this.border) {
            global.window_group.set_child_above_sibling(this.border, null);
            return true;
        }
        return false;
    }
    timeouts_remove() {
        if (this.active_hint_show_id) {
            GLib.source_remove(this.active_hint_show_id);
            this.active_hint_show_id = null;
        }
    }
}
export function activate(ext, move_mouse, win) {
    try {
        if (!win.get_compositor_private())
            return;
        if (ext.get_window(win)?.destroying)
            return;
        if (win.is_override_redirect())
            return;
        const workspace = win.get_workspace();
        if (!workspace)
            return;
        win.unminimize();
        workspace.activate_with_focus(win, global.get_current_time());
        win.raise();
        const pointer_placement_permitted = move_mouse &&
            Main.modalCount === 0 &&
            ext.settings.mouse_cursor_follows_active_window() &&
            !pointer_already_on_window(win) &&
            pointer_in_work_area();
        if (pointer_placement_permitted) {
            place_pointer_on(ext, win);
        }
    }
    catch (error) {
        log.error(`failed to activate window: ${error}`);
    }
}
function pointer_in_work_area() {
    const cursor = lib.cursor_rect();
    const indice = global.display.get_current_monitor();
    const mon = global.display
        .get_workspace_manager()
        .get_active_workspace()
        .get_work_area_for_monitor(indice);
    return mon ? cursor.intersects(mon) : false;
}
function place_pointer_on(ext, win) {
    const rect = win.get_frame_rect();
    let x = rect.x;
    let y = rect.y;
    let key = Object.keys(focus.FocusPosition)[ext.settings.mouse_cursor_focus_location()];
    let pointer_position_ = focus.FocusPosition[key];
    switch (pointer_position_) {
        case focus.FocusPosition.TopLeft:
            x += 8;
            y += 8;
            break;
        case focus.FocusPosition.BottomLeft:
            x += 8;
            y += rect.height - 16;
            break;
        case focus.FocusPosition.TopRight:
            x += rect.width - 16;
            y += 8;
            break;
        case focus.FocusPosition.BottomRight:
            x += rect.width - 16;
            y += rect.height - 16;
            break;
        default:
            x += 8;
            y += 8;
    }
    global.stage
        .get_context()
        .get_backend()
        .get_default_seat()
        .warp_pointer(x, y);
}
function pointer_already_on_window(meta) {
    const cursor = lib.cursor_rect();
    return cursor.intersects(meta.get_frame_rect());
}
export async function getBorderRadii(actor) {
    const opaqueLimit = 200;
    const { x, y, width, height } = actor.get_meta_window().get_frame_rect();
    const monitorIndex = actor.get_meta_window().get_monitor();
    const scale = Math.ceil(global.display.get_monitor_scale(monitorIndex));
    if (height <= 0)
        return;
    const capture = actor.paint_to_content(new Mtk.Rectangle({ x, y, width, height }));
    if (!capture)
        return;
    const memoryBuffer = Gio.MemoryOutputStream.new_resizable();
    const surface = capture.get_texture();
    const imageBuf = await Shell.Screenshot.composite_to_stream(surface, 0, 0, width, height * scale, 1, null, 0, 0, 1, memoryBuffer);
    const rawPixels = imageBuf.get_pixels();
    if (!rawPixels)
        return;
    memoryBuffer.close(null);
    const scanAlpha = (start) => {
        for (let x = 0; x < width / 2; x++) {
            const idx = (start * width + x) * 4;
            const alpha = rawPixels[idx + 3];
            if (alpha > opaqueLimit) {
                return x;
            }
        }
        return -1;
    };
    let alphaTop = -1;
    for (var row = 0; row < 3; row++) {
        alphaTop = scanAlpha(row);
        if (alphaTop > -1)
            break;
    }
    if (alphaTop === -1)
        alphaTop = 0;
    let alphaBottom = -1;
    for (var row = height * scale - 1; row > height * scale - 4; row--) {
        alphaBottom = scanAlpha(row);
        if (alphaBottom > -1)
            break;
    }
    if (alphaBottom === -1)
        alphaBottom = 0;
    const radiusTop = alphaTop / scale;
    const radiusBottom = alphaBottom / scale;
    return [radiusTop, radiusTop, radiusBottom, radiusBottom];
}
