import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Meta from 'gi://Meta';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as QuickSettings from 'resource:///org/gnome/shell/ui/quickSettings.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
class RestoreGeometryExtension {
    _settings;
    _trackedWindows;
    _windowAddedId;
    _windowTrackers = new Map();
    _windowList;
    _pendingSave;
    constructor(_settings) {
        this._settings = _settings;
        const json = this._settings.get_string('tracked-windows');
        this._trackedWindows = JSON.parse(json);
        this._windowList = new WindowListToggle(this);
        Main.panel.statusArea.quickSettings.menu.addItem(this._windowList);
        this._windowAddedId = global.display.connect('window-created', (_display, window) => {
            // Wait for window to be stable
            GLib.idle_add(GLib.PRIORITY_DEFAULT, () => {
                this._onWindowCreated(window);
                return GLib.SOURCE_REMOVE;
            });
        });
        const windows = global.get_window_actors();
        for (const windowActor of windows) {
            const window = windowActor.get_meta_window();
            if (window) {
                const wmclass = window.get_wm_class();
                if (wmclass && this.isTracked(wmclass)) {
                    this.trackWindow(window);
                }
            }
        }
    }
    destroy() {
        global.display.disconnect(this._windowAddedId);
        const disconnects = Array.from(this._windowTrackers.values());
        for (const disconnect of disconnects) {
            disconnect();
        }
        if (this._pendingSave) {
            GLib.source_remove(this._pendingSave);
            this._pendingSave = undefined;
            this._flushTrackedWindows();
        }
        this._windowTrackers.clear();
        this._windowList.destroy();
    }
    _onWindowCreated(window) {
        if (window.get_window_type() !== Meta.WindowType.NORMAL) {
            return;
        }
        const wmclass = window.get_wm_class();
        if (!wmclass) {
            return;
        }
        if (this.isTracked(wmclass)) {
            const actor = window.get_compositor_private();
            const signal = actor.connect('first-frame', () => {
                const { x, y, width, height } = this._trackedWindows[wmclass];
                window.move_resize_frame(true, x, y, width, height);
                this._trackWindow(window, wmclass);
                actor.disconnect(signal);
            });
        }
    }
    trackWindow(window) {
        const wmclass = window.get_wm_class();
        if (!wmclass) {
            return;
        }
        const frame = window.get_frame_rect();
        const geometry = {
            x: frame.x,
            y: frame.y,
            width: frame.width,
            height: frame.height,
        };
        this._trackedWindows[wmclass] = geometry;
        this._saveTrackedWindows();
        this._trackWindow(window, wmclass);
    }
    untrackWindow(wmclass) {
        delete this._trackedWindows[wmclass];
        this._saveTrackedWindows();
        for (const [window, disconnect] of this._windowTrackers.entries()) {
            if (window.get_wm_class() === wmclass) {
                disconnect();
            }
        }
    }
    isTracked(wmclass) {
        return wmclass in this._trackedWindows;
    }
    _trackWindow(window, wmclass) {
        if (this._windowTrackers.has(window)) {
            return;
        }
        const doUpdateGeometry = () => {
            const frame = window.get_frame_rect();
            const geometry = {
                x: frame.x,
                y: frame.y,
                width: frame.width,
                height: frame.height,
            };
            this._trackedWindows[wmclass] = geometry;
            this._saveTrackedWindows();
        };
        let pendingUpdate;
        const updateGeometry = () => {
            if (pendingUpdate) {
                GLib.source_remove(pendingUpdate);
            }
            pendingUpdate = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                pendingUpdate = undefined;
                doUpdateGeometry();
                return GLib.SOURCE_REMOVE;
            });
        };
        const signalIds = [
            window.connect('position-changed', updateGeometry),
            window.connect('size-changed', updateGeometry),
            window.connect('unmanaged', () => {
                if (pendingUpdate) {
                    GLib.source_remove(pendingUpdate);
                    pendingUpdate = undefined;
                }
                disconnect();
            }),
        ];
        const disconnect = () => {
            for (const signalId of signalIds) {
                window.disconnect(signalId);
            }
            if (pendingUpdate) {
                GLib.source_remove(pendingUpdate);
                pendingUpdate = undefined;
                doUpdateGeometry();
            }
            this._windowTrackers.delete(window);
        };
        this._windowTrackers.set(window, disconnect);
    }
    _saveTrackedWindows() {
        if (this._pendingSave) {
            GLib.source_remove(this._pendingSave);
        }
        this._pendingSave = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
            this._pendingSave = undefined;
            this._flushTrackedWindows();
            return GLib.SOURCE_REMOVE;
        });
    }
    _flushTrackedWindows() {
        const json = JSON.stringify(this._trackedWindows);
        this._settings.set_string('tracked-windows', json);
    }
    getOpenWindows() {
        const trackableWindows = [];
        for (const actor of global.get_window_actors()) {
            const window = actor.get_meta_window();
            if (!window || window.get_window_type() !== Meta.WindowType.NORMAL) {
                continue;
            }
            const wmclass = window.get_wm_class();
            if (!wmclass) {
                continue;
            }
            trackableWindows.push({
                window,
                wmclass,
                tracked: this.isTracked(wmclass),
            });
        }
        trackableWindows.sort((a, b) => a.wmclass.localeCompare(b.wmclass));
        return trackableWindows;
    }
}
const WindowListToggle = GObject.registerClass(class WindowListToggle extends QuickSettings.QuickMenuToggle {
    _extension;
    constructor(_extension) {
        super({
            title: 'Geometry',
            iconName: 'window-new-symbolic',
            toggleMode: false,
        });
        this._extension = _extension;
        this.menu.setHeader('window-new-symbolic', 'Restore Geometry');
        this.menu.connect('open-state-changed', (_menu, open) => {
            if (open) {
                this.updateMenu();
            }
            return false;
        });
    }
    updateMenu() {
        this.menu.removeAll();
        const windows = this._extension.getOpenWindows();
        if (windows.length === 0) {
            const item = new PopupMenu.PopupMenuItem('No windows open');
            item.sensitive = false;
            this.menu.addMenuItem(item);
            return;
        }
        for (const { window, wmclass, tracked } of windows) {
            const title = window.get_title() || wmclass;
            const item = new PopupMenu.PopupSwitchMenuItem(`${title} (${wmclass})`, tracked);
            item.connect('toggled', () => {
                if (item.state) {
                    this._extension.trackWindow(window);
                }
                else {
                    this._extension.untrackWindow(wmclass);
                }
            });
            this.menu.addMenuItem(item);
        }
    }
});
export default class RestoreGeometry extends Extension {
    _impl;
    enable() {
        if (this._impl) {
            this._impl.destroy();
        }
        this._impl = new RestoreGeometryExtension(this.getSettings());
    }
    disable() {
        if (this._impl) {
            this._impl.destroy();
            this._impl = undefined;
        }
    }
}
