// extension.js
import St from 'gi://St';
import Clutter from 'gi://Clutter';
import GObject from 'gi://GObject';
import GLib from 'gi://GLib';
import Gio from 'gi://Gio';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import * as Slider from 'resource:///org/gnome/shell/ui/slider.js';
import {Extension} from 'resource:///org/gnome/shell/extensions/extension.js';

const ShaderEffect = GObject.registerClass(
class ShaderEffect extends Clutter.ShaderEffect {
    _init(shaderSource) {
        super._init();
        this._param1 = 0.5;
        this._param2 = 0.5;
        this._param3 = 0.5;
        this._param4 = 0.5;
        this._time = 0.0;
        this.set_shader_source(shaderSource);
    }

    set param1(value) {
        this._param1 = value;
        this.set_uniform_value('param1', value);
        this.queue_repaint();
    }

    get param1() {
        return this._param1;
    }

    set param2(value) {
        this._param2 = value;
        this.set_uniform_value('param2', value);
        this.queue_repaint();
    }

    get param2() {
        return this._param2;
    }
    
    set param3(value) {
        this._param3 = value;
        this.set_uniform_value('param3', value);
        this.queue_repaint();
    }
    
    get param3() {
        return this._param3;
    }
    
    set param4(value) {
        this._param4 = value;
        this.set_uniform_value('param4', value);
        this.queue_repaint();
    }
    
    get param4() {
        return this._param4;
    }

    updateTime(time) {
        this._time = time;
        this.set_uniform_value('time', time);
    }
    
    vfunc_paint_target(paint_node, paint_context) {
        this.set_uniform_value('tex', 0);
        this.set_uniform_value('param1', this._param1);
        this.set_uniform_value('param2', this._param2);
        this.set_uniform_value('param3', this._param3);
        this.set_uniform_value('param4', this._param4);
        this.set_uniform_value('time', this._time);
        super.vfunc_paint_target(paint_node, paint_context);
    }   
});

const EyeProtectorMenu = GObject.registerClass(
class EyeProtectorMenu extends PanelMenu.Button {
    _init(extension) {
        super._init(0.0, 'Eye Protector', false);
        this._extension = extension;

        this._icon = new St.Icon({
            icon_name: 'view-reveal-symbolic',
            style_class: 'system-status-icon',
        });
        this.add_child(this._icon);

        // Parameter 1 slider
        this._param1Item = new PopupMenu.PopupBaseMenuItem({activate: false});
        this._param1Item.add_child(new St.Label({text: 'Strength: '}));
        this._param1Slider = new Slider.Slider(1.0);
        this._param1Slider.connect('notify::value', () => {
            this._extension.setParam1(this._param1Slider.value);
        });
        this._param1Item.add_child(this._param1Slider);
        this.menu.addMenuItem(this._param1Item);

        // Parameter 2 slider
        this._param2Item = new PopupMenu.PopupBaseMenuItem({activate: false});
        this._param2Item.add_child(new St.Label({text: 'Radius: '}));
        this._param2Slider = new Slider.Slider(0.00);
        this._param2Slider.connect('notify::value', () => {
            this._extension.setParam2(this._param2Slider.value);
        });
        this._param2Item.add_child(this._param2Slider);
        this.menu.addMenuItem(this._param2Item);
        
        // Add horizontal Seperator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Parameter 3 slider
        this._param3Item = new PopupMenu.PopupBaseMenuItem({activate: false});
        this._param3Item.add_child(new St.Label({text: 'BlueCutter: '}));
        this._param3Slider = new Slider.Slider(0.50);
        this._param3Slider.connect('notify::value', () => {
            this._extension.setParam3(this._param3Slider.value);
        });
        this._param3Item.add_child(this._param3Slider);
        this.menu.addMenuItem(this._param3Item);
        
        // Parameter 4 slider
        this._param4Item = new PopupMenu.PopupBaseMenuItem({activate: false});
        this._param4Item.add_child(new St.Label({text: 'LSCutter: '}));
        this._param4Slider = new Slider.Slider(0.35);
        this._param4Slider.connect('notify::value', () => {
            this._extension.setParam4(this._param4Slider.value);
        });
        this._param4Item.add_child(this._param4Slider);
        this.menu.addMenuItem(this._param4Item);

        // Add horizontal Seperator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());

        // Toggle effect
        this._toggleItem = new PopupMenu.PopupSwitchMenuItem('Enable Effect', true);
        this._toggleItem.connect('toggled', (item, state) => {
            this._extension.toggleEffect(state);
            this.updateIcon(state);
        });
        this.menu.addMenuItem(this._toggleItem);
    }

    updateIcon(enabled) {
        this._icon.icon_name = enabled ? 'view-reveal-symbolic' : 'view-conceal-symbolic';
    }
});

export default class EyeProtectorExtension extends Extension {
    enable() {
        this._shaderEffect = null;
        this._timeoutId = null;
        this._time = 0.0;
        this._effectName = 'eye-protector-effect';
        this._indicator = null;
        
        let shaderFile = this.dir.get_child('shader.glsl');
        
        // Asynchronous file loading
        shaderFile.load_contents_async(null, (file, result) => {
            try {
                let [ok, contents] = file.load_contents_finish(result);
                if (!ok) {
                    throw new Error("Failed to load shader file");
                }
                
                let shaderSource = new TextDecoder().decode(contents);
                
                this._shaderEffect = new ShaderEffect(shaderSource);
                Main.uiGroup.add_effect_with_name(this._effectName, this._shaderEffect);
                
                this._startAnimation();
                
                this._indicator = new EyeProtectorMenu(this);
                Main.panel.addToStatusArea(this.uuid, this._indicator);
                
                this.setParam1(this._indicator._param1Slider.value);
                this.setParam2(this._indicator._param2Slider.value);
                this.setParam3(this._indicator._param3Slider.value);
                this.setParam4(this._indicator._param4Slider.value);
                
                console.log('Eye Protector extension enabled successfully');
            } catch (e) {
                console.error('Eye Protector: Failed to load shader.glsl: ' + e);
                console.error(e.stack);
            }
        });
    }

    disable() {
        if (this._timeoutId) {
            GLib.source_remove(this._timeoutId);
            this._timeoutId = null;
        }

        if (this._effectName) {
            Main.uiGroup.remove_effect_by_name(this._effectName);
            this._shaderEffect = null;
            this._effectName = null;
        }
    
        if (this._indicator) {
            this._indicator.destroy();
            this._indicator = null;
        }
        
        console.log('Eye Protector extension disabled');
    }

    _startAnimation() {
        if (this._timeoutId) {
            GLib.source_remove(this._timeoutId);
            this._timeoutId = null;
        }

        this._timeoutId = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 16, () => {
            this._time += 0.016;
            if (this._shaderEffect)
                this._shaderEffect.updateTime(this._time);
            return GLib.SOURCE_CONTINUE;
        });
    }

    setParam1(value) {
        if (this._shaderEffect) {
            this._shaderEffect.param1 = value;
        }
    }

    setParam2(value) {
        if (this._shaderEffect) {
            this._shaderEffect.param2 = value;
        }
    }
    
    setParam3(value) {
        if (this._shaderEffect) {
            this._shaderEffect.param3 = value;
        }
    }
    
    setParam4(value) {
        if (this._shaderEffect) {
            this._shaderEffect.param4 = value;
        }
    }

    toggleEffect(enabled) {
        if (this._shaderEffect) {
            this._shaderEffect.set_enabled(enabled);
        }
    }
}
