// providers/LibreLinkProvider.js - SECURE VERSION
import Soup from 'gi://Soup';
import GLib from 'gi://GLib';
import { BaseProvider } from './BaseProvider.js';


const REGIONAL_URLS = {
    'EU': 'https://api-eu.libreview.io',
    'US': 'https://api.libreview.io',
    'DE': 'https://api-de.libreview.io',
    'FR': 'https://api-fr.libreview.io',
    'JP': 'https://api-jp.libreview.io',
    'AP': 'https://api-ap.libreview.io',
    'AU': 'https://api-au.libreview.io',
    'RU': 'https://api.libreview.ru',
};

const REQUIRED_HEADERS = {
    'User-Agent': 'Mozilla/5.0 (iPhone; CPU OS 17_4.1 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Version/17.4.1 Mobile/10A5355d Safari/8536.25',
    'accept-encoding': 'gzip',
    'cache-control': 'no-cache',
    'connection': 'Keep-Alive',
    'content-type': 'application/json',
    'product': 'llu.ios',
    'version': '4.16.0',
};

export class LibreLinkProvider extends BaseProvider {
    constructor(config, log) {
        super(config, log);
        this.session = new Soup.Session({ 
            timeout: 15,
            max_conns: 10,
            max_conns_per_host: 5
        });
        this.authToken = null;
        this.tokenExpiry = null;
        this.accountId = null;
    }



    isConfigured() {
        const librelinkConfig = this.config.get('librelink');
        return !!(librelinkConfig && librelinkConfig.email);
        // Note: We don't check password here since it's stored in keyring
    }

    getRequiredConfig() {
        return ['librelink.email']; // Password is handled separately via keyring
    }

    _getApiUrl() {
        const librelinkConfig = this.config.get('librelink');
        const region = librelinkConfig.region || 'EU';
        return REGIONAL_URLS[region] || REGIONAL_URLS['EU'];
    }

    _isTokenValid() {
        return this.authToken && this.tokenExpiry && new Date() < new Date(this.tokenExpiry);
    }

    async _getPassword() {
        try {
            // Get password from secure keyring instead of config
            const password = await this.config.getLibreLinkPassword();
            if (!password) {
                throw new Error('No LibreLink password found in keyring. Please check your settings.');
            }
            return password;
        } catch (error) {
            this.log(`Error retrieving password from keyring: ${error.message}`);
            throw error;
        }
    }

    async _login() {
        return new Promise(async (resolve, reject) => {
            // Get configuration and password FIRST, before creating the URL
            const librelinkConfig = this.config.get('librelink') || {};
            const apiUrl = this._getApiUrl();
            const loginUrl = `${apiUrl}/llu/auth/login`; 

            this.log(`LibreLink login attempt:`);
            this.log(`- API URL: ${apiUrl}`);
            this.log(`- Region: ${librelinkConfig.region || 'EU'}`);
            this.log(`- Email: ${librelinkConfig.email ? `${librelinkConfig.email.substring(0, 3)}***` : 'NOT SET'}`);
            this.log(`- Password: [RETRIEVING FROM KEYRING]`);

            try {
                // Get password securely from keyring
                const password = await this._getPassword();
                
                const loginData = {
                    email: librelinkConfig.email,
                    password: password
                };

                this.log(`Sending login request to: ${loginUrl}`);
                const message = Soup.Message.new('POST', loginUrl);
                
                // Set required headers
                Object.entries(REQUIRED_HEADERS).forEach(([key, value]) => {
                    message.get_request_headers().append(key, value);
                });

                const requestBody = JSON.stringify(loginData);
                message.set_request_body_from_bytes('application/json', 
                    new GLib.Bytes(new TextEncoder().encode(requestBody)));

                this.session.send_and_read_async(message, GLib.PRIORITY_DEFAULT, null, (session, result) => {
                    try {
                        const bytes = session.send_and_read_finish(result);
                        const decoder = new TextDecoder('utf-8');
                        const response = decoder.decode(bytes.get_data());
                        
                        const status = message.get_status();
                        this.log(`HTTP Status: ${status}`);
                        
                        if (!response || response.trim() === '') {
                            throw new Error(`Empty response from LibreLink login (HTTP ${status})`);
                        }
                        
                        const data = JSON.parse(response);
                        this.log(`Parsed response status: ${data.status}`);
                        
                        if (data.status !== 0) {
                            this.log(`Login error details: ${JSON.stringify(data, null, 2)}`);
                            throw new Error(`Login failed: ${data.error?.description || data.error || 'Unknown error'}`);
                        }

                        if (!data.data || !data.data.authTicket) {
                            this.log(`Invalid response structure: ${JSON.stringify(data, null, 2)}`);
                            throw new Error('Invalid login response: no auth ticket');
                        }

                        this.authToken = data.data.authTicket.token;
                        this.tokenExpiry = new Date(data.data.authTicket.expires * 1000);
                        
                        const userId = data.data.user.id.toString();
                        this.accountId = GLib.compute_checksum_for_string(GLib.ChecksumType.SHA256, userId, -1);

                        this.log(`LibreLink login successful!`);
                        this.log(`Token expires: ${this.tokenExpiry}`);
                        resolve();
                        
                    } catch (error) {
                        this.log(`LibreLink login error: ${error.message}`);
                        reject(error);
                    }
                });
            } catch (error) {
                this.log(`Failed to retrieve password from keyring: ${error.message}`);
                reject(error);
            }
        });
    }

    // Add option to select Patient
    async getPatients() {
        await this._ensureAuthenticated();         
        const apiUrl = this._getApiUrl();
        const url = `${apiUrl}/llu/connections`;

        return new Promise((resolve, reject) => {
            const message = Soup.Message.new('GET', url);
            // Set required headers including auth
            Object.entries(REQUIRED_HEADERS).forEach(([key, value]) => {
                message.get_request_headers().append(key, value);
            });
            message.get_request_headers().append('authorization', `Bearer ${this.authToken}`);
            message.get_request_headers().append('account-id', this.accountId);

            this.session.send_and_read_async(message, GLib.PRIORITY_DEFAULT, null, (session, result) => {
                try {
                    const bytes = session.send_and_read_finish(result);
                    const decoder = new TextDecoder('utf-8');
                    const response = decoder.decode(bytes.get_data());

                    const status = message.get_status();
                    this.log(`getPatients HTTP Status: ${status}`);
                    
                    // Check HTTP status first
                    if (status !== 200) {
                        throw new Error(`HTTP ${status}: Failed to fetch patients`);
                    }

                    if (!response || response.trim() === '') {
                        throw new Error('Empty response from LibreLink connections');
                    }

                    const data = JSON.parse(response);
                    this.log(`Raw LibreLink connections: ${JSON.stringify(data.data, null, 2)}`);
                    
                    // Check API response status
                    if (data.status !== 0) {
                        throw new Error(`API Error: ${data.error?.description || data.error || 'Unknown error'}`);
                    }
                    
                    if (!data.data || !Array.isArray(data.data) || data.data.length === 0) {
                        throw new Error('No connections found.');
                    }

                    // Map name
                    resolve(
                        data.data.map(conn => ({
                            id: conn.patientId,
                            name: (conn.firstName && conn.lastName)
                                ? `${conn.firstName} ${conn.lastName}`
                                : conn.patientId // fallback if no name
                        }))
                    );
                } catch (error) {
                    this.log(`getPatients error: ${error.message}`);
                    reject(error);
                }
            });
        });
    }
    
    // Only use config value for patientId
    async _getPatientId() {
        const librelinkConfig = this.config.get('librelink');
        if (librelinkConfig && librelinkConfig.patientId) {
            return librelinkConfig.patientId;
        }
        
        // If no patient selected, get list and use first one
        try {
            const patients = await this.getPatients();
            if (patients && patients.length > 0) {
                const currentConfig = this.config.get('librelink') || {};
                currentConfig.patientId = patients[0].id;
                this.config.set('librelink', currentConfig);
                return patients[0].id;
            }
        } catch (error) {
            this.log(`Failed to auto-select patient: ${error.message}`);
        }
        
        throw new Error("No patientId configured and no patients available.");
    }

    async _ensureAuthenticated() {
        // Login if needed
        if (!this._isTokenValid()) {
            await this._login();
        }
    }

    async _fetchGlucoseData() {
        // Get patientId FIRST before creating the Promise
        const patientId = await this._getPatientId();
        const apiUrl = this._getApiUrl();
        const url = `${apiUrl}/llu/connections/${patientId}/graph`;

        return new Promise((resolve, reject) => {
            this.log(`Fetching LibreLink glucose data: ${url}`);
            const message = Soup.Message.new('GET', url);
            
            // Set required headers including auth
            Object.entries(REQUIRED_HEADERS).forEach(([key, value]) => {
                message.get_request_headers().append(key, value);
            });
            message.get_request_headers().append('authorization', `Bearer ${this.authToken}`);
            message.get_request_headers().append('account-id', this.accountId);

            this.session.send_and_read_async(message, GLib.PRIORITY_DEFAULT, null, (session, result) => {
                try {
                    const bytes = session.send_and_read_finish(result);
                    const decoder = new TextDecoder('utf-8');
                    const response = decoder.decode(bytes.get_data());
                    
                    if (!response || response.trim() === '') {
                        throw new Error('Empty response from LibreLink glucose data');
                    }
                    
                    const data = JSON.parse(response);
                    
                    if (data.status !== 0) {
                        throw new Error(`Failed to get glucose data: ${data.error || 'Unknown error'}`);
                    }

                    if (!data.data) {
                        throw new Error('No glucose data in response');
                    }

                    this.log(`LibreLink data received with ${data.data.graphData ? data.data.graphData.length : 0} history entries`);
                    resolve(data.data);
                    
                } catch (error) {
                    this.log(`LibreLink glucose data error: ${error.message}`);
                    reject(error);
                }
            });
        });
    }

    async fetchCurrent() {
        await this._ensureAuthenticated();
        const data = await this._fetchGlucoseData();
        
        // Convert LibreLink current reading to Nightscout format
        if (data.connection && data.connection.glucoseMeasurement) {
            const reading = data.connection.glucoseMeasurement;
            this.log(`Current reading: ${reading.ValueInMgPerDl} mg/dL, Trend: ${reading.TrendArrow}`);
            return this._convertToNightscoutFormat(reading, true);
        }
        
        throw new Error('No current glucose measurement found');
    }

    async fetchHistory() {
        await this._ensureAuthenticated();
        const data = await this._fetchGlucoseData();
        
        // Convert LibreLink history to Nightscout format
        if (data.graphData && Array.isArray(data.graphData)) {
            this.log(`Converting ${data.graphData.length} history entries`);
            return data.graphData.map(reading => this._convertToNightscoutFormat(reading, false));
        }
        
        throw new Error('No glucose history found');
    }

    _convertToNightscoutFormat(librelinkReading, isCurrent = false) {
        // LibreLink uses PascalCase field names
        const result = {
            sgv: librelinkReading.ValueInMgPerDl || librelinkReading.Value,
            dateString: librelinkReading.Timestamp,
            date: new Date(librelinkReading.Timestamp).getTime(),
            type: 'sgv'
        };

        // Only current readings have TrendArrow, not historical data
        if (isCurrent && librelinkReading.TrendArrow !== undefined) {
            result.direction = this._convertTrendArrow(librelinkReading.TrendArrow);
        }

        return result;
    }

    _convertTrendArrow(librelinkTrend) {
        // LibreLink trend arrow values (INVERTED - lower numbers = falling, higher = rising):
        // 1 = SingleDown (falling)
        // 2 = DoubleDown (falling rapidly) - ASSUMPTION, needs verification
        // 3 = Flat (stable)
        // 4 = SingleUp (rising) - ASSUMPTION, needs verification
        // 5 = DoubleUp (rising rapidly)
        switch (librelinkTrend) {
            case 1: return 'SingleDown';
            case 2: return 'DoubleDown';
            case 3: return 'Flat';
            case 4: return 'SingleUp';
            case 5: return 'DoubleUp';
            default: return 'NONE';
        }
    }

    getCgmInterval() {
        // LibreLink typically updates every 1 minute for Libre 3, 15 minutes for older versions
        // We could detect this from the data, but 1 minute is a safe default
        return 1;
    }


    destroy() {
        if (this.session) {
            this.session.abort();
            this.session = null;
        }
        this.authToken = null;
        this.tokenExpiry = null;
        this.accountId = null;
        }
}