/*
Copyright (C) 2014  spin83

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, visit https://www.gnu.org/licenses/.
*/

import Clutter from 'gi://Clutter';
import Gio from 'gi://Gio';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import {ANIMATION_TIME} from 'resource:///org/gnome/shell/ui/overview.js';
import {Extension} from 'resource:///org/gnome/shell/extensions/extension.js';
import * as PanelModule from 'resource:///org/gnome/shell/ui/panel.js';

import * as MMLayout from './mmlayout.js';
import * as MMOverview from './mmoverview.js';
import * as MMIndicator from './indicator.js';
import * as MMPanel from './mmpanel.js';

const MUTTER_SCHEMA = 'org.gnome.mutter';
const WORKSPACES_ONLY_ON_PRIMARY_ID = 'workspaces-only-on-primary';

const SHOW_INDICATOR_ID = 'show-indicator';
const THUMBNAILS_SLIDER_POSITION_ID = 'thumbnails-slider-position';

export function patchAddActorMethod(prototype) {
    if (!prototype.add_actor) {
        if (prototype.add_child) {
            prototype.add_actor = function(actor) {
                return this.add_child(actor);
            };
        } else {
            let parent = Object.getPrototypeOf(prototype);
            if (parent && parent.add_child) {
                prototype.add_actor = function(actor) {
                    return this.add_child(actor);
                };
            }
        }
    }
}

export function copyClass (s, d) {
	if (!s) {
		return;
	}
    let propertyNames = Reflect.ownKeys(s.prototype);
    for (let pName of propertyNames.values()) {
        if (typeof pName === "symbol") continue;
        if (Object.prototype.hasOwnProperty.call(d.prototype, pName)) continue;
        if (pName === "prototype") continue;
        if (pName === "constructor") continue;
        let pDesc = Reflect.getOwnPropertyDescriptor(s.prototype, pName);
        if (typeof pDesc !== 'object') continue;
        Reflect.defineProperty(d.prototype, pName, pDesc);
    }

    patchAddActorMethod(d.prototype);
};

export let mmPanel = [];
export let mmOverview = null;
export let mmLayoutManager = null;

export default class MultiMonitorsExtension extends Extension {
    constructor(metadata) {
        super(metadata);
        this._settings = null;
        this._mu_settings = null;
        this.mmIndicator = null;
        this._mmMonitors = 0;
        this.syncWorkspacesActualGeometry = null;
        
        this._switchOffThumbnailsMuId = null;
        this._showIndicatorId = null;
        this._showPanelId = null;
        this._thumbnailsSliderPositionId = null;
        this._relayoutId = null;
    }

    _showIndicator() {
		if(this._settings.get_boolean(SHOW_INDICATOR_ID)) {
			if(!this.mmIndicator) {
				this.mmIndicator = Main.panel.addToStatusArea('MultiMonitorsAddOn', new MMIndicator.MultiMonitorsIndicator(this._settings, this.path));
			}
		}
		else {
			this._hideIndicator();
		}
    }

    _hideIndicator() {
		if(this.mmIndicator) {
			this.mmIndicator.destroy();
			this.mmIndicator = null;
		}
    }

    _showThumbnailsSlider() {
		if (this._settings.get_string(THUMBNAILS_SLIDER_POSITION_ID) === 'none') {
			this._hideThumbnailsSlider();
			return;
		}

		if(this._mu_settings.get_boolean(WORKSPACES_ONLY_ON_PRIMARY_ID))
			this._mu_settings.set_boolean(WORKSPACES_ONLY_ON_PRIMARY_ID, false);

		if (mmOverview)
			return;

		mmOverview = [];

		for (let idx = 0; idx < Main.layoutManager.monitors.length; idx++) {
			if (idx != Main.layoutManager.primaryIndex) {
				mmOverview[idx] = new MMOverview.MultiMonitorsOverview(idx, this._settings);
			}
		}

		if (Main.overview.searchController && 
			Main.overview.searchController._workspacesDisplay &&
			Main.overview.searchController._workspacesDisplay._syncWorkspacesActualGeometry) {
			this.syncWorkspacesActualGeometry = Main.overview.searchController._workspacesDisplay._syncWorkspacesActualGeometry;
			Main.overview.searchController._workspacesDisplay._syncWorkspacesActualGeometry = function() {
				if (this._inWindowFade)
					return;

				const primaryView = this._getPrimaryView();
				if (primaryView) {
					primaryView.ease({
						...this._actualGeometry,
						duration: Main.overview.animationInProgress ? ANIMATION_TIME : 0,
						mode: Clutter.AnimationMode.EASE_OUT_QUAD,
					});
				}

				if (mmOverview) {
					for (let idx = 0; idx < mmOverview.length; idx++) {
						if (!mmOverview[idx])
							continue;
						if (!mmOverview[idx]._overview)
							continue;
						const mmView = mmOverview[idx]._overview._controls._workspacesViews;
						if (!mmView)
							continue;

						const mmGeometry = mmOverview[idx].getWorkspacesActualGeometry();
						mmView.ease({
							...mmGeometry,
							duration: Main.overview.animationInProgress ? ANIMATION_TIME : 0,
							mode: Clutter.AnimationMode.EASE_OUT_QUAD,
						});
					}
				}
			}
		} else {
			this.syncWorkspacesActualGeometry = null;
		}
	}

	_hideThumbnailsSlider() {
        if (!mmOverview)
            return;

        for (let idx = 0; idx < mmOverview.length; idx++) {
            if (mmOverview[idx])
                mmOverview[idx].destroy();
        }
        mmOverview = null;
        
        if (this.syncWorkspacesActualGeometry &&
            Main.overview.searchController &&
            Main.overview.searchController._workspacesDisplay) {
            Main.overview.searchController._workspacesDisplay._syncWorkspacesActualGeometry = this.syncWorkspacesActualGeometry;
        }
    }

    _relayout() {
		if(this._mmMonitors!=Main.layoutManager.monitors.length){
			this._mmMonitors = Main.layoutManager.monitors.length;
			this._hideThumbnailsSlider();
			this._showThumbnailsSlider();
		}
    }

    _switchOffThumbnails() {
		if (this._mu_settings.get_boolean(WORKSPACES_ONLY_ON_PRIMARY_ID)) {
			this._settings.set_string(THUMBNAILS_SLIDER_POSITION_ID, 'none');
		}
    }

    enable() {
		if(Main.panel.statusArea.MultiMonitorsAddOn)
			this.disable();
		
		this._mmMonitors = 0;

		this._settings = this.getSettings();
		this._mu_settings = new Gio.Settings({ schema: MUTTER_SCHEMA });

		this._switchOffThumbnailsMuId = this._mu_settings.connect('changed::'+WORKSPACES_ONLY_ON_PRIMARY_ID,
																	this._switchOffThumbnails.bind(this));

		this._showIndicatorId = this._settings.connect('changed::'+SHOW_INDICATOR_ID, this._showIndicator.bind(this));
		this._showIndicator();

		mmLayoutManager = new MMLayout.MultiMonitorsLayoutManager(this._settings);
		
		this._showPanelId = this._settings.connect('changed::'+MMLayout.SHOW_PANEL_ID, mmLayoutManager.showPanel.bind(mmLayoutManager));
		mmLayoutManager.showPanel();
		
		this._thumbnailsSliderPositionId = this._settings.connect('changed::'+THUMBNAILS_SLIDER_POSITION_ID, this._showThumbnailsSlider.bind(this));
		this._relayoutId = Main.layoutManager.connect('monitors-changed', this._relayout.bind(this));
		this._relayout();

        mmPanel.length = 0;
        MMLayout.setMMPanelArrayRef(mmPanel);
        MMPanel.setMMPanelArrayRef(mmPanel);
        MMOverview.setMMPanelArrayRef(mmPanel);

        Main.panel._ensureIndicator = function(role) {
            let indicator = this.statusArea[role];
            if (indicator) {
                indicator.container.show();
                return null;
            }
            else {
				let constructor = PanelModule.PANEL_ITEM_IMPLEMENTATIONS[role];
                if (!constructor) {
                    return null;
                }
                indicator = new constructor(this);
                this.statusArea[role] = indicator;
            }
            return indicator;
        };
    }

    disable() {
		if (this._relayoutId) {
			Main.layoutManager.disconnect(this._relayoutId);
			this._relayoutId = null;
		}
		
		if (this._switchOffThumbnailsMuId) {
			this._mu_settings.disconnect(this._switchOffThumbnailsMuId);
			this._switchOffThumbnailsMuId = null;
		}
		
		if (this._showPanelId) {
			this._settings.disconnect(this._showPanelId);
			this._showPanelId = null;
		}
		
		if (this._thumbnailsSliderPositionId) {
			this._settings.disconnect(this._thumbnailsSliderPositionId);
			this._thumbnailsSliderPositionId = null;
		}
		
		if (this._showIndicatorId) {
			this._settings.disconnect(this._showIndicatorId);
			this._showIndicatorId = null;
		}

		this._hideIndicator();

		if (mmLayoutManager) {
			mmLayoutManager.hidePanel();
			mmLayoutManager = null;
		}

		this._hideThumbnailsSlider();
		this._mmMonitors = 0;

        mmPanel.length = 0;
		
		this._settings = null;
		this._mu_settings = null;
    }
}
