'use strict';

import Gio from 'gi://Gio';
import Gtk from 'gi://Gtk';
import Adw from 'gi://Adw';
import Gdk from 'gi://Gdk';
import {
    ExtensionPreferences,
    gettext as _
} from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';

export default class ClassicAppSwitcherPrefs extends ExtensionPreferences {
    fillPreferencesWindow(window) {
        const settings = this.getSettings();

        // Register custom icon directory with the icon theme
        const iconTheme = Gtk.IconTheme.get_for_display(Gdk.Display.get_default());
        iconTheme.add_search_path(this.path + '/icons');

        // ============================================================
        // GENERAL PAGE
        // ============================================================
        const page = new Adw.PreferencesPage({
            title: _('Panel Indicator'),
            icon_name: 'focus-top-bar-symbolic',
        });

        // Create branded header group
        const headerGroup = new Adw.PreferencesGroup();

        // Create header box with icon and title
        const headerBox = new Gtk.Box({
            orientation: Gtk.Orientation.VERTICAL,
            spacing: 12,
            margin_top: 24,
            margin_bottom: 24,
            halign: Gtk.Align.CENTER,
            valign: Gtk.Align.CENTER
        });

        // Use the registered icon - symbolic icons will be recolored by theme
        const iconWidget = new Gtk.Image({
            icon_name: 'happy-computer-symbolic',
            pixel_size: 96,
            use_fallback: false
        });

        headerBox.append(iconWidget);

        // Add extension name
        const titleLabel = new Gtk.Label({
            label: '<span size="x-large" weight="bold">' + _('Classic App Switcher') + '</span>',
            use_markup: true,
            halign: Gtk.Align.CENTER
        });
        headerBox.append(titleLabel);

        // Add subtitle
        const subtitleLabel = new Gtk.Label({
            label: _('Mac OS 9-style application switching for GNOME'),
            halign: Gtk.Align.CENTER
        });
        subtitleLabel.add_css_class('dim-label');
        headerBox.append(subtitleLabel);

        headerGroup.add(headerBox);
        page.add(headerGroup);

        // Appearance Group
        const appearanceGroup = new Adw.PreferencesGroup({
            title: _('Appearance'),
            description: _('Customize how the switcher looks'),
        });

        // Show Label Row
        const showLabelRow = new Adw.SwitchRow({
            title: _('Show Application Name'),
            subtitle: _('Display the name of the focused application in the panel'),
        });
        settings.bind(
            'show-label',
            showLabelRow,
            'active',
            Gio.SettingsBindFlags.DEFAULT
        );
        appearanceGroup.add(showLabelRow);
        page.add(appearanceGroup);

        // Position Group
        const positionGroup = new Adw.PreferencesGroup({
            title: _('Panel Position'),
            description: _('Configure where the switcher appears in the panel'),
        });

        // Panel Box Row
        const panelBoxRow = new Adw.ComboRow({
            title: _('Panel Area'),
            subtitle: _('Choose which section of the panel to use'),
        });

        const panelBoxModel = new Gtk.StringList();
        panelBoxModel.append(_('Left'));
        panelBoxModel.append(_('Center'));
        panelBoxModel.append(_('Right'));
        panelBoxRow.model = panelBoxModel;

        // Set initial value
        const panelBoxValue = settings.get_string('panel-box');
        const boxIndex = ['left', 'center', 'right'].indexOf(panelBoxValue);
        panelBoxRow.selected = boxIndex >= 0 ? boxIndex : 2;

        // Connect change handler
        panelBoxRow.connect('notify::selected', (widget) => {
            const selected = ['left', 'center', 'right'][widget.selected];
            settings.set_string('panel-box', selected);
        });

        positionGroup.add(panelBoxRow);

        // Position Offset Row
        const positionOffsetRow = new Adw.SpinRow({
            title: _('Position Offset'),
            subtitle: _('Fine-tune position (0 = default, negative = left, positive = right)'),
            adjustment: new Gtk.Adjustment({
                lower: -10,
                upper: 10,
                step_increment: 1,
                page_increment: 1,
                value: settings.get_int('position-in-box'),
            }),
        });

        settings.bind(
            'position-in-box',
            positionOffsetRow,
            'value',
            Gio.SettingsBindFlags.DEFAULT
        );

        positionGroup.add(positionOffsetRow);
        page.add(positionGroup);

        // Add the General page to the window
        window.add(page);

        // ============================================================
        // KEYBOARD SHORTCUTS PAGE
        // ============================================================
        const keyboardPage = new Adw.PreferencesPage({
            title: _('Keyboard Access'),
            icon_name: 'keyboard-shortcuts-symbolic',
        });

        // Keyboard Shortcuts Group
        const keyboardGroup = new Adw.PreferencesGroup({
            title: _('Keyboard Shortcuts'),
            description: _('Mac-style keyboard shortcuts for application and window management'),
        });

        // Enable Keyboard Shortcuts Row
        const enableShortcutsRow = new Adw.SwitchRow({
            title: _('Enable Keyboard Shortcuts'),
            subtitle: _('Activate Mac-style shortcuts for enhanced window management'),
        });
        settings.bind(
            'enable-keyboard-shortcuts',
            enableShortcutsRow,
            'active',
            Gio.SettingsBindFlags.DEFAULT
        );
        keyboardGroup.add(enableShortcutsRow);

        // Show Keyboard Hints Row
        const showHintsRow = new Adw.SwitchRow({
            title: _('Show Keyboard Hints'),
            subtitle: _('Display shortcut keys next to menu items'),
        });
        settings.bind(
            'show-keyboard-hints',
            showHintsRow,
            'active',
            Gio.SettingsBindFlags.DEFAULT
        );

        // Bind sensitivity to enable-keyboard-shortcuts
        settings.bind(
            'enable-keyboard-shortcuts',
            showHintsRow,
            'sensitive',
            Gio.SettingsBindFlags.GET
        );

        keyboardGroup.add(showHintsRow);
        keyboardPage.add(keyboardGroup);

        // Shortcut Reference Group
        const referenceGroup = new Adw.PreferencesGroup({
            title: _('Configuration'),
            description: _('The following shortcuts are available when enabled above'),
        });

        // Create info box with shortcut list
        const shortcutsBox = new Gtk.Box({
            orientation: Gtk.Orientation.VERTICAL,
            spacing: 8,
            margin_top: 12,
            margin_bottom: 12,
            margin_start: 12,
            margin_end: 12,
        });

        const shortcuts = [{
                key: _('⌥ Super+H'),
                desc: _('Hide the current application')
            },
            {
                key: _('⌥ Alt+Super+H'),
                desc: _('Hide all other applications')
            },
            {
                key: _('⌥ Super+U'),
                desc: _('Show most recently hidden application')
            },
            {
                key: _('⌥ Alt+Super+U'),
                desc: _('Show all hidden applications/windows')
            },
            {
                key: _('⌥ Super+M'),
                desc: _('Minimize the current window')
            },
            {
                key: _('⌥ Alt+Super+M'),
                desc: _('Unminimize most recently minimized window')
            },
            {
                key: _('⌥ Super+W'),
                desc: _('Close the current window')
            },
            {
                key: _('⌥ Super+Q'),
                desc: _('Quit the current application')
            },
        ];

        shortcuts.forEach(shortcut => {
            const row = new Gtk.Box({
                orientation: Gtk.Orientation.HORIZONTAL,
                spacing: 12,
            });

            const keyLabel = new Gtk.Label({
                label: `<tt>${shortcut.key}</tt>`,
                use_markup: true,
                xalign: 0,
                width_chars: 20,
            });
            keyLabel.add_css_class('monospace');

            const descLabel = new Gtk.Label({
                label: shortcut.desc,
                xalign: 0,
                hexpand: true,
                wrap: true,
            });
            descLabel.add_css_class('dim-label');

            row.append(keyLabel);
            row.append(descLabel);
            shortcutsBox.append(row);
        });

        referenceGroup.add(shortcutsBox);
        keyboardPage.add(referenceGroup);

        // Warning Group
        const warningGroup = new Adw.PreferencesGroup();

        const warningBox = new Gtk.Box({
            orientation: Gtk.Orientation.HORIZONTAL,
            spacing: 12,
            margin_top: 12,
            margin_bottom: 12,
            margin_start: 12,
            margin_end: 12,
        });

        const warningIcon = new Gtk.Image({
            icon_name: 'dialog-warning-symbolic',
            pixel_size: 16,
        });

        const warningLabel = new Gtk.Label({
            label: _('Note: Enabling shortcuts will remap the system default for Super+H'),
            wrap: true,
            xalign: 0,
            hexpand: true,
        });
        warningLabel.add_css_class('dim-label');

        warningBox.append(warningIcon);
        warningBox.append(warningLabel);
        warningGroup.add(warningBox);
        keyboardPage.add(warningGroup);

        // Add the Keyboard Shortcuts page to the window
        window.add(keyboardPage);
    }
}
