

import Gtk from 'gi://Gtk';
import Gdk from 'gi://Gdk';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import Adw from 'gi://Adw';
import { ExtensionPreferences, gettext as _ } from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';

const SETTINGS_APPLY_DELAY_TIME = 500;

export default class AppGridTunerPrefs extends ExtensionPreferences {
  fillPreferencesWindow(window) {
    const settings = this.getSettings();

    let reloadSourceId = null;
    const scheduleReload = () => {
      if (reloadSourceId)
        GLib.source_remove(reloadSourceId);
      reloadSourceId = GLib.timeout_add(GLib.PRIORITY_DEFAULT, SETTINGS_APPLY_DELAY_TIME, () => {
        settings.set_boolean('reload-signal', !settings.get_boolean('reload-signal'));
        reloadSourceId = null;
        return GLib.SOURCE_REMOVE;
      });
    };

    const addSpinIntRow = (group, key, min, max, step) => {
      const row = new Adw.ActionRow({
        title: settings.settings_schema.get_key(key).get_summary(),
      });
      const spin = Gtk.SpinButton.new_with_range(min, max, step);
      spin.set_value(settings.get_int(key));
      spin.connect('notify::value', (w) => {
        settings.set_int(key, w.get_value_as_int());
        scheduleReload();
      });
      spin.set_valign(Gtk.Align.CENTER);
      row.add_suffix(spin);
      group.add(row);
    };

    const addSpinDoubleRow = (group, key, min, max, step) => {
      const row = new Adw.ActionRow({
        title: settings.settings_schema.get_key(key).get_summary(),
      });
      const spin = Gtk.SpinButton.new_with_range(min, max, step);
      spin.set_value(settings.get_double(key));
      spin.connect('notify::value', (w) => {
        settings.set_double(key, w.get_value());
        scheduleReload();
      });
      spin.set_valign(Gtk.Align.CENTER);
      row.add_suffix(spin);
      group.add(row);
    };

    const addComboRow = (group, key, options, labels) => {
      const row = new Adw.ActionRow({
        title: settings.settings_schema.get_key(key).get_summary(),
      });
      const combo = new Gtk.ComboBoxText();
      for (let i = 0; i < options.length; i++)
        combo.append(options[i], labels[i]);
      combo.set_active(options.indexOf(settings.get_string(key)));
      combo.connect('changed', (w) => {
        settings.set_string(key, options[w.get_active()]);
        scheduleReload();
      });
      combo.set_valign(Gtk.Align.CENTER);
      row.add_suffix(combo);
      group.add(row);
    };

    const addColorRow = (group, key) => {
      const row = new Adw.ActionRow({
        title: settings.settings_schema.get_key(key).get_summary(),
      });
      
      const colorButton = new Gtk.ColorDialogButton();
      const dialog = new Gtk.ColorDialog();
      colorButton.set_dialog(dialog);
      colorButton.set_valign(Gtk.Align.CENTER);

      let isUpdating = false;

      const updatePickerVisuals = (hex) => {
          isUpdating = true;
          const rgba = new Gdk.RGBA();
          if (hex && hex.startsWith('#')) {
            rgba.parse(hex);
          } else {
            rgba.parse('#ffffff');
          }
          colorButton.set_rgba(rgba);
          isUpdating = false;
      };

      // Initial load
      updatePickerVisuals(settings.get_string(key));

      colorButton.connect('notify::rgba', (btn) => {
        if (isUpdating) return;
        
        const c = btn.get_rgba();
        const toHex = (v) => {
          const hex = Math.round(v * 255).toString(16);
          return hex.length === 1 ? '0' + hex : hex;
        };
        const hexString = `#${toHex(c.red)}${toHex(c.green)}${toHex(c.blue)}`;
        settings.set_string(key, hexString);
        scheduleReload();
      });

      const resetButton = new Gtk.Button({
        icon_name: 'edit-clear-symbolic',
        tooltip_text: _('Reset to system default'),
        valign: Gtk.Align.CENTER,
      });
      resetButton.add_css_class('flat');
      
      resetButton.connect('clicked', () => {
          updatePickerVisuals(''); // Set to white visually
          settings.set_string(key, '');
          scheduleReload();
      });

      row.add_suffix(colorButton);
      row.add_suffix(resetButton);
      group.add(row);
    };

    const addSwitchRow = (group, key) => {
      const row = new Adw.ActionRow({
        title: settings.settings_schema.get_key(key).get_summary(),
      });
      const switchWidget = new Gtk.Switch();
      switchWidget.set_active(settings.get_boolean(key));
      switchWidget.connect('notify::active', (w) => {
        settings.set_boolean(key, w.get_active());
        scheduleReload();
      });
      switchWidget.set_valign(Gtk.Align.CENTER);
      row.add_suffix(switchWidget);
      group.add(row);
    };

    const appsPage = new Adw.PreferencesPage({
      title: _('Apps'),
      icon_name: 'view-grid-symbolic',
    });

    const foldersPage = new Adw.PreferencesPage({
      title: _('Folders'),
      icon_name: 'folder-symbolic',
    });

    const appsGroup = new Adw.PreferencesGroup();
    addSpinIntRow(appsGroup, 'appgrid-max-rows', 1, 20, 1);
    addSpinIntRow(appsGroup, 'appgrid-max-columns', 1, 20, 1);
    addSpinIntRow(appsGroup, 'appgrid-icon-size', 32, 256, 1);
    addSpinIntRow(appsGroup, 'side-padding', -128, 256, 1);
    addSpinDoubleRow(appsGroup, 'app-icon-font-size', 0, 48, 0.1);
    
    addColorRow(appsGroup, 'label-color');

    addComboRow(
      appsGroup,
      'label-style',
      [
        'font-weight: normal;',
        'font-weight: normal; text-shadow: 2px 3px 3px #000000;',
        'font-weight: bold;',
        'font-weight: bold; text-shadow: 2px 3px 3px #000000;',
      ],
      [
        _('Normal'),
        _('Normal with Shadow'),
        _('Bold'),
        _('Bold with Shadow'),
      ],
    );
    addSpinIntRow(appsGroup, 'open-animation-time', 1, 10000, 1);
    addSpinIntRow(appsGroup, 'page-switch-animation-time', 1, 10000, 1);

    const foldersGroup = new Adw.PreferencesGroup();
    addSpinIntRow(foldersGroup, 'folder-max-rows', 1, 20, 1);
    addSpinIntRow(foldersGroup, 'folder-max-columns', 1, 20, 1);
    addSwitchRow(foldersGroup, 'hide-folder-title');
    
    addSpinIntRow(foldersGroup, 'folder-icon-size', 32, 256, 1);
    addSpinIntRow(foldersGroup, 'folder-popup-width', 400, 800, 10);
    addSpinIntRow(foldersGroup, 'folder-popup-height-offset', 50, 300, 5);
    addSpinIntRow(foldersGroup, 'folder-item-spacing', 40, 150, 2);
    addSpinIntRow(foldersGroup, 'folder-item-height-offset', 30, 120, 2);
    addSpinIntRow(foldersGroup, 'folder-side-padding', -128, 256, 2);

    appsPage.add(appsGroup);
    foldersPage.add(foldersGroup);

    window.add(appsPage);
    window.add(foldersPage);
    window.set_default_size(700, 580);
  }
}
