# Window Control Library Integration Guide

This document explains how to integrate the Window Control library into another GNOME Shell extension.

## Overview

The Window Control library consists of:

- **`lib/state-matcher.js`** - Core window tracking and matching logic (`WindowStateMatcher`)
- **`lib/state-session.js`** - Session management (`StateSession`, `OperationHandler`) and state persistence helpers
- **`lib/window-state.js`** - Window state constants and utility functions
- **`lib/utils.js`** - Debugging and debounce utilities
- **`lib/gnome-shell.js`** - GNOME Shell integration (`ShellWindowMonitor`, `ShellWindowExecutor`, `getWindowState`)
- **`lib/window-manager.js`** - High-level `WindowManager` class for complete integration
- **`lib/dbus.js`** - D-Bus interface (`DbusExecutor`, `DbusClient`) for external clients

## Core Library API

### WindowManager Class (Recommended)

**WindowManager** is the simplest way to integrate window tracking into a GNOME Shell extension. It handles all monitor setup, event processing, and state management automatically.

```javascript
import { WindowManager } from './lib/window-manager.js'
import { createFileStorage } from './lib/state-session.js'
import GLib from 'gi://GLib'

export default class YourExtension extends Extension {
    enable() {
        const stateFile = GLib.get_home_dir() + '/.my-extension-state.json'
        const storage = createFileStorage(stateFile)

        this._manager = new WindowManager({
            config: {
                 // Optional: Config overrides
                 SETTLE_IDLE_TIMEOUT: 200,
                 WORKSPACE_SETTLE_TIMEOUT: 500
            },
            stateLoader: storage.stateLoader,
            stateSaver: storage.stateSaver
        })

        this._manager.enable()
    }

    disable() {
        this._manager.disable()
        delete this._manager
    }
}
```

**WindowManager Options:**
- `config` - Tracker configuration overrides
- `stateLoader` - Function to load state: `() => state | null`
- `stateSaver` - Function to save state: `(state) => void`
- `trackerHandler` - Optional custom `OperationHandler` instance
- `executorConfig` - Configuration for `ShellWindowExecutor` (e.g., `activate_on_move`)

**WindowManager Methods:**
- `enable()` - Start monitoring windows
- `disable()` - Stop monitoring and cleanup
- `getTracker()` - Access the `WindowStateMatcher` instance
- `getExecutor()` - Access the `ShellWindowExecutor` instance
- `getSession()` - Access the `StateSession` instance
- `getStats()` - Get current tracker statistics
- `updateConfig(config)` - Update tracker configuration
- `refreshWindowState()` - Manually refresh state from current windows

### StateSession Class

**StateSession** manages the lifecycle of `WindowStateMatcher` with pluggable state persistence.

```javascript
import { StateSession, createFileStorage } from './lib/state-session.js'
import { ShellWindowExecutor } from './lib/gnome-shell.js'

// Create session with file-based state persistence
const session = new StateSession(new ShellWindowExecutor(), {
    ...createFileStorage('/path/to/state.json'),
    config: {
        SETTLE_MAX_WAIT: 3000,
        WORKSPACE_SETTLE_TIMEOUT: 500,
    }
})

// Process window events
session.onWindowModified(winid, eventType, JSON.stringify(details))

// Access tracker if needed
const tracker = session.getTracker()

// Cleanup
session.destroy()
```

**Constructor options:**
- `executor` - Window executor (`ShellWindowExecutor` or `DbusExecutor`)
- `config` - Tracker configuration overrides
- `stateLoader` - Function to load state: `() => state | null`
- `stateSaver` - Function to save state: `(state) => void`
- `trackerHandler` - Optional custom `OperationHandler` instance
- `actorQueryCallback` - Optional callback to query current windows
- `preserveOccupiedState` - If true, preserve occupied status from saved state
- `readOnly` - If true, prevent state saving
- `operationFilter` - Optional filter for operations: `(op) => boolean`
- `policyCallback` - Optional policy callback for tracker: `(winid, details) => boolean`

**Storage helpers:**
- `createFileStorage(path)` - JSON file storage
- `createGSettingsStorage(settings, key)` - GSettings storage

### State Management

`StateSession` provides methods to manage and clean up the tracked state.

```javascript
// List stale entries (default maxAgeHours = 24)
const staleEntries = session.listStaleEntries(48)

// Clean up stale entries
// Returns { removedSlots: number, staleEntries: Array }
const cleanupStats = session.cleanupStaleEntries(48)

// Remove a specific slot by index
// Returns { success: boolean, message: string, wm_class, title }
const result = session.removeSlot(5)

// Wait until all pending windows are processed or timeout
// Returns true if settled, false if timed out
const settled = session.runUntilSettled(mainLoop, 3000)
```

### WindowStateMatcher Class

For advanced use cases, you can use `WindowStateMatcher` directly.

```javascript
import { WindowStateMatcher } from './lib/state-matcher.js'

// Create a configuration object (optional)
const myConfig = {
    SETTLE_MAX_WAIT: 3000,
    WORKSPACE_SETTLE_TIMEOUT: 500,
    DEFAULT_MATCH_THRESHOLD: 0.8,
    OVERRIDES: {
        'org.gnome.Nautilus': {
            action: 'RESTORE',
            threshold: 0.9
        }
    }
}

// Load initial state from persistent source
const initialState = loadState()

// Create tracker instance with callbacks, config, and initial state
const tracker = new WindowStateMatcher(
    onAsyncProcessing,
    onStateChange,
    myConfig,
    initialState,
    actorQueryCallback,
    preserveOccupiedState,
    policyCallback
)

function onAsyncProcessing(result) {
    // Handle operations that need to be executed
    processTrackerResult(result)
}

function onStateChange(newState) {
    // Persist the new state
    saveState(newState)
}
```

## Integration Guide

### Basic Integration using WindowManager

The easiest way to use this library in another extension is via the `WindowManager` class. It handles all signal connections and event processing for you.

```javascript
import { WindowManager } from './lib/window-manager.js'
import { createFileStorage } from './lib/state-session.js'
import GLib from 'gi://GLib'

export default class YourExtension extends Extension {
    enable() {
        const stateFile = GLib.get_home_dir() + '/.my-extension-state.json'
        const storage = createFileStorage(stateFile)

        this._manager = new WindowManager({
            config: {
                 // Optional: Config overrides
                 SETTLE_IDLE_TIMEOUT: 200
            },
            stateLoader: storage.stateLoader,
            stateSaver: storage.stateSaver
        })

        this._manager.enable()
    }

    disable() {
        this._manager.disable()
        delete this._manager
    }
}
```

### GSettings Integration (Production Ready)

For a proper GNOME Shell extension, you should use GSettings to store the persistent state and configuration.

**1. Define your schema (`org.gnome.shell.extensions.yourextension.gschema.xml`)**

```xml
<schemalist>
  <schema id="org.gnome.shell.extensions.yourextension" path="/org/gnome/shell/extensions/yourextension/">
    <key name="window-state" type="s">
      <default>""</default>
      <summary>Serialized window state</summary>
    </key>
    <key name="settle-max-wait" type="i">
      <default>2500</default>
      <summary>Max wait time for window matching (ms)</summary>
    </key>
    <key name="workspace-settle-timeout" type="i">
      <default>500</default>
      <summary>Timeout for workspace change settling (ms)</summary>
    </key>
  </schema>
</schemalist>
```

**2. Initialize WindowManager with GSettings storage**

```javascript
import { WindowManager } from './lib/window-manager.js'
import { createGSettingsStorage } from './lib/state-session.js'

export default class YourExtension extends Extension {
    enable() {
        this._settings = this.getSettings()

        // Create storage helper that reads/writes to the 'window-state' key
        const storage = createGSettingsStorage(this._settings, 'window-state')

        // Initial configuration from settings
        const config = {
            SETTLE_MAX_WAIT: this._settings.get_int('settle-max-wait'),
            WORKSPACE_SETTLE_TIMEOUT: this._settings.get_int('workspace-settle-timeout'),
        }

        this._manager = new WindowManager({
            config: config,
            stateLoader: storage.stateLoader,
            stateSaver: storage.stateSaver
        })

        this._manager.enable()

        // Listen for configuration changes
        this._settingsId = this._settings.connect('changed', (settings, key) => {
            switch (key) {
                case 'settle-max-wait':
                    this._manager.updateConfig({
                        SETTLE_MAX_WAIT: settings.get_int(key)
                    })
                    break
                case 'workspace-settle-timeout':
                    this._manager.updateConfig({
                        WORKSPACE_SETTLE_TIMEOUT: settings.get_int(key)
                    })
                    break
            }
        })
    }

    disable() {
        if (this._settingsId) {
            this._settings.disconnect(this._settingsId)
        }
        this._manager.disable()
        delete this._manager
        delete this._settings
    }
}
```

### Customizing Behavior

#### Custom Operation Handler

For highly custom behavior, you can override the default `OperationHandler` by passing your own instance. This is useful if you need to modify how tracker operations are executed or want to add custom logging.

Your custom handler should inherit from `OperationHandler` and override the `processTrackerResult` method.

```javascript
import { WindowManager } from './lib/window-manager.js'
import { OperationHandler } from './lib/state-session.js'
import { ShellWindowExecutor } from './lib/gnome-shell.js'

class CustomOperationHandler extends OperationHandler {
    processTrackerResult(result) {
        // Add custom logic here
        console.log('Custom processing!')

        // Call default implementation to execute operations
        super.processTrackerResult(result)
    }
}

// In your extension:
class MyExtension {
    enable() {
        const executor = new ShellWindowExecutor()
        const customHandler = new CustomOperationHandler(executor)

        this._manager = new WindowManager({
            trackerHandler: customHandler
        })
        this._manager.enable()
    }
}
```

#### Operation Filtering

You can filter which operations get executed by providing an `operationFilter` function:

```javascript
const manager = new WindowManager({
    operationFilter: (op) => {
        // Only allow placement operations, skip workspace moves
        if (op.type === 'MoveToWorkspace') return false
        return true
    }
})
```

#### Advanced: State-Only Updates

If you want to run the tracker to update its internal state (and persistent storage) based on the current window layout, but *without* actually moving or resizing any windows, you can provide a dummy `OperationHandler`.

This is useful for "taking a snapshot" of the current state or running a dry-run synchronization.

```javascript
import { StateSession } from './lib/state-session.js'

// A dummy handler that suppresses all operations
const suppressHandler = {
    processTrackerResult: (result) => {
        // We simply ignore the 'operations' list.
        // The tracker has already updated its internal state by this point.

        // You can still log events if you want
        if (result.events.length > 0) {
            console.log(`Processed ${result.events.length} events (operations suppressed)`)
        }
    }
}

// Initialize session with the dummy handler
const session = new StateSession(executor, {
    trackerHandler: suppressHandler,
    // ... other config
})

// When you run updates, the state will be tracked/saved, but windows won't move.
```

## Constants and Utilities

### Window State Constants (from `lib/window-state.js`)

```javascript
import {
    MAXIMIZED_NONE,        // 0
    MAXIMIZED_HORIZONTAL,  // 1
    MAXIMIZED_VERTICAL,    // 2
    MAXIMIZED_BOTH,        // 3
    shouldTrackWindow,
    isNormalWindow,
    shouldDebounceEvent
} from './lib/window-state.js'
```

### Utility Functions

- `shouldTrackWindow(details)` - Returns true if window should be tracked (has wm_class, allows move/resize, valid geometry)
- `isNormalWindow(details)` - Returns true if window is a normal user-facing window (has title, not skip_taskbar)
- `shouldDebounceEvent(eventType)` - Returns true if the event should be debounced (all except 'notify::title' and 'destroy')
- `isValidGeometry(frameRect)` - Returns true if geometry is valid (width > 0, height > 0)

## Configuration Options

The tracker supports these configuration options (passed via `config` parameter):

```javascript
{
    // Window matching timeouts
    SETTLE_IDLE_TIMEOUT: 500,              // ms - idle time before considering window settled
    SETTLE_MAX_WAIT: 2500,                 // ms - max wait time for window matching
    MIN_IDLE_TIME_BEFORE_MATCH: 300,       // ms - minimum idle time before matching
    GENERIC_TITLE_EXTENDED_WAIT: 15000,    // ms - extended wait for windows with generic titles

    // Workspace/monitor move handling
    WORKSPACE_SETTLE_TIMEOUT: 500,         // ms - timeout for workspace/monitor change events

    // Matching thresholds
    MIN_SCORE_SPREAD: 0.6,                 // minimum score difference for confident matches
    AMBIGUOUS_SIMILARITY_THRESHOLD: 0.95,  // threshold for detecting ambiguous windows
    DEFAULT_MATCH_THRESHOLD: 0.8,          // default similarity threshold for matching

    // Title matching
    MIN_SPECIFIC_TITLE_LENGTH: 15,         // minimum length for "specific" titles
    TITLE_CHANGE_SIGNIFICANCE_RATIO: 2.0,  // ratio for detecting significant title changes
    MIN_TITLE_LEN_FOR_PENALTY: 8,          // minimum title length before applying penalty
    TITLE_LEN_PENALTY_RATIO: 0.5,          // ratio for title length penalty
    TITLE_LEN_PENALTY_FACTOR: 0.5,         // penalty factor for short titles

    // State management
    WINDOW_STATE_PATH: 'window_control_state.json',
    SERIALIZE_INTERVAL_MS: 1000,

    // Per-application overrides
    OVERRIDES: {
        'org.gnome.Nautilus': {
            action: 'RESTORE',           // 'RESTORE' or 'IGNORE'
            threshold: 0.9,              // custom match threshold
            match_properties: ['workspace', 'monitor', 'frame_rect', 'maximized']
        }
    },

    // Default behavior
    DEFAULT_SYNC_MODE: 'RESTORE',          // 'RESTORE' or 'IGNORE'

    // Tracked properties
    SIGNIFICANT_PROPS: ['title', 'wm_class'],
    MANAGED_PROPS: [
        'monitor',
        'workspace',
        'frame_rect',
        'minimized',
        'maximized',
        'fullscreen',
        'on_all_workspaces',
        'above',
    ],
}
```

## Event Types & Window Details

The tracker responds to standard GNOME Shell window events. The `windowDetails` object generally contains:

```javascript
{
    "id": 12345,
    "title": "Window Title",
    "wm_class": "app.class.Name",
    "monitor": 0,
    "workspace": 0,
    "frame_rect": { "x": 100, "y": 200, "width": 800, "height": 600 },
    "minimized": false,
    "maximized": 2,                  // 0=none, 1=horizontal, 2=vertical, 3=both
    "maximized_horizontally": false,
    "maximized_vertically": true,
    "fullscreen": false,
    "on_all_workspaces": false,
    "above": false,
    // Additional properties
    "wm_class_instance": "...",
    "gtk_application_id": "...",
    "sandboxed_app_id": "...",
    "allows_move": true,
    "allows_resize": true,
    "can_move": true,
    "can_resize": true,
    "can_maximize": true,
    "can_minimize": true,
    "can_close": true
}
```

## Operations

The tracker emits operations to restore window state:

- **`Place`** - Move and resize window (`args: [x, y, width, height]`)
- **`MoveToWorkspace`** - (`args: [workspaceIndex]`)
- **`MoveToMonitor`** - (`args: [monitorIndex]`)
- **`Maximize`** - (`args: [state]`) - state is 1 (horizontal), 2 (vertical), or 3 (both)
- **`Unmaximize`** - (`args: []`)
- **`Minimize`** - (`args: []`)
- **`SetFullscreen`** - (`args: [state]`)
- **`ToggleFullscreen`** - (`args: []`)
- **`SetOnAllWorkspaces`** - (`args: [state]`)
- **`SetAbove`** - (`args: [state]`)
- **`Close`** - (`args: [isForced]`)
- **`Move`** - Move window without resizing (`args: [x, y]`)

### Operation Deferral for Workspace/Monitor Moves

Operations following a `MoveToWorkspace` or `MoveToMonitor` operation are automatically deferred until the workspace/monitor change completes. This ensures operations like `Place` and `Maximize` execute on the target workspace after the window has finished moving.

The deferral mechanism:
1. Executes the workspace/monitor move immediately
2. Queues subsequent operations for that window
3. Waits for the workspace-changed/monitor-changed event
4. Executes queued operations when event arrives
5. Falls back to timeout (`WORKSPACE_SETTLE_TIMEOUT`, default 500ms) if event doesn't arrive

## D-Bus Interface

For external clients (non-extension code), use the D-Bus interface:

```javascript
import { DbusClient } from './lib/dbus.js'

const client = new DbusClient()
await client.connect()

const executor = client.getExecutor()
const windows = executor.list()
const details = executor.getDetails(winid)

executor.moveToWorkspace(winid, 1)
executor.place(winid, 0, 0, 640, 480)

client.disconnect()
```

The D-Bus interface is exported at:
- **Bus Name**: `org.gnome.Shell`
- **Object Path**: `/org/gnome/Shell/Extensions/WindowControl`
- **Interface**: `org.gnome.Shell.Extensions.WindowControl`

## ShellWindowExecutor Methods

### Read Operations:
- `list()` - List all windows with id, title, wm_class
- `listNormalWindows()` - List normal user-facing windows
- `getDetails(winid)` - Get full window details
- `getFrameRect(winid)` - Get window frame rectangle
- `getBufferRect(winid)` - Get window buffer rectangle
- `getTitle(winid)` - Get window title
- `getFocusedMonitorDetails()` - Get focused monitor info
- `getAllWindowDetails()` - Get all windows with full details (batch operation)

### Write Operations:
- `moveToWorkspace(winid, wsid)`
- `moveToMonitor(winid, mid)`
- `place(winid, x, y, width, height)`
- `move(winid, x, y)`
- `maximize(winid, state)` - state: 1=horizontal, 2=vertical, 3=both
- `minimize(winid)`
- `unmaximize(winid)`
- `close(winid, isForced)`
- `setFullscreen(winid, state)`
- `toggleFullscreen(winid)`
- `setOnAllWorkspaces(winid, state)`
- `setAbove(winid, state)`
