/*
    Copyright 2025 Roman Lefler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
import Gio from "gi://Gio";
import { str, i64, i32 } from "./gvariant.js";
let bus = null;
let mediaChangedCallback = null;
let proxies = {};
let subs = [];
export function setBusSession(dbusSession) {
    bus = dbusSession;
    proxies = {};
    subs = [];
}
export async function mediaLaunched(started, exited, changed) {
    if (!bus)
        throw new Error("Set bus session first.");
    mediaChangedCallback = changed;
    const id = bus.signal_subscribe("org.freedesktop.DBus", "org.freedesktop.DBus", "NameOwnerChanged", null, null, Gio.DBusSignalFlags.NONE, async (_conn, _sender, _objectPath, _interfaceName, _signalName, params) => {
        let nameV, oldOwnerV, newOwnerV;
        [nameV, oldOwnerV, newOwnerV] = params.unpack();
        const name = nameV.get_string()[0];
        const oldOwner = oldOwnerV.get_string()[0];
        const newOwner = newOwnerV.get_string()[0];
        if (!name.startsWith("org.mpris.MediaPlayer2."))
            return;
        if (!oldOwner && newOwner) {
            proxies[name] = await createProxy(name);
            mediaChanged(proxies[name], () => changed(name));
            started(name);
        }
        else if (oldOwner && !newOwner) {
            delete proxies[name];
            exited(name);
        }
    });
    subs.push(id);
    // Add change handlers and add proxies for existing players
    getMediaPlayers();
}
export async function getMediaPlayers() {
    return new Promise((resolve, reject) => {
        if (!bus)
            throw new Error("Set bus session first.");
        bus.call("org.freedesktop.DBus", "/org/freedesktop/DBus", "org.freedesktop.DBus", "ListNames", null, null, Gio.DBusCallFlags.NONE, -1, null, async (_conn, result) => {
            try {
                if (!bus)
                    return reject(new Error("Bus set to NULL before response."));
                const namesV = bus.call_finish(result);
                const names = namesV.deep_unpack()[0];
                const players = names.filter(s => s.startsWith("org.mpris.MediaPlayer2."));
                // If there were players before the extension started they wont be in the proxy list yet
                for (let p of players) {
                    if (!(p in proxies)) {
                        proxies[p] = await createProxy(p);
                        mediaChanged(proxies[p], () => mediaChangedCallback?.(p));
                    }
                }
                resolve(players);
            }
            catch (e) {
                console.error(`Dropbeat: Failed to list media players: ${e}`);
                resolve([]);
            }
        });
    });
}
async function createProxy(name) {
    return new Promise((resolve, reject) => {
        if (!bus)
            return reject(new Error("Set bus session first."));
        Gio.DBusProxy.new(bus, Gio.DBusProxyFlags.NONE, null, name, "/org/mpris/MediaPlayer2", "org.mpris.MediaPlayer2.Player", null, (_, result) => {
            const proxy = Gio.DBusProxy.new_finish(result);
            resolve(proxy);
        });
    });
}
function mediaChanged(proxy, callback) {
    proxy.connect("g-properties-changed", (_, props) => {
        const changed = props.deep_unpack();
        if ("Metadata" in changed || "PlaybackStatus" in changed)
            callback();
    });
}
export function mediaQueryPlayer(name) {
    try {
        const proxy = proxies[name];
        if (!proxy)
            throw new Error(`No proxy for media player ${name}`);
        const statusV = proxy.get_cached_property("PlaybackStatus");
        const status = statusV?.deep_unpack() ?? null;
        if (status !== "Playing" && status !== "Paused" && status !== "Stopped" && status !== null) {
            throw new Error(`Unknown playback status "${status}" for media player ${name}`);
        }
        const metaV = proxy.get_cached_property("Metadata");
        if (!metaV)
            throw new Error(`No metadata for media player ${name}`);
        const meta = {};
        const len = metaV.n_children();
        if (!len)
            return null;
        for (let i = 0; i < len; i++) {
            const item = metaV.get_child_value(i);
            const key = str(item.get_child_value(0));
            const value = item.get_child_value(1);
            meta[key] = value.get_variant();
        }
        const date = str(meta["xesam:contentCreated"]);
        const sec = i64(meta["mpris:length"]);
        return {
            title: str(meta["xesam:title"]),
            artists: meta["xesam:artist"]?.deep_unpack() ?? null,
            album: str(meta["xesam:album"]),
            trackN: i32(meta["xesam:trackNumber"]),
            discN: i32(meta["xesam:discNumber"]),
            genres: meta["xesam:genre"]?.deep_unpack() ?? null,
            release: date ? new Date(date) : null,
            artUrl: str(meta["mpris:artUrl"]),
            seconds: sec ? sec / 1000000 : null,
            status: status
        };
    }
    catch (e) {
        console.error(`Dropbeat: Failed to query media player ${name}: ${e}`);
        return null;
    }
}
async function mediaCallMethod(name, method) {
    const proxy = proxies[name];
    if (!proxy)
        throw new Error(`No proxy for media player ${name}.`);
    return new Promise((resolve, reject) => {
        proxy.call(method, null, Gio.DBusCallFlags.NONE, -1, null, (p, result) => {
            if (!p)
                return reject("Media player was NULL.");
            try {
                p.call_finish(result);
            }
            catch (e) {
                return reject(new Error(`Toggle pause failed on player ${name}: ${e}`));
            }
            resolve();
        });
    });
}
;
export async function mediaTogglePause(name) {
    return mediaCallMethod(name, "PlayPause");
}
export async function mediaPrev(name) {
    return mediaCallMethod(name, "Previous");
}
export async function mediaNext(name) {
    return mediaCallMethod(name, "Next");
}
export function mediaFree() {
    let id;
    if (bus) {
        while ((id = subs.pop()) !== undefined)
            bus.signal_unsubscribe(id);
    }
    proxies = {};
    subs = [];
    bus = null;
}
