/*
Copyright (C) 2014  spin83

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, visit https://www.gnu.org/licenses/.
*/

import St from 'gi://St';
import Atk from 'gi://Atk';
import Clutter from 'gi://Clutter';
import GObject from 'gi://GObject';
import GLib from 'gi://GLib';

import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';

// Lightweight mirrored indicator that visually clones an existing indicator
// (e.g., Vitals) from the main panel and opens its menu anchored to this button.
export const MirroredIndicatorButton = GObject.registerClass(
    class MirroredIndicatorButton extends PanelMenu.Button {
        _init(panel, role) {
            super._init(0.0, null, false);

            this._role = role;
            this._panel = panel;

            if (role === 'activities') {
                this._initActivitiesButton();
            } else {
                this._initGenericIndicator(role);
            }
        }

        _initActivitiesButton() {
            // Create the activities indicator with workspace dots like main panel
            this.accessible_role = Atk.Role.TOGGLE_BUTTON;
            this.name = 'mmPanelActivities';
            this.add_style_class_name('panel-button');
            this.add_style_class_name('mm-activities');

            // Set up for full height hover
            this.y_expand = true;
            this.y_align = Clutter.ActorAlign.FILL;

            // Container for workspace dots - centered vertically
            this._workspaceDotsBox = new St.BoxLayout({
                style_class: 'workspace-dots',
                y_align: Clutter.ActorAlign.CENTER,
                x_align: Clutter.ActorAlign.CENTER,
                y_expand: true,
            });

            this.add_child(this._workspaceDotsBox);
            this.label_actor = this._workspaceDotsBox;

            // Store workspace manager reference first
            this._workspaceManager = global.workspace_manager;

            // Build initial workspace dots
            this._updateWorkspaceDots();

            // Connect to workspace changes
            this._activeWsChangedId = this._workspaceManager.connect('active-workspace-changed',
                this._updateWorkspaceDots.bind(this));
            this._nWorkspacesChangedId = this._workspaceManager.connect('notify::n-workspaces',
                this._updateWorkspaceDots.bind(this));

            // Sync with overview state
            this._showingId = Main.overview.connect('showing', () => {
                this.add_style_pseudo_class('overview');
                this.add_accessible_state(Atk.StateType.CHECKED);
            });

            this._hidingId = Main.overview.connect('hiding', () => {
                this.remove_style_pseudo_class('overview');
                this.remove_accessible_state(Atk.StateType.CHECKED);
            });

            this._sourceIndicator = null;
        }

        _updateWorkspaceDots() {
            if (!this._workspaceDotsBox || !this._workspaceManager)
                return;

            // Remove existing dots
            this._workspaceDotsBox.remove_all_children();

            const nWorkspaces = this._workspaceManager.n_workspaces;
            const activeIndex = this._workspaceManager.get_active_workspace_index();

            for (let i = 0; i < nWorkspaces; i++) {
                const isActive = (i === activeIndex);
                const dot = new St.Widget({
                    style_class: isActive ? 'workspace-dot active' : 'workspace-dot',
                    width: isActive ? 34 : 7,
                    height: isActive ? 8 : 7,
                    style: `border-radius: 6px; background-color: rgba(255, 255, 255, ${isActive ? '1' : '0.5'}); margin: 0 2px;`,
                    y_align: Clutter.ActorAlign.CENTER,
                });
                this._workspaceDotsBox.add_child(dot);
            }
        }

        _initGenericIndicator(role) {
            this._sourceIndicator = Main.panel.statusArea[role] || null;

            if (this._sourceIndicator) {
                this._createIndicatorClone();
            } else {
                this._createFallbackIcon();
            }
        }

        _createIndicatorClone() {
            try {
                const sourceChild = this._sourceIndicator.get_first_child();
                if (sourceChild && sourceChild instanceof St.BoxLayout) {
                    // For dateMenu, create a real label (not a clone) for independent hover
                    if (this._role === 'dateMenu' && this._sourceIndicator._clockDisplay) {
                        // Create clock label directly - no extra container
                        const clockDisplay = new St.Label({
                            style_class: 'clock',
                            y_align: Clutter.ActorAlign.CENTER,
                            y_expand: true,
                        });

                        const updateClock = () => {
                            if (this._sourceIndicator._clockDisplay) {
                                clockDisplay.text = this._sourceIndicator._clockDisplay.text;
                            }
                        };

                        updateClock();

                        // Remove existing timeout before creating new one
                        if (this._clockUpdateId) {
                            GLib.source_remove(this._clockUpdateId);
                            this._clockUpdateId = null;
                        }

                        this._clockUpdateId = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, 1, () => {
                            updateClock();
                            return GLib.SOURCE_CONTINUE;
                        });

                        this.add_child(clockDisplay);
                        this._clockDisplay = clockDisplay;
                    } else {
                        // For quickSettings, use FILL to match panel height
                        if (this._role === 'quickSettings') {
                            this.add_style_class_name('mm-quick-settings');
                            this.y_expand = true;
                            this.y_align = Clutter.ActorAlign.FILL;
                            const container = new St.BoxLayout({
                                style_class: 'mm-quick-settings-box',
                                y_align: Clutter.ActorAlign.FILL,
                                y_expand: true,
                            });
                            this._createQuickSettingsClone(container, sourceChild);
                            this.add_child(container);
                        } else if (this._role === 'favorites-menu' || this._role.toLowerCase().includes('favorites') || this._role.toLowerCase().includes('favorite')) {
                            // For favorites-menu@fthx extension (registers as 'Favorites Menu Button'), use real widget copy
                            this.add_style_class_name('mm-favorites-menu');
                            this.y_expand = true;
                            this.y_align = Clutter.ActorAlign.FILL;
                            const container = new St.BoxLayout({
                                style_class: 'mm-favorites-menu-box',
                                y_align: Clutter.ActorAlign.FILL,
                                y_expand: true,
                            });
                            this._createFillClone(container, sourceChild);
                            this.add_child(container);
                        } else {
                            // For other indicators, use clone approach
                            // Container is FILL to get full-height hover, but clone inside is centered
                            const container = new St.BoxLayout({
                                style_class: sourceChild.get_style_class_name() || 'panel-status-menu-box',
                                y_align: Clutter.ActorAlign.FILL,
                                y_expand: true,
                            });
                            this._createSimpleClone(container, sourceChild);
                            this.add_child(container);
                        }
                    }
                } else {
                    this._createSimpleClone(this, sourceChild);
                }
            } catch (e) {
                console.error('[Multi Monitors Add-On] Failed to create mirrored indicator:', String(e));
                this._createFallbackIcon();
            }
        }

        _createClockDisplay(container) {
            const clockDisplay = new St.Label({
                style_class: 'clock',
                y_align: Clutter.ActorAlign.CENTER,
            });

            const updateClock = () => {
                if (this._sourceIndicator._clockDisplay) {
                    clockDisplay.text = this._sourceIndicator._clockDisplay.text;
                }
            };

            updateClock();

            // Remove existing timeout before creating new one
            if (this._clockUpdateId) {
                GLib.source_remove(this._clockUpdateId);
                this._clockUpdateId = null;
            }

            this._clockUpdateId = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, 1, () => {
                updateClock();
                return GLib.SOURCE_CONTINUE;
            });

            container.add_child(clockDisplay);
            this._clockDisplay = clockDisplay;
        }

        _createSimpleClone(parent, source) {
            // Check if this is a problematic extension that needs static icon copies
            // (Tiling Shell and similar extensions that resize during fullscreen)
            const problematicExtensions = ['tiling', 'tilingshell', 'forge', 'pop-shell'];
            const isProblematic = problematicExtensions.some(name =>
                this._role && this._role.toLowerCase().includes(name)
            );

            if (isProblematic) {
                // Use static icon copies for problematic extensions
                this._createStaticIconCopy(parent, source);
                return;
            }

            // For regular indicators, use Clutter.Clone (works fine)
            const clone = new Clutter.Clone({
                source: source,
                y_align: Clutter.ActorAlign.CENTER,
                y_expand: false,
            });

            parent.add_child(clone);
        }

        _createQuickSettingsClone(parent, source) {
            // For quick settings (system tray), use FILL to match panel height
            const clone = new Clutter.Clone({
                source: source,
                y_align: Clutter.ActorAlign.FILL,
                y_expand: true,
            });

            parent.add_child(clone);
            this._quickSettingsClone = clone;
            this._quickSettingsSource = source;

            // Lock the clone's size after it's allocated to prevent collapse during fullscreen
            this._lockQuickSettingsSize();
        }

        _lockQuickSettingsSize() {
            // Wait for the clone to be properly sized, then lock it with min_width/min_height
            if (this._lockSizeTimeoutId) {
                GLib.source_remove(this._lockSizeTimeoutId);
                this._lockSizeTimeoutId = null;
            }

            this._lockSizeTimeoutId = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 100, () => {
                if (this._quickSettingsClone && this._quickSettingsSource) {
                    // Get the source's current size (when main panel is visible)
                    const [sourceWidth, sourceHeight] = this._quickSettingsSource.get_size();

                    if (sourceWidth > 0 && sourceHeight > 0) {
                        // Lock the clone's minimum size to prevent collapse
                        this._quickSettingsClone.set_size(sourceWidth, sourceHeight);
                        this._quickSettingsClone.min_width = sourceWidth;
                        this._quickSettingsClone.min_height = sourceHeight;
                        this._cachedWidth = sourceWidth;
                        this._cachedHeight = sourceHeight;
                    }
                }
                this._lockSizeTimeoutId = null;
                return GLib.SOURCE_REMOVE;
            });

            // Also listen for source size changes to update when not in fullscreen
            if (!this._sourceSizeChangedId && this._quickSettingsSource) {
                this._sourceSizeChangedId = this._quickSettingsSource.connect(
                    'notify::width', () => this._onSourceSizeChanged()
                );
            }
        }

        _onSourceSizeChanged() {
            if (!this._quickSettingsSource || !this._quickSettingsClone) {
                return;
            }

            const [sourceWidth, sourceHeight] = this._quickSettingsSource.get_size();

            // Ignore invalid sizes
            if (sourceWidth <= 0 || sourceHeight <= 0) {
                return;
            }

            // During fullscreen on primary monitor, only allow GROWING (for new icons)
            // Block SHRINKING to prevent collapse when main panel hides
            if (this._isPrimaryMonitorFullscreen()) {
                const currentWidth = this._cachedWidth || 0;
                // Only update if the new size is BIGGER (new icon added)
                if (sourceWidth > currentWidth) {
                    this._quickSettingsClone.set_size(sourceWidth, sourceHeight);
                    this._quickSettingsClone.min_width = sourceWidth;
                    this._quickSettingsClone.min_height = sourceHeight;
                    this._cachedWidth = sourceWidth;
                    this._cachedHeight = sourceHeight;
                }
                this._wasFullscreen = true;
                return;
            }

            // Normal operation: update size if source is visible and not in overview
            if (this._quickSettingsSource.visible && !Main.overview.visible) {
                // If we just exited fullscreen, debounce the size update
                // to avoid temporary large sizes during panel animation
                if (this._wasFullscreen) {
                    this._wasFullscreen = false;
                    this._debounceSizeUpdate(sourceWidth, sourceHeight);
                    return;
                }

                this._applySize(sourceWidth, sourceHeight);
            }
        }

        _debounceSizeUpdate(width, height) {
            // Clear any pending debounce
            if (this._sizeDebounceId) {
                GLib.source_remove(this._sizeDebounceId);
                this._sizeDebounceId = null;
            }

            // Wait for panel to fully settle after exiting fullscreen
            this._sizeDebounceId = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 300, () => {
                // Re-check the actual source size after settling
                if (this._quickSettingsSource && this._quickSettingsClone) {
                    const [finalWidth, finalHeight] = this._quickSettingsSource.get_size();
                    if (finalWidth > 0 && finalHeight > 0) {
                        this._applySize(finalWidth, finalHeight);
                    }
                }
                this._sizeDebounceId = null;
                return GLib.SOURCE_REMOVE;
            });
        }

        _applySize(width, height) {
            this._quickSettingsClone.set_size(width, height);
            this._quickSettingsClone.min_width = width;
            this._quickSettingsClone.min_height = height;
            this._cachedWidth = width;
            this._cachedHeight = height;
        }

        _isPrimaryMonitorFullscreen() {
            // Check if any window is fullscreen on the primary monitor
            const primaryIndex = Main.layoutManager.primaryIndex;
            const windows = global.get_window_actors();

            for (const actor of windows) {
                const metaWindow = actor.get_meta_window();
                if (metaWindow &&
                    metaWindow.is_fullscreen() &&
                    metaWindow.get_monitor() === primaryIndex) {
                    return true;
                }
            }
            return false;
        }

        _createStaticIconCopy(parent, source) {
            // Create static icon copies for problematic extensions (Tiling Shell, etc.)
            // These are immune to source changes during fullscreen
            const container = new St.BoxLayout({
                style_class: 'panel-status-menu-box',
                x_align: Clutter.ActorAlign.CENTER,
                y_align: Clutter.ActorAlign.CENTER,
                y_expand: false,
                reactive: false,
            });

            // Copy all icons from the source
            this._copyIconsFromSource(container, source);
            parent.add_child(container);
            this._iconContainer = container;
            this._iconSource = source;

            // Periodically sync icons (every 5 seconds) to catch icon changes
            this._startIconSync();
        }

        _copyIconsFromSource(container, source) {
            // Remove existing children
            container.remove_all_children();

            // Find all icons in the source and create copies
            const icons = this._findAllIconsInActor(source);

            if (icons.length > 0) {
                for (const icon of icons) {
                    const iconCopy = new St.Icon({
                        gicon: icon.gicon,
                        icon_name: icon.icon_name,
                        icon_size: icon.icon_size || 16,
                        style_class: icon.get_style_class_name() || 'system-status-icon',
                        y_align: Clutter.ActorAlign.CENTER,
                    });
                    container.add_child(iconCopy);
                }
            } else {
                // Fallback: use a clone but wrap it to prevent resize
                const clone = new Clutter.Clone({
                    source: source,
                    y_align: Clutter.ActorAlign.CENTER,
                });
                container.add_child(clone);
            }
        }

        _findAllIconsInActor(actor) {
            // Recursively find all St.Icon instances in an actor tree
            const icons = [];
            if (actor instanceof St.Icon) {
                icons.push(actor);
            }
            const children = actor.get_children ? actor.get_children() : [];
            for (const child of children) {
                icons.push(...this._findAllIconsInActor(child));
            }
            return icons;
        }

        _startIconSync() {
            if (this._iconSyncId) {
                GLib.source_remove(this._iconSyncId);
                this._iconSyncId = null;
            }

            this._iconSyncId = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, 5, () => {
                if (this._iconContainer && this._iconSource) {
                    this._copyIconsFromSource(this._iconContainer, this._iconSource);
                }
                return GLib.SOURCE_CONTINUE;
            });
        }

        _createFillClone(parent, source) {
            // For favorites-menu@fthx - create real widget copy instead of Clutter.Clone
            // This prevents visual glitches when the main panel hides during fullscreen
            const container = new St.BoxLayout({
                style_class: source.get_style_class_name ? source.get_style_class_name() : 'panel-status-menu-box',
                x_align: Clutter.ActorAlign.CENTER,
                y_align: Clutter.ActorAlign.CENTER,
                y_expand: true,
                reactive: false,
            });

            // Find and copy the icon from the source (favorites-menu uses starred-symbolic icon)
            const icon = this._findIconInActor(source);
            if (icon) {
                const iconCopy = new St.Icon({
                    gicon: icon.gicon,
                    icon_name: icon.icon_name || 'starred-symbolic',
                    icon_size: icon.icon_size || 16,
                    style_class: icon.get_style_class_name() || 'system-status-icon',
                    y_align: Clutter.ActorAlign.CENTER,
                });
                container.add_child(iconCopy);
            } else {
                // Fallback: create the starred icon directly
                const fallbackIcon = new St.Icon({
                    icon_name: 'starred-symbolic',
                    style_class: 'system-status-icon',
                    y_align: Clutter.ActorAlign.CENTER,
                });
                container.add_child(fallbackIcon);
            }

            parent.add_child(container);
            this._favoritesContainer = container;
        }

        _findIconInActor(actor) {
            // Recursively find St.Icon in an actor tree
            if (actor instanceof St.Icon) {
                return actor;
            }
            const children = actor.get_children ? actor.get_children() : [];
            for (const child of children) {
                const found = this._findIconInActor(child);
                if (found) return found;
            }
            return null;
        }

        _createFallbackIcon() {
            const label = new St.Label({
                text: '⚙',
                y_align: Clutter.ActorAlign.CENTER
            });
            this.add_child(label);
        }

        vfunc_button_press_event(buttonEvent) {
            this._onButtonPress();
            return Clutter.EVENT_STOP;
        }

        vfunc_event(event) {
            if (event.type() === Clutter.EventType.BUTTON_PRESS) {
                return this.vfunc_button_press_event(event);
            }
            return super.vfunc_event(event);
        }

        _onButtonPress() {
            if (this._role === 'activities') {
                Main.overview.toggle();
                return Clutter.EVENT_STOP;
            }

            // Check for standard menu first
            if (this._sourceIndicator && this._sourceIndicator.menu) {
                return this._openMirroredMenu();
            }

            // Handle extensions with custom popup menus (like favorite-apps-menu@venovar)
            // These extensions have _popupFavoriteAppsMenu or similar custom menus
            if (this._sourceIndicator) {
                // Try to find and open custom popup menus
                const customMenus = [
                    '_popupFavoriteAppsMenu',
                    '_popupPowerItemsMenu',
                    '_popup',
                    '_popupMenu'
                ];

                for (const menuName of customMenus) {
                    if (this._sourceIndicator[menuName] && typeof this._sourceIndicator[menuName].toggle === 'function') {
                        return this._openCustomPopupMenu(this._sourceIndicator[menuName]);
                    }
                }

                // If no menu found, try to emit a button press on the source indicator
                // This allows the source indicator to handle the click itself
                if (typeof this._sourceIndicator.vfunc_button_press_event === 'function' ||
                    typeof this._sourceIndicator.emit === 'function') {
                    return this._forwardClickToSource();
                }
            }

            return Clutter.EVENT_PROPAGATE;
        }

        _forwardClickToSource() {
            // Forward the click to the source indicator
            // This makes the source indicator handle the click as if it was clicked directly
            this.add_style_pseudo_class('active');

            // Emit button-press-event on the source
            const event = Clutter.get_current_event();
            if (event && this._sourceIndicator.emit) {
                this._sourceIndicator.emit('button-press-event', event);
            }

            // Also try button-release-event which some extensions use
            if (event && this._sourceIndicator.emit) {
                // Clean up any existing timeout before creating a new one
                if (this._forwardClickTimeoutId) {
                    GLib.source_remove(this._forwardClickTimeoutId);
                    this._forwardClickTimeoutId = null;
                }
                this._forwardClickTimeoutId = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 50, () => {
                    this._sourceIndicator.emit('button-release-event', event);
                    this.remove_style_pseudo_class('active');
                    this._forwardClickTimeoutId = null;
                    return GLib.SOURCE_REMOVE;
                });
            }

            return Clutter.EVENT_STOP;
        }

        _openCustomPopupMenu(popupMenu) {
            const monitorIndex = Main.layoutManager.findIndexForActor(this);
            const originalSourceActor = popupMenu.sourceActor;

            try {
                // Close the menu if it's already open
                if (popupMenu.isOpen) {
                    popupMenu.close();
                    return Clutter.EVENT_STOP;
                }

                // Add active style to this button
                this.add_style_pseudo_class('active');

                // Update popup's sourceActor to position correctly
                popupMenu.sourceActor = this;

                // Update positioning for the correct monitor
                if (popupMenu.box) {
                    const monitor = Main.layoutManager.monitors[monitorIndex];
                    if (monitor && popupMenu.box._updateFlip) {
                        popupMenu.box._updateFlip(monitor);
                    }
                }

                // Setup cleanup on menu close
                const openStateId = popupMenu.connect('open-state-changed', (_m, isOpen) => {
                    if (isOpen) {
                        this.add_style_pseudo_class('active');
                    } else {
                        this.remove_style_pseudo_class('active');
                        popupMenu.sourceActor = originalSourceActor;
                        popupMenu.disconnect(openStateId);
                    }
                });

                popupMenu.open();
            } catch (e) {
                console.error('[Multi Monitors Add-On] Failed to open custom popup menu:', String(e));

                // Cleanup on error
                this.remove_style_pseudo_class('active');
                if (popupMenu) {
                    popupMenu.sourceActor = originalSourceActor;
                }
            }

            return Clutter.EVENT_STOP;
        }

        _openMirroredMenu() {
            const monitorIndex = Main.layoutManager.findIndexForActor(this);
            const menu = this._sourceIndicator.menu;

            // Store original state variables
            let originalSourceActor = menu.sourceActor;
            let originalBoxPointer = menu.box?._sourceActor;
            let originalSetActive = this._sourceIndicator.setActive?.bind(this._sourceIndicator);
            let originalAddPseudoClass = this._sourceIndicator.add_style_pseudo_class?.bind(this._sourceIndicator);

            // State for restoring menu box modifications
            let menuBoxState = null;

            let openStateId = 0;

            try {
                if (menu.isOpen) {
                    menu.close();
                    return Clutter.EVENT_STOP;
                }

                // Prevent active state on main panel indicator
                this._preventMainPanelActiveState(originalAddPseudoClass);

                // Add active style to THIS button
                this.add_style_pseudo_class('active');

                // Update menu's sourceActor
                menu.sourceActor = this;

                // Update BoxPointer positioning and save state for restoration
                if (menu.box) {
                    menuBoxState = this._updateMenuPositioning(menu, monitorIndex);
                }

                // Setup cleanup on menu close
                openStateId = menu.connect('open-state-changed', (m, isOpen) => {
                    if (isOpen) {
                        this.add_style_pseudo_class('active');
                    } else {
                        this._restoreMenuState(menu, originalSourceActor, originalBoxPointer, originalSetActive, originalAddPseudoClass, menuBoxState);
                        menu.disconnect(openStateId);
                    }
                });

                menu.open();
            } catch (e) {
                console.error('[Multi Monitors Add-On] Failed to open mirrored menu:', String(e));

                // Immediate cleanup on error
                this._restoreMenuState(menu, originalSourceActor, originalBoxPointer, originalSetActive, originalAddPseudoClass, menuBoxState);

                if (openStateId > 0) {
                    menu.disconnect(openStateId);
                }

                this.remove_style_pseudo_class('active');
            }

            return Clutter.EVENT_STOP;
        }

        _preventMainPanelActiveState(originalAddPseudoClass) {
            if (this._sourceIndicator.setActive) {
                this._sourceIndicator.setActive = () => { };
            }

            if (this._sourceIndicator.add_style_pseudo_class) {
                const originalMethod = this._sourceIndicator.add_style_pseudo_class.bind(this._sourceIndicator);
                this._sourceIndicator.add_style_pseudo_class = (pseudoClass) => {
                    if (pseudoClass !== 'active' && pseudoClass !== 'checked') {
                        originalMethod(pseudoClass);
                    }
                };
            }

            if (this._sourceIndicator.remove_style_pseudo_class) {
                this._sourceIndicator.remove_style_pseudo_class('active');
                this._sourceIndicator.remove_style_pseudo_class('checked');
            }
        }

        _updateMenuPositioning(menu, monitorIndex) {
            const menuBox = menu.box;

            // 1. Save original source actor
            menuBox._sourceActor = this;
            menuBox._sourceAllocation = null;

            // 2. Handle constraints
            const removedConstraints = [];
            const constraints = menuBox.get_constraints();
            for (let constraint of constraints) {
                if (constraint.constructor.name === 'BindConstraint' ||
                    constraint.constructor.name === 'AlignConstraint') {
                    menuBox.remove_constraint(constraint);
                    removedConstraints.push(constraint);
                }
            }

            // 3. Handle setPosition override - FULL MANUAL REPLACEMENT
            // We do NOT call oldSetPosition because it likely crashes/fails on extended monitors
            const originalSetPosition = menuBox.setPosition;

            const monitor = Main.layoutManager.monitors[monitorIndex] || Main.layoutManager.primaryMonitor;

            menuBox.setPosition = function (sourceActor, alignment) {
                // Calculate position manually
                const [btnX, btnY] = sourceActor.get_transformed_position();
                const [btnW, btnH] = sourceActor.get_transformed_size();
                const [menuW, menuH] = this.get_preferred_size(); // Get preferred size (min, nat)
                const finalMenuW = menuW[1]; // Use natural width
                // Height might be dynamic, use current size or preferred?
                // BoxPointer usually has size by now.
                const [currW, currH] = this.get_size();
                const finalMenuH = currH > 0 ? currH : menuH[1];

                // Center horizontally on the button
                let newX = btnX + (btnW / 2) - (finalMenuW / 2);
                let newY = btnY + btnH; // Below the button

                // Constraint to monitor bounds
                if (newX + finalMenuW > monitor.x + monitor.width) {
                    newX = monitor.x + monitor.width - finalMenuW;
                }
                if (newX < monitor.x) {
                    newX = monitor.x;
                }

                // Vertical constraint (flip if needed, though usually bar is top)
                if (newY + finalMenuH > monitor.y + monitor.height) {
                    newY = btnY - finalMenuH;
                    if (this.setArrowSide) this.setArrowSide(St.Side.BOTTOM);
                } else {
                    if (this.setArrowSide) this.setArrowSide(St.Side.TOP);
                }

                this.set_position(Math.round(newX), Math.round(newY));
            };

            return {
                originalSetPosition: originalSetPosition,
                removedConstraints: removedConstraints
            };
        }

        _restoreMenuState(menu, originalSourceActor, originalBoxPointer, originalSetActive, originalAddPseudoClass, menuBoxState) {
            try {
                // 1. Restore standard menu properties
                if (originalSourceActor) {
                    menu.sourceActor = originalSourceActor;
                }

                if (menu.box && originalBoxPointer) {
                    menu.box._sourceActor = originalBoxPointer;
                }

                // 2. Restore hijacked indicator methods
                if (originalSetActive && this._sourceIndicator) {
                    this._sourceIndicator.setActive = originalSetActive;
                }

                if (originalAddPseudoClass && this._sourceIndicator) {
                    this._sourceIndicator.add_style_pseudo_class = originalAddPseudoClass;
                }

                // 3. Restore menu box modifications (setPosition and constraints)
                if (menu.box && menuBoxState) {
                    if (menuBoxState.originalSetPosition) {
                        menu.box.setPosition = menuBoxState.originalSetPosition;
                    }

                    if (menuBoxState.removedConstraints && menuBoxState.removedConstraints.length > 0) {
                        menuBoxState.removedConstraints.forEach(constraint => {
                            menu.box.add_constraint(constraint);
                        });
                    }
                }

                // 4. Reset style classes on source
                if (this._sourceIndicator && this._sourceIndicator.remove_style_pseudo_class) {
                    this._sourceIndicator.remove_style_pseudo_class('active');
                    this._sourceIndicator.remove_style_pseudo_class('checked');
                }
            } catch (e) {
                console.error('[Multi Monitors Add-On] Error restoring menu state:', String(e));
            }

            // Always try to reset this button's state
            try {
                if (this.remove_style_pseudo_class) {
                    this.remove_style_pseudo_class('active');
                    this.remove_style_pseudo_class('checked');
                }
            } catch (e) { }
        }

        destroy() {
            if (this._clockUpdateId) {
                GLib.source_remove(this._clockUpdateId);
                this._clockUpdateId = null;
            }

            if (this._forwardClickTimeoutId) {
                GLib.source_remove(this._forwardClickTimeoutId);
                this._forwardClickTimeoutId = null;
            }

            if (this._iconSyncId) {
                GLib.source_remove(this._iconSyncId);
                this._iconSyncId = null;
            }

            if (this._lockSizeTimeoutId) {
                GLib.source_remove(this._lockSizeTimeoutId);
                this._lockSizeTimeoutId = null;
            }

            if (this._sourceSizeChangedId && this._quickSettingsSource) {
                this._quickSettingsSource.disconnect(this._sourceSizeChangedId);
                this._sourceSizeChangedId = null;
            }

            if (this._sizeDebounceId) {
                GLib.source_remove(this._sizeDebounceId);
                this._sizeDebounceId = null;
            }

            if (this._role === 'activities') {
                if (this._showingId) {
                    Main.overview.disconnect(this._showingId);
                    this._showingId = null;
                }
                if (this._hidingId) {
                    Main.overview.disconnect(this._hidingId);
                    this._hidingId = null;
                }
                if (this._activeWsChangedId) {
                    this._workspaceManager.disconnect(this._activeWsChangedId);
                    this._activeWsChangedId = null;
                }
                if (this._nWorkspacesChangedId) {
                    this._workspaceManager.disconnect(this._nWorkspacesChangedId);
                    this._nWorkspacesChangedId = null;
                }
            }
            super.destroy();
        }
    });
