import os
import glob
import shutil
import subprocess
import concurrent.futures
import json
import datetime
import re
from pathlib import Path

# --- Classes de Données ---
class AppInfo:
    def __init__(self, name, desktop_file, exec_cmd, origin=None, package_name=None):
        self.name = name if name else "Inconnu"
        self.desktop_file = desktop_file
        self.exec_cmd = exec_cmd
        self.origin = origin if origin else "unknown"
        self.package_name = package_name if package_name else ""

    def __repr__(self):
        return f"[{self.origin.upper()}] {self.name}"

    def to_dict(self):
        """Sérialise pour le JSON"""
        return {
            "name": self.name,
            "desktop_file": self.desktop_file,
            "origin": self.origin,
            "package_name": self.package_name,
            "timestamp": datetime.datetime.now().isoformat()
        }

# --- Gestionnaire d'Historique et Sauvegarde ---
class HistoryManager:
    def __init__(self):
        # Structure : 
        # ~/.local/share/UniversalUninstaller/
        #   ├── history.json
        #   ├── backups/ (fichiers .desktop)
        #   └── logs/    (sorties terminal)
        
        self.base_dir = os.path.join(os.path.expanduser("~"), ".local", "share", "UniversalUninstaller")
        self.history_file = os.path.join(self.base_dir, "history.json")
        self.backup_dir = os.path.join(self.base_dir, "backups")
        self.logs_dir = os.path.join(self.base_dir, "logs")
        
        for d in [self.backup_dir, self.logs_dir]:
            if not os.path.exists(d):
                os.makedirs(d)
            
        if not os.path.exists(self.history_file):
            with open(self.history_file, 'w') as f:
                json.dump([], f)

    def start_transaction(self, app):
        """Prépare l'entrée d'historique et sauvegarde le .desktop"""
        timestamp_str = datetime.datetime.now().strftime('%Y%m%d_%H%M%S')
        
        # 1. Backup du .desktop
        backup_path = None
        if os.path.exists(app.desktop_file):
            safe_name = os.path.basename(app.desktop_file)
            backup_path = os.path.join(self.backup_dir, f"{timestamp_str}_{safe_name}")
            try:
                shutil.copy2(app.desktop_file, backup_path)
            except Exception as e:
                print(f"Erreur backup: {e}")

        # 2. Préparation du chemin de log (vide pour l'instant)
        log_filename = f"{timestamp_str}_{app.name.replace(' ', '_')}.log"
        log_path = os.path.join(self.logs_dir, log_filename)

        # 3. Création de l'objet entrée
        entry = app.to_dict()
        entry["timestamp"] = datetime.datetime.now().isoformat()
        entry["backup_path"] = backup_path
        entry["log_path"] = log_path
        entry["exec_cmd"] = app.exec_cmd
        entry["status"] = "started" # On mettra à jour à "success" ou "error" après
        
        return entry

    def save_transaction(self, entry, output_log, success=True):
        """Sauvegarde finale avec les logs du terminal"""
        
        # 1. Écriture du contenu du terminal dans le fichier .log
        try:
            with open(entry["log_path"], "w", encoding="utf-8") as f:
                f.write(f"--- COMMANDE : {entry['exec_cmd']} ---\n")
                f.write(f"--- DATE : {datetime.datetime.now()} ---\n\n")
                f.write(output_log)
                f.write(f"\n\n--- CODE DE SORTIE : {'0 (Succès)' if success else 'Erreur'} ---")
        except Exception as e:
            print(f"Impossible d'écrire les logs : {e}")

        # 2. Mise à jour du statut
        entry["status"] = "success" if success else "error"

        # 3. Ajout au JSON global
        try:
            history = []
            if os.path.exists(self.history_file):
                with open(self.history_file, 'r') as f:
                    history = json.load(f)
            
            history.append(entry)
            
            with open(self.history_file, 'w') as f:
                json.dump(history, f, indent=4)
        except Exception as e:
            print(f"Erreur sauvegarde JSON: {e}")

# --- Moteur de détection (Votre version optimisée précédente) ---
class UniversalDetector:
    def __init__(self):
        self.desktop_paths = [
            "/usr/share/applications",
            "/var/lib/flatpak/exports/share/applications",
            "/var/lib/snapd/desktop/applications",
            os.path.expanduser("~/.local/share/applications"),
            os.path.expanduser("~/.local/share/flatpak/exports/share/applications")
        ]

    def _clean_exec_cmd(self, exec_cmd):
        if not exec_cmd: return None
        # On ne garde que le binaire pour la résolution de chemin
        # ex: "/opt/google/chrome/chrome --app-id=..." -> "/opt/google/chrome/chrome"
        parts = exec_cmd.split()
        if parts:
            return parts[0].strip('"').strip("'")
        return None

    def _parse_desktop_file(self, file_path):
        name = None
        exec_cmd = None
        no_display = False
        try:
            with open(file_path, 'r', errors='ignore') as f:
                for line in f:
                    if line.startswith("Name="):
                        if not name: name = line[5:].strip()
                    elif line.startswith("Exec="):
                        if not exec_cmd: exec_cmd = line[5:].strip()
                    elif line.startswith("NoDisplay=true"):
                        no_display = True
            if no_display or not exec_cmd or not name: return None
            return {"name": name, "path": file_path, "exec": exec_cmd}
        except: return None

    def _resolve_binary_paths_batch(self, apps_data):
        resolved = []
        for app in apps_data:
            clean_cmd = self._clean_exec_cmd(app["exec"])
            real_path = None
            if os.path.isabs(clean_cmd) and os.path.exists(clean_cmd): real_path = clean_cmd
            else: real_path = shutil.which(clean_cmd)
            
            if real_path:
                try: real_path = os.path.realpath(real_path)
                except: pass
                app["real_path"] = real_path
                resolved.append(app)
        return resolved

    def _identify_apt_batch(self, apt_candidates):
        if not apt_candidates: return {}
        path_to_app_map = {app["real_path"]: app for app in apt_candidates}
        paths = list(path_to_app_map.keys())
        results = {}
        chunk_size = 50
        for i in range(0, len(paths), chunk_size):
            chunk = paths[i:i + chunk_size]
            try:
                cmd = ["dpkg", "-S"] + chunk
                process = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.DEVNULL, text=True)
                stdout, _ = process.communicate()
                for line in stdout.splitlines():
                    if ": " in line:
                        pkg, fpath = line.split(": ", 1)
                        results[fpath.strip()] = pkg
            except: pass
        return results

    def scan_applications(self):
        # ... (Logique identique à la précédente version) ...
        raw_apps = []
        with concurrent.futures.ThreadPoolExecutor() as executor:
            futures = []
            for directory in self.desktop_paths:
                if os.path.exists(directory):
                    files = glob.glob(os.path.join(directory, "*.desktop"))
                    futures.extend([executor.submit(self._parse_desktop_file, f) for f in files])
            for future in concurrent.futures.as_completed(futures):
                res = future.result()
                if res: raw_apps.append(res)

        apps_with_paths = self._resolve_binary_paths_batch(raw_apps)
        final_apps_list = []
        apt_candidates = []

        for app in apps_with_paths:
            path = app["real_path"]
            desktop_path = app["path"]
            exec_line = app["exec"]
            filename = os.path.basename(desktop_path)

            # --- A. WEB APPS ---
            if "--app=" in exec_line or "--app-id=" in exec_line:
                 final_apps_list.append(AppInfo(app["name"], desktop_path, exec_line, "web-app", "Raccourci Web"))
                 continue

            # --- B. SNAP ---
            if "/var/lib/snapd/desktop/applications" in desktop_path:
                pkg_name = filename.split("_")[0] if "_" in filename else filename.replace(".desktop", "")
                final_apps_list.append(AppInfo(app["name"], desktop_path, exec_line, "snap", pkg_name))
                continue
            if "/snap/bin/" in path or "/snap/" in path:
                 parts = path.split(os.sep)
                 if "snap" in parts:
                     try:
                         idx = parts.index("snap")
                         potential_name = parts[idx + 1]
                         if potential_name not in ["bin", "usr", "core"]:
                             final_apps_list.append(AppInfo(app["name"], desktop_path, exec_line, "snap", potential_name))
                             continue
                     except: pass

            # --- C. FLATPAK ---
            if "/flatpak/" in path or "/flatpak/" in desktop_path:
                app_id = filename.replace(".desktop", "")
                final_apps_list.append(AppInfo(app["name"], desktop_path, exec_line, "flatpak", app_id))
                continue
            
            if path == "/usr/bin/snap": continue 
            apt_candidates.append(app)

        if apt_candidates:
            apt_results = self._identify_apt_batch(apt_candidates)
            for app in apt_candidates:
                path = app["real_path"]
                if path in apt_results:
                    pkg_name = apt_results[path]
                    if pkg_name == "snapd": continue 
                    final_apps_list.append(AppInfo(app["name"], app["path"], app["exec"], "apt", pkg_name))
                else:
                    final_apps_list.append(AppInfo(app["name"], app["path"], app["exec"], "unknown", "Non identifié"))

        return final_apps_list

    def generate_uninstall_command(self, app):
        """Génère la commande intelligente selon le type"""
        
        if app.origin == "apt":
            return f"pkexec apt remove -y {app.package_name}"
            
        elif app.origin == "snap":
            return f"pkexec snap remove {app.package_name}"
            
        elif app.origin == "flatpak":
            return f"flatpak uninstall -y {app.package_name}"
            
        elif app.origin == "web-app":
            # --- NOUVELLE LOGIQUE POUR CHROME/EDGE/BRAVE ---
            # On veut passer de : 
            # Exec=/opt/google/chrome/chrome --profile-directory=Default --app-id=fmgjj...
            # À :
            # /opt/google/chrome/chrome --profile-directory=Default --uninstall-app-id=fmgjj...
            
            try:
                # 1. Trouver l'ID de l'app
                # Regex qui cherche --app-id=SUITE_DE_CARACTERES ou --app=...
                match_id = re.search(r'--app-id=([a-zA-Z0-9]+)', app.exec_cmd)
                
                # 2. Trouver le binaire du navigateur (la première partie de la commande)
                # On utilise shlex ou split simple. Attention aux guillemets.
                browser_bin = app.exec_cmd.split()[0].strip('"')
                
                if match_id:
                    app_id = match_id.group(1)
                    # On conserve le flag --profile-directory s'il existe, car l'app est installée dans un profil spécifique
                    profile_flag = ""
                    match_profile = re.search(r'--profile-directory=[\"\']?([\w\s]+)[\"\']?', app.exec_cmd)
                    if match_profile:
                        profile_flag = f"--profile-directory='{match_profile.group(1)}'"
                    
                    # Commande native de Chromium pour désinstaller proprement
                    # On ajoute aussi "rm" du fichier desktop en fin de chaîne au cas où le navigateur échoue ou laisse le fichier
                    browser_cmd = f"\"{browser_bin}\" {profile_flag} --uninstall-app-id={app_id}"
                    
                    # On combine : Commande Navigateur + Suppression manuelle du fichier .desktop pour être sûr
                    # Le ; permet de lancer la deuxième commande même si la première réussit
                    return f"{browser_cmd} ; rm \"{app.desktop_file}\""
                
                else:
                    # Fallback si on arrive pas à parser l'ID
                    return f"rm \"{app.desktop_file}\""
            except Exception as e:
                print(f"Erreur parsing web-app: {e}")
                return f"rm \"{app.desktop_file}\""
                
        else:
            # Sécurité pour les inconnus
            return f"rm \"{app.desktop_file}\""

if __name__ == "__main__":
    # Test simple du logger
    m = HistoryManager()
    print(f"Log file: {m.history_file}")