import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as AppDisplay from 'resource:///org/gnome/shell/ui/appDisplay.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';

export default class UninstallerExtension extends Extension {
    enable() {
        console.log("Désinstalleur : Activation sur l'UUID " + this.uuid);
        
        // Initialiser gettext
        this._gettext = this.gettext.bind(this);
        
        const self = this;
        
        // Surcharge de la méthode popupMenu pour ajouter l'élément de désinstallation
        this._originalPopupMenu = AppDisplay.AppIcon.prototype.popupMenu;
        
        AppDisplay.AppIcon.prototype.popupMenu = function(side) {
            // Appeler la méthode originale pour afficher le menu
            self._originalPopupMenu.call(this, side);
            
            let app = this.app;
            if (!app || !this._menu) return;
            
            // Vérifier si le bouton de désinstallation existe déjà
            let menuItems = this._menu._getMenuItems();
            let hasUninstallButton = menuItems.some(item => 
                item.label && item.label.get_text() === "Désinstaller"
            );
            
            if (hasUninstallButton) {
                return; // Ne pas ajouter de bouton en double
            }
            
            // Ajouter le séparateur et le sous-menu de désinstallation
            this._menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
            
            // Créer le sous-menu
            let uninstallSubMenu = new PopupMenu.PopupSubMenuMenuItem(self._gettext('Uninstall'), true);
            uninstallSubMenu.label.set_style('color: #ed333b; font-weight: bold;');
            
            let desktopId = app.get_id();
            let appName = app.get_name();
            
            // Obtenir la commande de désinstallation
            self._getUninstallCommand(desktopId, (command) => {
                if (!command) {
                    // Si pas de commande trouvée, afficher un message
                    let errorItem = new PopupMenu.PopupMenuItem(self._gettext('Unable to determine command'), { reactive: false });
                    errorItem.label.set_style('font-style: italic; color: #999;');
                    uninstallSubMenu.menu.addMenuItem(errorItem);
                    return;
                }
                
                // Ajouter le titre avec la commande
                let commandItem = new PopupMenu.PopupMenuItem(command, { reactive: false });
                commandItem.label.set_style('font-family: monospace; font-size: 10px; color: #ccc;');
                uninstallSubMenu.menu.addMenuItem(commandItem);
                
                uninstallSubMenu.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
                
                // Bouton "Confirmer la désinstallation"
                let confirmItem = new PopupMenu.PopupMenuItem(self._gettext('Confirm uninstallation'));
                confirmItem.label.set_style('color: #ed333b; font-weight: bold;');
                confirmItem.connect('activate', () => {
                    self._executeUninstall(command, appName);
                    uninstallSubMenu.menu.close();
                });
                uninstallSubMenu.menu.addMenuItem(confirmItem);
                
                // Bouton "Annuler"
                let cancelItem = new PopupMenu.PopupMenuItem(self._gettext('Cancel'));
                cancelItem.connect('activate', () => {
                    uninstallSubMenu.menu.close();
                });
                uninstallSubMenu.menu.addMenuItem(cancelItem);
            });
            
            this._menu.addMenuItem(uninstallSubMenu);
        };
    }

    _getUninstallCommand(desktopId, callback) {
        let scriptPath = GLib.build_filenamev([this.path, "get_uninstall_cmd.py"]);
        
        try {
            let proc = Gio.Subprocess.new(
                ['python3', scriptPath, desktopId],
                Gio.SubprocessFlags.STDOUT_PIPE | Gio.SubprocessFlags.STDERR_PIPE
            );
            
            proc.communicate_utf8_async(null, null, (proc, res) => {
                try {
                    let [, stdout, stderr] = proc.communicate_utf8_finish(res);
                    let command = stdout.trim();
                    callback(command || null);
                } catch (e) {
                    console.error('Erreur obtention commande:', e);
                    callback(null);
                }
            });
        } catch (e) {
            console.error('Erreur lancement script:', e);
            callback(null);
        }
    }
    
    _executeUninstall(command, appName) {
        Main.notify(`🗑️ ${this._gettext('Uninstallation in progress')}`, `${this._gettext('Uninstalling')} ${appName}...`);
        
        try {
            let proc = Gio.Subprocess.new(
                ['bash', '-c', command],
                Gio.SubprocessFlags.STDOUT_PIPE | Gio.SubprocessFlags.STDERR_PIPE
            );
            
            let self = this;
            proc.communicate_utf8_async(null, null, (proc, res) => {
                try {
                    let [, stdout, stderr] = proc.communicate_utf8_finish(res);
                    let success = proc.get_exit_status() === 0;
                    
                    if (success) {
                        Main.notify(`✅ ${self._gettext('Success')}`, `${appName} ${self._gettext('has been uninstalled')}`);
                    } else {
                        Main.notify(`❌ ${self._gettext('Error')}`, `${self._gettext('Failed to uninstall')} ${appName}`);
                        console.error('Erreur désinstallation:', stderr);
                    }
                } catch (e) {
                    Main.notify(`❌ ${self._gettext('Error')}`, self._gettext('Error during uninstallation'));
                    console.error('Erreur:', e);
                }
            });
        } catch (e) {
            Main.notify(`❌ ${this._gettext('Error')}`, this._gettext('Unable to execute command'));
            console.error('Erreur exécution:', e);
        }
    }

    disable() {
        console.log("Désinstalleur : Désactivation");
        
        // Restauration de la fonction originale
        if (this._originalPopupMenu) {
            AppDisplay.AppIcon.prototype.popupMenu = this._originalPopupMenu;
            this._originalPopupMenu = null;
        }
    }
}
