import sys
import subprocess
import gi
import threading

gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')

from gi.repository import Gtk, Adw, Gio, GLib
from backend_detector import UniversalDetector, HistoryManager

class MainWindow(Adw.ApplicationWindow):
    def __init__(self, app, target_desktop_id=None):
        super().__init__(application=app, title="Désinstalleur Universel")
        self.set_default_size(800, 600)

        # Initialisation du moteur de détection
        self.detector = UniversalDetector()
        self.history = HistoryManager()
        self.all_apps = []
        self.target_desktop_id = target_desktop_id

        # --- Interface ---
        self.overlay = Adw.ToastOverlay()
        self.set_content(self.overlay)

        content = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        self.overlay.set_child(content)

        header = Adw.HeaderBar()
        content.append(header)

        title = Adw.WindowTitle(title="Mes Applications", subtitle="Mode Extension GNOME")
        header.set_title_widget(title)

        refresh_btn = Gtk.Button(icon_name="view-refresh-symbolic")
        refresh_btn.connect("clicked", self.refresh_list)
        header.pack_end(refresh_btn)

        self.list_box = Gtk.ListBox()
        self.list_box.set_selection_mode(Gtk.SelectionMode.NONE)
        
        scrolled = Gtk.ScrolledWindow()
        scrolled.set_vexpand(True)
        scrolled.set_child(self.list_box)
        content.append(scrolled)

        # --- Chargement des données ---
        self.refresh_list(None)

        # --- LOGIQUE SPÉCIFIQUE EXTENSION ---
        # Si lancé avec --uninstall-id par l'extension
        if self.target_desktop_id:
            # On utilise idle_add pour attendre que l'UI soit prête
            GLib.idle_add(self.auto_trigger_uninstall, self.target_desktop_id)

    def auto_trigger_uninstall(self, desktop_id):
        """Cherche l'application et lance la désinstallation immédiatement"""
        target = next((a for a in self.all_apps if desktop_id in a.desktop_file), None)
        if target:
            print(f"Cible trouvée via extension : {target.name}")
            # S'assurer que la fenêtre est visible et au premier plan
            self.present()
            self.set_focus(None)
            # Forcer le focus sur cette fenêtre
            GLib.timeout_add(100, lambda: (self.present(), False))
            self.on_uninstall_clicked(None, target)
        else:
            print(f"Application {desktop_id} non trouvée dans le scan.")
            # Fermer la fenêtre si aucune app trouvée
            self.close()

    def refresh_list(self, btn):
        self.all_apps = self.detector.scan_applications()
        # Vider la liste
        while hasattr(self.list_box.get_first_child(), 'get_child'):
             self.list_box.remove(self.list_box.get_first_child())
        
        for app in self.all_apps:
            row = Adw.ActionRow(title=app.name, subtitle=f"Source: {app.origin}")
            btn_del = Gtk.Button(icon_name="user-trash-symbolic")
            btn_del.set_valign(Gtk.Align.CENTER)
            btn_del.connect("clicked", self.on_uninstall_clicked, app)
            row.add_suffix(btn_del)
            self.list_box.append(row)

    def on_uninstall_clicked(self, btn, app):
        # 1. Génération de la commande
        cmd = self.detector.generate_uninstall_command(app)
        
        # 2. Afficher la boîte de dialogue de confirmation
        self.show_confirmation_dialog(app, cmd)
    
    def show_confirmation_dialog(self, app, cmd):
        """Affiche une boîte de dialogue de confirmation avec la commande"""
        dialog = Adw.MessageDialog.new(self)
        dialog.set_heading(f"Désinstaller {app.name} ?")
        dialog.set_body(f"Cette action va exécuter la commande suivante :\n\n<tt>{GLib.markup_escape_text(cmd)}</tt>\n\nCette action est irréversible.")
        dialog.set_body_use_markup(True)
        dialog.set_modal(True)
        
        dialog.add_response("cancel", "Annuler")
        dialog.add_response("uninstall", "Désinstaller")
        dialog.set_response_appearance("uninstall", Adw.ResponseAppearance.DESTRUCTIVE)
        dialog.set_default_response("cancel")
        dialog.set_close_response("cancel")
        
        dialog.connect("response", self.on_dialog_response, app, cmd)
        dialog.present()
        
        print(f"Dialogue de confirmation affiché pour {app.name}")
    
    def on_dialog_response(self, dialog, response, app, cmd):
        """Callback quand l'utilisateur répond à la boîte de dialogue"""
        print(f"Réponse du dialogue : {response}")
        
        if response != "uninstall":
            print("Désinstallation annulée par l'utilisateur")
            # Fermer la fenêtre si lancé depuis l'extension
            if self.target_desktop_id:
                self.close()
            return
        
        print(f"Exécution confirmée : {cmd}")
        
        # Démarrage de la transaction d'historique
        entry = None
        try:
            entry = self.history.start_transaction(app)
        except Exception as e:
            print(f"Erreur historique: {e}")
        
        # Exécution de la commande
        try:
            result = subprocess.run(cmd, shell=True, capture_output=True, text=True, timeout=30)
            output_log = f"STDOUT:\n{result.stdout}\n\nSTDERR:\n{result.stderr}"
            success = result.returncode == 0
            
            # Sauvegarde finale avec les logs
            if entry:
                self.history.save_transaction(entry, output_log, success)
            
            if success:
                toast = Adw.Toast.new(f"Désinstallation de {app.name} réussie")
            else:
                toast = Adw.Toast.new(f"Erreur lors de la désinstallation de {app.name}")
            self.overlay.add_toast(toast)
            
            # Rafraîchir la liste après désinstallation
            GLib.timeout_add_seconds(2, self.refresh_list, None)
            
            # Fermer la fenêtre si lancé depuis l'extension
            if self.target_desktop_id:
                GLib.timeout_add_seconds(3, self.close)
        except Exception as e:
            print(f"Erreur d'exécution : {e}")
            if entry:
                self.history.save_transaction(entry, str(e), False)
            
            # Fermer la fenêtre même en cas d'erreur
            if self.target_desktop_id:
                GLib.timeout_add_seconds(2, self.close)

class UniversalUninstallerApp(Adw.Application):
    def __init__(self):
        super().__init__(application_id="com.tonnom.UniversalUninstaller",
                         flags=Gio.ApplicationFlags.HANDLES_COMMAND_LINE)
        self.target_desktop_id = None

    def do_command_line(self, command_line):
        """Gère les arguments de ligne de commande"""
        args = command_line.get_arguments()
        
        # Chercher --uninstall-id
        for i, arg in enumerate(args):
            if arg == "--uninstall-id" and i + 1 < len(args):
                self.target_desktop_id = args[i + 1]
                break
        
        self.activate()
        return 0

    def do_activate(self):
        win = self.props.active_window
        if not win:
            win = MainWindow(self, self.target_desktop_id)
        win.present()

if __name__ == "__main__":
    app = UniversalUninstallerApp()
    app.run(sys.argv)
