import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import PanelMenuIcon from './icon.js';
import PopupMenuProfile from '../popupMenu/profile.js';
import PreviewMenuItem from '../popupMenu/preview_menu_item.js';
import ControlsMenuItem from '../popupMenu/controls_menu_item.js';
import OpenCurrentMenuItem from '../popupMenu/open_current_menu_item.js';
/**
 * Main indicator button for DeskChanger
 *
 * This is an interface between the extension/settings/service and the menu it
 * contains. All internal item properties that need to be bound to external
 * objects are exposed and bound here so the extension itself can manage the
 * binding between the necessary object. This also provides signals that expose
 * internal control signals for binding back to the service as well as a signal
 * to let the extension open the prefs with the built-in method.
 */
export default class PanelMenuButton extends PanelMenu.Button {
    static {
        GObject.registerClass(
            {
                GTypeName: 'DeskChangerUiPanelMenuButton',
                Properties: {
                    'icon-preview-enabled': GObject.param_spec_boolean(
                        'icon-preview-enabled',
                        'Icon preview enabled',
                        'If enabled the icon will turn into a preview of the next wallpaper',
                        false,
                        GObject.ParamFlags.READWRITE
                    ),
                    notifications: GObject.param_spec_boolean(
                        'notifications',
                        'Notifications',
                        'If enabled the extension will show notifications',
                        true,
                        GObject.ParamFlags.READWRITE
                    ),
                    preview: GObject.param_spec_string(
                        'preview',
                        'Preview',
                        'Preview URI to be passed into the menu',
                        null,
                        GObject.ParamFlags.READWRITE
                    ),
                    profile: GObject.param_spec_string(
                        'profile',
                        'Profile',
                        'Current profile selected',
                        null,
                        GObject.ParamFlags.READWRITE
                    ),
                    profiles: GObject.param_spec_variant(
                        'profiles',
                        'Profiles',
                        'Profiles object to provide into the menu',
                        new GLib.VariantType('a{sa(sb)}'),
                        null,
                        GObject.ParamFlags.READWRITE
                    ),
                    random: GObject.param_spec_boolean(
                        'random',
                        'Random',
                        'Random flag for the controls in the menu',
                        true,
                        GObject.ParamFlags.READWRITE
                    ),
                    'remember-profile-state': GObject.param_spec_boolean(
                        'remember-profile-state',
                        'Remember Profile State',
                        'Remember the wallpaper queue on unload',
                        false,
                        GObject.ParamFlags.READWRITE
                    ),
                    'service-running': GObject.param_spec_boolean(
                        'service-running',
                        'Service Running',
                        'Check if the service is running',
                        false,
                        GObject.ParamFlags.READWRITE
                    ),
                },
                Signals: {
                    'next-clicked': [],
                    'open-prefs': [],
                    'previous-clicked': [],
                },
            },
            this
        );
    }
    #bindings;
    #controls_menu_item;
    #icon;
    #icon_preview_enabled;
    #icon_preview_switch_menu_item;
    #next_clicked_id;
    #notifications;
    #notifications_switch_menu_item;
    #open_current_menu_item;
    #preferences_activate_id;
    #preferences_menu_item;
    #preview;
    #preview_menu_item;
    #previous_clicked_id;
    #profile;
    #profile_activate_id;
    #profile_menu_item;
    #profiles;
    #random;
    #remember_profile_state;
    #remember_profile_state_switch_menu_item;
    #service_running;
    #settings_submenu;

    get icon_preview_enabled() {
        return this.#icon_preview_enabled;
    }

    get notifications() {
        return this.#notifications;
    }

    get preview() {
        return this.#preview || null;
    }

    get profile() {
        return this.#profile || null;
    }

    get profiles() {
        return this.#profiles || null;
    }

    get random() {
        return this.#random;
    }

    get remember_profile_state() {
        return this.#remember_profile_state;
    }

    get service_running() {
        return this.#service_running;
    }

    set icon_preview_enabled(value) {
        this.#icon_preview_enabled = value;
        this.notify('icon-preview-enabled');
    }

    set notifications(value) {
        this.#notifications = value;
        this.notify('notifications');
    }

    set preview(value) {
        this.#preview = value || undefined;
        this.notify('preview');
    }

    set profile(value) {
        this.#profile = value || undefined;
        this.notify('profile');
    }

    set profiles(value) {
        this.#profiles = value || undefined;
        this.notify('profiles');
    }

    set random(value) {
        this.#random = value;
        this.notify('random');
    }

    set remember_profile_state(value) {
        this.#remember_profile_state = value;
        this.notify('remember-profile-state');
    }

    set service_running(value) {
        this.#service_running = value;
        this.notify('service-running');
    }

    constructor(uuid) {
        super(0.0, uuid);
        this.#bindings = [];
        this.#icon_preview_enabled = false;
        this.#notifications = true;
        this.#random = true;
        this.#remember_profile_state = false;
        this.#service_running = false;
        // first set up the icon
        this.#icon = new PanelMenuIcon();
        this.#bindings.push(
            this.bind_property(
                'icon-preview-enabled',
                this.#icon,
                'preview-enabled',
                GObject.BindingFlags.SYNC_CREATE
            )
        );
        this.#bindings.push(
            this.bind_property(
                'preview',
                this.#icon,
                'preview',
                GObject.BindingFlags.SYNC_CREATE
            )
        );
        this.add_child(this.#icon);
        // now add the menu items, profile first
        this.#profile_menu_item = new PopupMenuProfile();
        this.#bindings.push(
            this.bind_property(
                'profile',
                this.#profile_menu_item,
                'profile',
                GObject.BindingFlags.SYNC_CREATE
            )
        );
        this.#bindings.push(
            this.bind_property(
                'profiles',
                this.#profile_menu_item,
                'profiles',
                GObject.BindingFlags.SYNC_CREATE
            )
        );
        this.#profile_activate_id = this.#profile_menu_item.connect(
            'profile-activate',
            (_element, menu_item) => {
                this.profile = menu_item.label.get_text();
            }
        );
        this.menu.addMenuItem(this.#profile_menu_item);
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        // this section is for controls
        this.#preview_menu_item = new PreviewMenuItem();
        this.#bindings.push(
            this.bind_property(
                'preview',
                this.#preview_menu_item,
                'preview',
                GObject.BindingFlags.SYNC_CREATE
            )
        );
        this.menu.addMenuItem(this.#preview_menu_item);
        this.#open_current_menu_item = new OpenCurrentMenuItem();
        this.menu.addMenuItem(this.#open_current_menu_item);
        this.#controls_menu_item = new ControlsMenuItem();
        this.#bindings.push(
            this.bind_property(
                'random',
                this.#controls_menu_item,
                'random',
                GObject.BindingFlags.SYNC_CREATE |
                    GObject.BindingFlags.BIDIRECTIONAL
            )
        );
        this.#bindings.push(
            this.bind_property(
                'service-running',
                this.#controls_menu_item,
                'service-running',
                GObject.BindingFlags.SYNC_CREATE |
                    GObject.BindingFlags.BIDIRECTIONAL
            )
        );
        this.#next_clicked_id = this.#controls_menu_item.connect(
            'next-clicked',
            () => this.emit('next-clicked')
        );
        this.#previous_clicked_id = this.#controls_menu_item.connect(
            'previous-clicked',
            () => this.emit('previous-clicked')
        );
        this.menu.addMenuItem(this.#controls_menu_item);
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        // quick settings submenu
        this.#settings_submenu = new PopupMenu.PopupSubMenuMenuItem(
            _('Quick Settings')
        );
        this.menu.addMenuItem(this.#settings_submenu);
        this.#icon_preview_switch_menu_item = new PopupMenu.PopupSwitchMenuItem(
            _('Icon as Preview'),
            this.icon_preview_enabled
        );
        this.#bindings.push(
            this.bind_property(
                'icon-preview-enabled',
                this.#icon_preview_switch_menu_item,
                'state',
                GObject.BindingFlags.SYNC_CREATE |
                    GObject.BindingFlags.BIDIRECTIONAL
            )
        );
        this.#settings_submenu.menu.addMenuItem(
            this.#icon_preview_switch_menu_item
        );
        this.#remember_profile_state_switch_menu_item =
            new PopupMenu.PopupSwitchMenuItem(
                _('Remember Profile State'),
                this.remember_profile_state
            );
        this.#bindings.push(
            this.bind_property(
                'remember-profile-state',
                this.#remember_profile_state_switch_menu_item,
                'state',
                GObject.BindingFlags.SYNC_CREATE |
                    GObject.BindingFlags.BIDIRECTIONAL
            )
        );
        this.#settings_submenu.menu.addMenuItem(
            this.#remember_profile_state_switch_menu_item
        );
        this.#notifications_switch_menu_item =
            new PopupMenu.PopupSwitchMenuItem(
                _('Notifications'),
                this.notifications
            );
        this.#bindings.push(
            this.bind_property(
                'notifications',
                this.#notifications_switch_menu_item,
                'state',
                GObject.BindingFlags.SYNC_CREATE |
                    GObject.BindingFlags.BIDIRECTIONAL
            )
        );
        this.#settings_submenu.menu.addMenuItem(
            this.#notifications_switch_menu_item
        );
        // preferences
        this.#preferences_menu_item = new PopupMenu.PopupMenuItem(
            _('Preferences')
        );
        this.#preferences_activate_id = this.#preferences_menu_item.connect(
            'activate',
            () => this.emit('open-prefs')
        );
        this.menu.addMenuItem(this.#preferences_menu_item);
        // fin.
    }

    destroy() {
        for (const binding of this.#bindings) {
            binding.unbind();
        }
        this.#bindings = [];
        if (this.#profile_activate_id) {
            this.#profile_menu_item.disconnect(this.#profile_activate_id);
            this.#profile_activate_id = undefined;
        }
        if (this.#preferences_activate_id) {
            this.#preferences_menu_item.disconnect(
                this.#preferences_activate_id
            );
            this.#preferences_activate_id = undefined;
        }
        if (this.#next_clicked_id) {
            this.#controls_menu_item.disconnect(this.#next_clicked_id);
            this.#next_clicked_id = undefined;
        }
        if (this.#previous_clicked_id) {
            this.#controls_menu_item.disconnect(this.#previous_clicked_id);
            this.#previous_clicked_id = undefined;
        }
        this.#icon_preview_switch_menu_item?.destroy();
        this.#icon_preview_switch_menu_item = undefined;
        this.#notifications_switch_menu_item?.destroy();
        this.#notifications_switch_menu_item = undefined;
        this.#open_current_menu_item?.destroy();
        this.#open_current_menu_item = undefined;
        this.#preferences_menu_item?.destroy();
        this.#preferences_menu_item = undefined;
        this.#controls_menu_item?.destroy();
        this.#controls_menu_item = undefined;
        this.#preview_menu_item?.destroy();
        this.#preview_menu_item = undefined;
        this.#profile_menu_item?.destroy();
        this.#profile_menu_item = undefined;
        this.#settings_submenu?.menu.removeAll();
        this.#settings_submenu?.destroy();
        this.#settings_submenu = undefined;
        this.menu.removeAll();
        this.#icon?.destroy();
        this.#icon = undefined;
        super.destroy();
    }
}
