import Cogl from "gi://Cogl";
import Clutter from "gi://Clutter";
import GLib from "gi://GLib";
import Pango from "gi://Pango";
import St from "gi://St";

import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";
import * as Main from "resource:///org/gnome/shell/ui/main.js";

import { DATE_TIME_FORMATS, UPDATE_INTERVAL_MS, EXTENSION_UUID } from './constants.js';

/**
 * Log an error message
 */
function logError(message) {
    console.error(`[${EXTENSION_UUID}] ERROR: ${message}`);
}

/**
 * Log a debug message
 */
function logDebug(message) {
    console.log(`[${EXTENSION_UUID}] DEBUG: ${message}`);
}

export default class DateTimeFormatExtension extends Extension {
    /**
     * Update the date/time label with the current time
     * @returns {boolean} True to continue the timeout
     */
    _updateDateTime() {
        if (!this._dateTimeLabel) {
            return false;
        }

        const formatString = this._getFormatString(this._format);
        const formattedText = GLib.DateTime.new_now_local().format(
            this._format === 'other' ? this._userFormatString : formatString
        );
        this._dateTimeLabel.clutter_text.set_markup(formattedText);

        return true;
    }

    /**
     * Get the format string for a given format key
     * @param {string} formatKey - The format key to look up
     * @returns {string} The format string, or default if key is invalid
     */
    _getFormatString(formatKey) {
        return DATE_TIME_FORMATS[formatKey] || DATE_TIME_FORMATS['datetime'];
    }

    /**
     * Update the label width
     */
    _updateDatetimeLabelWidth() {
        const dateTimeTextLenght = this._dateTimeLabel.clutter_text.get_text().length;
        if (dateTimeTextLenght > 30) {
            const dateTimeLabelWidth = this._dateTimeLabel.clutter_text.get_width() - 5;
            this._dateTimeLabel.clutter_text.set_width(dateTimeLabelWidth);
        } else {
            this._dateTimeLabel.clutter_text.set_width(-1);
        }
    }

    /**
     * Enable the extension
     */
    enable() {
        // Check if panel and dateMenu exist
        if (!Main.panel?.statusArea?.dateMenu?._clockDisplay) {
            logError('Clock display not found. Panel structure may have changed.');
            return;
        }

        this._systemClockLabel = Main.panel.statusArea.dateMenu._clockDisplay;

        // Create the custom date/time label
        this._dateTimeLabel = new St.Label({ style_class: "clock" });
        this._dateTimeLabel.clutter_text.y_align = Clutter.ActorAlign.CENTER;
        this._dateTimeLabel.clutter_text.ellipsize = Pango.EllipsizeMode.END;
        this._dateTimeLabel.clutter_text.use_markup = true;

        // Initial update
        this._updateDateTime();
        this._updateDatetimeLabelWidth();

        this._settings = this.getSettings();
        this._format = this._settings.get_string("format");
        this._userFormatString = this._settings.get_value('user-format').deepUnpack().userFormat;

        // Hide system clock and insert our label
        this._systemClockLabel.hide();
        const parent = this._systemClockLabel.get_parent();
        if (parent) {
            parent.insert_child_below(this._dateTimeLabel, this._systemClockLabel);
        } else {
            logError('System clock label has no parent');
            this.disable();

            return;
        }

        // Set up periodic updates
        this._timeoutID = GLib.timeout_add(
            GLib.PRIORITY_DEFAULT,
            UPDATE_INTERVAL_MS,
            this._updateDateTime.bind(this),
        );

        // Listen for format changes
        this._formatChangedId = this._settings.connect('changed::format', (settings, key) => {
            this._format = settings.get_string(key);

            this._updateDateTime();
        });

        // Listen for custom format changes
        this._userFormatChangedId = this._settings.connect('changed::user-format', (settings, key) => {
            this._userFormatString = this._settings.get_value('user-format').deepUnpack().userFormat;

            this._updateDateTime();
            this._updateDatetimeLabelWidth();
        });

        logDebug('Extension enabled successfully');
    }

    /**
     * Disable the extension and clean up resources
     */
    disable() {
        // Remove timeout
        if (this._timeoutID) {
            GLib.Source.remove(this._timeoutID);
            this._timeoutID = null;
        }

        // Disconnect settings signal datetime format
        if (this._formatChangedId) {
            this._settings.disconnect(this._formatChangedId);
            this._formatChangedId = null;
        }

        // Disconnect settings signal user datetime format
        if (this._userFormatChangedId) {
            this._settings.disconnect(this._userFormatChangedId);
            this._userFormatChangedId = null;
        }

        // Clear settings reference
        this._settings = null;

        // Clean up label
        if (this._systemClockLabel) {
            if (this._dateTimeLabel?.get_parent()) {
                this._systemClockLabel.get_parent().remove_child(this._dateTimeLabel);
                this._dateTimeLabel.destroy();
                this._dateTimeLabel = null;
            }

            // Show system clock again
            this._systemClockLabel.show();
        }

        logDebug('Extension disabled successfully');
    }
}
