import St from 'gi://St';
import Clutter from 'gi://Clutter';
import GObject from 'gi://GObject';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import { RingProgress } from './ring.js';
import { ICONS } from './icons.js';
export const VitalItem = GObject.registerClass(class VitalItem extends St.BoxLayout {
    _type;
    _settings;
    _ringProgress = null;
    _icon = null;
    _label = null;
    _handlerIds = [];
    _currentValue = 0;
    _isDestroyed = false;
    constructor(type, settings) {
        super({
            style_class: 'vital-item',
            vertical: true,
            x_align: Clutter.ActorAlign.CENTER,
        });
        this._type = type;
        this._settings = settings;
        this._buildUI();
        this._connectSettings();
        this._updateStyle();
    }
    _getGIcon(color) {
        const svgFunc = ICONS[this._type];
        const svgString = svgFunc ? svgFunc(color) : '';
        // Convert string to Uint8Array before creating GLib.Bytes
        const encoder = new TextEncoder();
        const data = encoder.encode(svgString);
        const bytes = new GLib.Bytes(data);
        return Gio.BytesIcon.new(bytes);
    }
    _buildUI() {
        if (this._isDestroyed)
            return;
        this.vertical = this._settings.get_string('vital-orientation') === 'vertical';
        const container = new St.Widget({
            layout_manager: new Clutter.BinLayout(),
        });
        this._ringProgress = new RingProgress(this._type, this._settings);
        container.add_child(this._ringProgress);
        if (this._settings.get_boolean('show-icons')) {
            this._icon = this._createIcon();
            container.add_child(this._icon);
        }
        this.add_child(container);
        if (this._settings.get_boolean('show-labels')) {
            this._label = new St.Label({
                text: '0%',
                style_class: 'vital-label',
                x_align: Clutter.ActorAlign.CENTER,
            });
            this.add_child(this._label);
        }
    }
    _createIcon() {
        const diameter = this._settings.get_int('ring-diameter');
        const iconColor = this._settings.get_string('icon-color');
        return new St.Icon({
            gicon: this._getGIcon(iconColor),
            style_class: 'vital-icon',
            icon_size: Math.round(diameter * 0.5),
            style: 'stroke-width: 1px;',
            x_align: Clutter.ActorAlign.CENTER,
            y_align: Clutter.ActorAlign.CENTER,
            x_expand: true, y_expand: true,
        });
    }
    _connectSettings() {
        if (this._isDestroyed)
            return;
        const keys = [
            'show-icons', 'show-labels', 'ring-diameter', 'vital-orientation'
        ];
        keys.forEach(key => {
            this._handlerIds.push(this._settings.connect(`changed::${key}`, () => {
                if (!this._isDestroyed) {
                    this._rebuildUI();
                }
            }));
        });
        const styleKeys = [`${this._type}-color`, 'icon-color', 'inactive-ring-color', 'label-font-size'];
        styleKeys.forEach(key => {
            this._handlerIds.push(this._settings.connect(`changed::${key}`, () => {
                if (!this._isDestroyed) {
                    this._updateStyle();
                }
            }));
        });
    }
    _updateStyle() {
        // Check for existence AND parentage AND destruction state
        if (this._isDestroyed || !this._ringProgress || !this.get_parent())
            return;
        const vitalColor = this._settings.get_string(`${this._type}-color`);
        const iconColor = this._settings.get_string('icon-color');
        if (this._icon) {
            this._icon.gicon = this._getGIcon(iconColor);
        }
        if (this._label) {
            this._label.set_style(`color: ${iconColor}; font-size: ${this._settings.get_int('label-font-size')}px;`);
        }
    }
    _rebuildUI() {
        if (this._isDestroyed || !this.get_parent())
            return;
        if (this._ringProgress) {
            this._ringProgress.destroy();
        }
        this.destroy_all_children();
        // Explicitly nullify references to destroyed children
        this._label = null;
        this._icon = null;
        this._ringProgress = null;
        this._buildUI();
        // Only update if we successfully rebuilt
        if (this._ringProgress) {
            this.update(this._currentValue);
        }
    }
    update(value) {
        if (this._isDestroyed)
            return;
        if (!this._ringProgress || !this.get_parent())
            return;
        try {
            this._currentValue = Math.min(100, Math.max(0, value));
            this._ringProgress.setValue(this._currentValue);
            if (this._label) {
                this._label.set_text(`${Math.round(this._currentValue)}%`);
            }
        }
        catch (e) {
            if (!this._isDestroyed) {
                console.debug(`[VitalsWidget] Update suppressed: ${e}`);
            }
        }
    }
    destroy() {
        this._isDestroyed = true;
        this._handlerIds.forEach(id => this._settings.disconnect(id));
        this._handlerIds = [];
        if (this._ringProgress) {
            this._ringProgress.destroy();
            this._ringProgress = null;
        }
        this._label = null;
        this._icon = null;
        super.destroy();
    }
});
