import GLib from 'gi://GLib';
import Gio from 'gi://Gio';
export class GPUSensor {
    _gpuType = 'none';
    _sysfsPath = '';
    _errorCount = 0;
    _maxErrors = 5;
    _disabled = false;
    constructor() {
        this._detectGPU();
    }
    _detectGPU() {
        const nvidiaSmiPath = GLib.find_program_in_path('nvidia-smi');
        if (nvidiaSmiPath) {
            this._gpuType = 'nvidia';
            return;
        }
        for (let i = 0; i < 10; i++) {
            const path = `/sys/class/drm/card${i}/device/gpu_busy_percent`;
            if (Gio.File.new_for_path(path).query_exists(null)) {
                this._gpuType = 'amd_sysfs';
                this._sysfsPath = path;
                return;
            }
        }
        if (GLib.find_program_in_path('radeontop')) {
            this._gpuType = 'amd_radeontop';
            return;
        }
        this._gpuType = 'none';
    }
    async getValue() {
        if (this._disabled || this._gpuType === 'none')
            return 0;
        switch (this._gpuType) {
            case 'nvidia': return await this._getNvidiaUsage();
            case 'amd_sysfs': return await this._getAmdSysfsUsage();
            case 'amd_radeontop': return await this._getAmdRadeontopUsage();
            default: return 0;
        }
    }
    async _getNvidiaUsage() {
        try {
            const proc = Gio.Subprocess.new(['nvidia-smi', '--query-gpu=utilization.gpu', '--format=csv,noheader,nounits'], Gio.SubprocessFlags.STDOUT_PIPE | Gio.SubprocessFlags.STDERR_PIPE);
            const [stdout] = await proc.communicate_utf8_async(null, null);
            const usage = parseInt(stdout?.trim() || '0');
            this._errorCount = 0;
            return isNaN(usage) ? 0 : Math.max(0, Math.min(100, usage));
        }
        catch (e) {
            this._handleError(e);
            return 0;
        }
    }
    async _getAmdSysfsUsage() {
        try {
            const file = Gio.File.new_for_path(this._sysfsPath);
            const [contents] = await file.load_contents_async(null);
            if (!contents)
                return 0;
            const data = new TextDecoder().decode(contents);
            const usage = parseInt(data.trim());
            this._errorCount = 0;
            return isNaN(usage) ? 0 : Math.max(0, Math.min(100, usage));
        }
        catch (e) {
            this._handleError(e);
            return 0;
        }
    }
    async _getAmdRadeontopUsage() {
        try {
            const proc = Gio.Subprocess.new(['radeontop', '-d', '-', '-l', '1'], Gio.SubprocessFlags.STDOUT_PIPE);
            const [stdout] = await proc.communicate_utf8_async(null, null);
            const match = stdout?.match(/gpu\s+(\d+(?:\.\d+)?)%/);
            return match ? Math.round(parseFloat(match[1])) : 0;
        }
        catch (e) {
            return 0;
        }
    }
    _handleError(e) {
        this._errorCount++;
        if (this._errorCount >= this._maxErrors)
            this._disabled = true;
    }
}
